/**
 * \file InsetMathBoldSymbol.cpp
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Andr Pnitz
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "InsetMathBoldSymbol.h"
#include "MathStream.h"
#include "MathData.h"
#include "LaTeXFeatures.h"
#include "support/std_ostream.h"


namespace lyx {

using std::auto_ptr;


InsetMathBoldSymbol::InsetMathBoldSymbol(Kind kind)
	: InsetMathNest(1), kind_(kind)
{}


auto_ptr<Inset> InsetMathBoldSymbol::doClone() const
{
	return auto_ptr<Inset>(new InsetMathBoldSymbol(*this));
}


bool InsetMathBoldSymbol::metrics(MetricsInfo & mi, Dimension & dim) const
{
	//FontSetChanger dummy(mi.base, "mathbf");
	cell(0).metrics(mi, dim);
	metricsMarkers(dim);
	++dim.wid;  // for 'double stroke'
	if (dim_ == dim)
		return false;
	dim_ = dim;
	return true;
}


void InsetMathBoldSymbol::draw(PainterInfo & pi, int x, int y) const
{
	//FontSetChanger dummy(pi.base, "mathbf");
	cell(0).draw(pi, x + 1, y);
	cell(0).draw(pi, x + 2, y);
	drawMarkers(pi, x, y);
}


void InsetMathBoldSymbol::metricsT(TextMetricsInfo const & mi, Dimension & /*dim*/) const
{
	cell(0).metricsT(mi, dim_);
}


void InsetMathBoldSymbol::drawT(TextPainter & pain, int x, int y) const
{
	cell(0).drawT(pain, x, y);
}


void InsetMathBoldSymbol::validate(LaTeXFeatures & features) const
{
	InsetMathNest::validate(features);
	if (kind_ == AMS_BOLD)
		features.require("amsbsy");
	else
		features.require("bm");
}


void InsetMathBoldSymbol::write(WriteStream & os) const
{
	switch (kind_) {
	case AMS_BOLD:
		os << "\\boldsymbol{" << cell(0) << "}";
		break;
	case BM_BOLD:
		os << "\\bm{" << cell(0) << "}";
		break;
	case BM_HEAVY:
		os << "\\hm{" << cell(0) << "}";
		break;
	}
}


void InsetMathBoldSymbol::infoize(odocstream & os) const
{
	switch (kind_) {
	case AMS_BOLD:
		os << "Boldsymbol ";
		break;
	case BM_BOLD:
		os << "Boldsymbol (bm)";
		break;
	case BM_HEAVY:
		os << "Heavysymbol (bm)";
		break;
	}
}


} // namespace lyx
