package ntlmssp

import (
	"errors"
	"io"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestNegotiateMessageUnmarshal(t *testing.T) {
	tables := []struct {
		got  []byte
		want negotiateMessage
		err  error
	}{
		{
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0xb2, 0x02, 0xe2,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x06, 0x01, 0xb1, 0x1d, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x43, 0x4f,
				0x4d, 0x50, 0x55, 0x54, 0x45, 0x52,
			},
			negotiateMessage{
				negotiateMessageFields: negotiateMessageFields{
					messageHeader:  newMessageHeader(ntLmNegotiate),
					NegotiateFlags: 0xe202b232,
					DomainNameFields: payload{
						Len:    6,
						MaxLen: 6,
						Offset: 40,
					},
					WorkstationFields: payload{
						Len:    8,
						MaxLen: 8,
						Offset: 46,
					},
				},
				DomainName:  "Domain",
				Workstation: "COMPUTER",
				Version: &Version{
					ProductMajorVersion: 6,
					ProductMinorVersion: 1,
					ProductBuild:        7601,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			nil,
		},
		{
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0xb2, 0x02, 0xe0,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x43, 0x4f,
				0x4d, 0x50, 0x55, 0x54, 0x45, 0x52,
			},
			negotiateMessage{
				negotiateMessageFields: negotiateMessageFields{
					messageHeader:  newMessageHeader(ntLmNegotiate),
					NegotiateFlags: ntlmsspNegotiateVersion.Unset(0xe202b232),
					DomainNameFields: payload{
						Len:    6,
						MaxLen: 6,
						Offset: 40,
					},
					WorkstationFields: payload{
						Len:    8,
						MaxLen: 8,
						Offset: 46,
					},
				},
				DomainName:  "Domain",
				Workstation: "COMPUTER",
				Version:     nil,
			},
			nil,
		},
		{
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0x82, 0x02, 0xe2,
				0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x06, 0x01, 0xb1, 0x1d, 0x00, 0x00, 0x00, 0x0f,
			},
			negotiateMessage{
				negotiateMessageFields: negotiateMessageFields{
					messageHeader:  newMessageHeader(ntLmNegotiate),
					NegotiateFlags: 0xe2028232,
					DomainNameFields: payload{
						Len:    0,
						MaxLen: 0,
						Offset: 40,
					},
					WorkstationFields: payload{
						Len:    0,
						MaxLen: 0,
						Offset: 40,
					},
				},
				DomainName:  "",
				Workstation: "",
				Version: &Version{
					ProductMajorVersion: 6,
					ProductMinorVersion: 1,
					ProductBuild:        7601,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			nil,
		},
		{
			// 31 bytes to force a short negotiateMessageFields read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0x82, 0x02, 0xe2,
				0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00,
			},
			negotiateMessage{},
			io.ErrUnexpectedEOF,
		},
		{
			// 9th byte changed to force invalid negotiate message
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x02, 0x00, 0x00, 0x00, 0x32, 0xb2, 0x02, 0xe2,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x06, 0x01, 0xb1, 0x1d, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x43, 0x4f,
				0x4d, 0x50, 0x55, 0x54, 0x45, 0x52,
			},
			negotiateMessage{},
			errInvalidNegotiateMessage,
		},
		{
			// 45 bytes to force a short DomainName read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0xb2, 0x02, 0xe0,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x44, 0x6f, 0x6d, 0x61, 0x69,
			},
			negotiateMessage{},
			errors.New("expected 6 bytes, only read 5"),
		},
		{
			// 54 bytes to force a short Workstation read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0xb2, 0x02, 0xe0,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x43, 0x4f,
				0x4d, 0x50, 0x55, 0x54, 0x45,
			},
			negotiateMessage{},
			errors.New("expected 8 bytes, only read 7"),
		},
		{
			// 39 bytes to force a short Version read
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0x82, 0x02, 0xe2,
				0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x06, 0x01, 0xb1, 0x1d, 0x00, 0x00, 0x00,
			},
			negotiateMessage{},
			io.ErrUnexpectedEOF,
		},
	}

	for _, table := range tables {
		m := negotiateMessage{}
		err := m.Unmarshal(table.got)
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, m)
		}
	}
}

func TestNegotiateMessageMarshal(t *testing.T) {
	tables := []struct {
		got  negotiateMessage
		want []byte
		err  error
	}{
		{
			negotiateMessage{
				negotiateMessageFields: negotiateMessageFields{
					messageHeader:  newMessageHeader(ntLmNegotiate),
					NegotiateFlags: 0xe202b232,
				},
				DomainName:  "Domain",
				Workstation: "COMPUTER",
				Version: &Version{
					ProductMajorVersion: 6,
					ProductMinorVersion: 1,
					ProductBuild:        7601,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0xb2, 0x02, 0xe2,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x06, 0x01, 0xb1, 0x1d, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x43, 0x4f,
				0x4d, 0x50, 0x55, 0x54, 0x45, 0x52,
			},
			nil,
		},
		{
			negotiateMessage{
				negotiateMessageFields: negotiateMessageFields{
					messageHeader:  newMessageHeader(ntLmNegotiate),
					NegotiateFlags: ntlmsspNegotiateVersion.Unset(0xe202b232),
				},
				DomainName:  "Domain",
				Workstation: "COMPUTER",
				Version:     nil,
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x01, 0x00, 0x00, 0x00, 0x32, 0xb2, 0x02, 0xe0,
				0x06, 0x00, 0x06, 0x00, 0x28, 0x00, 0x00, 0x00,
				0x08, 0x00, 0x08, 0x00, 0x2e, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x44, 0x6f, 0x6d, 0x61, 0x69, 0x6e, 0x43, 0x4f,
				0x4d, 0x50, 0x55, 0x54, 0x45, 0x52,
			},
			nil,
		},
	}

	for _, table := range tables {
		b, err := table.got.Marshal()
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, b)
		}
	}
}
