// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief e+e- -> pi+pi- and 3pi0 or 2pi0eta
  class BABAR_2018_I1700745 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BABAR_2018_I1700745);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {


      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");

      for (size_t ix=0; ix<5; ++ix) {
        book(_sigma[ix], 1+ix, 1, 1);
        for (const string& en : _sigma[0].binning().edges<0>()) {
          const double eval = std::stod(en)*GeV;
          if (isCompatibleWithSqrtS(eval)) {
            _sqs = en;
            break;
          }
        }
      }
      if (_sqs.empty() && isCompatibleWithSqrtS(1.075)) {
        _sqs = "1.075";
      }
      else raiseBeamErrorIf(_sqs.empty());
    }

    void findChildren(const Particle& p, map<long,int> & nRes, int& ncount) const {
      for (const Particle &child : p.children()) {
        if (child.children().empty()) {
          --nRes[child.pid()];
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      if (ntotal==5 && nCount[211]==1 && nCount[-211]==1 && nCount[111]==3) {
	      _sigma[0]->fill(_sqs);
      }
      const FinalState& ufs = apply<FinalState>(event, "UFS");
      for (const Particle& p : ufs.particles()) {
        if (p.children().empty()) continue;
        if (p.pid()==221 || p.pid()==223 ) {	// find eta/omegas
          map<long,int> nRes = nCount;
          int ncount = ntotal;
          findChildren(p,nRes,ncount);
          if (p.pid()==221) { // eta
            if (ncount==2) { // 2 pi eta
              bool matched = true;
              for (const auto& val : nRes) {
                if (abs(val.first)==211) {
                  if (val.second !=1) {
                    matched = false;
                    break;
                  }
                }
                else if (val.second!=0) {
                  matched = false;
                  break;
                }
              }
              if (matched)  _sigma[1]->fill(_sqs);
            }
            else if (ncount==4) { // 4 pi eta
              bool matched = true;
              for (const auto& val : nRes) {
                if (abs(val.first)==211) {
                  if (val.second !=1) {
                    matched = false;
                    break;
                  }
                }
                else if (abs(val.first)==111) {
                  if (val.second !=2) {
                    matched = false;
                    break;
                  }
                }
                else if (val.second!=0) {
                  matched = false;
                  break;
                }
              }
              if (matched) _sigma[3]->fill(_sqs);
            }
            for (const Particle& p2 : ufs.particles()) { // pi0 omega eta
              if (p2.pid()!=223) continue;
              map<long,int> nResB = nRes;
              int ncountB = ncount;
              findChildren(p2,nResB,ncountB);
              if (ncountB!=1) continue;
              bool matched = true;
              for (const auto& val : nResB) {
                if (abs(val.first)==111) {
                  if (val.second !=1) {
                    matched = false;
                    break;
                  }
                }
                else if (val.second!=0) {
                  matched = false;
                  break;
                }
              }
              if (matched) {
                _sigma[4]->fill(_sqs);
                break;
              }
            }
          }
          else {
            if (ncount!=2) continue;
            bool matched = true;
            for (const auto& val : nRes) {
              if (abs(val.first)==111) {
                if (val.second !=2) {
                  matched = false;
                  break;
                }
              }
              else if (val.second!=0) {
                matched = false;
                break;
              }
            }
            if (matched)  _sigma[2]->fill(_sqs);
          }
        }
      }
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigma, crossSection()/ sumOfWeights() /nanobarn);
    }
    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _sigma[5];
    string _sqs = "";
    /// @}

  };


  RIVET_DECLARE_PLUGIN(BABAR_2018_I1700745);


}
