// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief pi, K and proton spectra at 14,22 and 34 GeV
  class TASSO_1983_I181470 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1983_I181470);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(FinalState(), "FS");

      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        size_t ih1 = 0, ih2 = 0;
        if (en == "14"s) { ih1 = 19; ih2 = 22; }
        if (en == "22"s) { ih1 = 25; ih2 = 10; }
        if (en == "34"s) { ih1 = 13; ih2 = 16; }
        book(_h[en+"p_pi"], ih1,   1, 1);
        book(_h[en+"p_K"],  ih1+2, 1, 1);
        book(_h[en+"p_p"],  ih2+1, 1, 1);
        book(_h[en+"x_pi"], ih1+1, 1, 1);
        book(_h[en+"x_K"],  ih2,   1, 1);
        book(_h[en+"x_p"],  ih2+2, 1, 1);
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const FinalState& fs = apply<FinalState>(event, "FS");
      const size_t numParticles = fs.particles().size();

      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if (numParticles < 2)  vetoEvent;

      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      MSG_DEBUG("Avg beam momentum = " << meanBeamMom);

      for (const Particle& p : fs.particles()) {
        double xE = p.E()/meanBeamMom;
        if (p.abspid()==211) {
          _h[_sqs+"p_pi"]->fill(p.p3().mod());
          _h[_sqs+"x_pi"]->fill(xE          );
        }
        else if (p.abspid()==321) {
          _h[_sqs+"p_K"]->fill(p.p3().mod());
          _h[_sqs+"x_K"]->fill(xE          );
        }
        else if(p.abspid()==2212) {
          _h[_sqs+"p_p"]->fill(p.p3().mod());
          _h[_sqs+"x_p"]->fill(xE          );
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {

      for (auto& item : _h) {
        if (item.first.find("p_") != string::npos) {
          scale(item.second, crossSection()/nanobarn/sumOfWeights());
        }
        else {
          const double en = stod(item.first.substr(0,2));
          scale(item.second, sqr(en)/GeV2*crossSection()/microbarn/sumOfWeights());
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1983_I181470);
}
