// Copyright 2020 The Gitea Authors. All rights reserved.
// SPDX-License-Identifier: MIT

package integration

import (
	"net/http"
	"net/url"
	"strings"
	"testing"
	"time"

	auth_model "code.gitea.io/gitea/models/auth"
	git_model "code.gitea.io/gitea/models/git"
	issues_model "code.gitea.io/gitea/models/issues"
	"code.gitea.io/gitea/models/perm"
	repo_model "code.gitea.io/gitea/models/repo"
	"code.gitea.io/gitea/models/unit"
	"code.gitea.io/gitea/models/unittest"
	user_model "code.gitea.io/gitea/models/user"
	pull_service "code.gitea.io/gitea/services/pull"
	repo_service "code.gitea.io/gitea/services/repository"
	files_service "code.gitea.io/gitea/services/repository/files"

	"github.com/stretchr/testify/assert"
)

func TestAPIPullUpdate(t *testing.T) {
	onGiteaRun(t, func(t *testing.T, giteaURL *url.URL) {
		// Create PR to test
		user := unittest.AssertExistsAndLoadBean(t, &user_model.User{ID: 2})
		org26 := unittest.AssertExistsAndLoadBean(t, &user_model.User{ID: 26})
		pr := createOutdatedPR(t, user, org26)

		// Test GetDiverging
		diffCount, err := pull_service.GetDiverging(t.Context(), pr)
		assert.NoError(t, err)
		assert.Equal(t, 1, diffCount.Behind)
		assert.Equal(t, 1, diffCount.Ahead)
		assert.NoError(t, pr.LoadBaseRepo(t.Context()))
		assert.NoError(t, pr.LoadIssue(t.Context()))

		session := loginUser(t, "user2")
		token := getTokenForLoggedInUser(t, session, auth_model.AccessTokenScopeWriteRepository)
		req := NewRequestf(t, "POST", "/api/v1/repos/%s/%s/pulls/%d/update", pr.BaseRepo.OwnerName, pr.BaseRepo.Name, pr.Issue.Index).
			AddTokenAuth(token)
		session.MakeRequest(t, req, http.StatusOK)

		// Test GetDiverging after update
		diffCount, err = pull_service.GetDiverging(t.Context(), pr)
		assert.NoError(t, err)
		assert.Equal(t, 0, diffCount.Behind)
		assert.Equal(t, 2, diffCount.Ahead)
	})
}

func enableRepoAllowUpdateWithRebase(t *testing.T, repoID int64, allow bool) {
	t.Helper()

	repoUnit := unittest.AssertExistsAndLoadBean(t, &repo_model.RepoUnit{RepoID: repoID, Type: unit.TypePullRequests})
	repoUnit.PullRequestsConfig().AllowRebaseUpdate = allow
	assert.NoError(t, repo_model.UpdateRepoUnit(t.Context(), repoUnit))
}

func TestAPIPullUpdateByRebase(t *testing.T) {
	onGiteaRun(t, func(t *testing.T, giteaURL *url.URL) {
		// Create PR to test
		user := unittest.AssertExistsAndLoadBean(t, &user_model.User{ID: 2})
		org26 := unittest.AssertExistsAndLoadBean(t, &user_model.User{ID: 26})
		pr := createOutdatedPR(t, user, org26)

		// Test GetDiverging
		diffCount, err := pull_service.GetDiverging(t.Context(), pr)
		assert.NoError(t, err)
		assert.Equal(t, 1, diffCount.Behind)
		assert.Equal(t, 1, diffCount.Ahead)
		assert.NoError(t, pr.LoadBaseRepo(t.Context()))
		assert.NoError(t, pr.LoadIssue(t.Context()))

		enableRepoAllowUpdateWithRebase(t, pr.BaseRepo.ID, false)

		session := loginUser(t, "user2")
		token := getTokenForLoggedInUser(t, session, auth_model.AccessTokenScopeWriteRepository)
		req := NewRequestf(t, "POST", "/api/v1/repos/%s/%s/pulls/%d/update?style=rebase", pr.BaseRepo.OwnerName, pr.BaseRepo.Name, pr.Issue.Index).
			AddTokenAuth(token)
		session.MakeRequest(t, req, http.StatusForbidden)

		enableRepoAllowUpdateWithRebase(t, pr.BaseRepo.ID, true)
		assert.NoError(t, pr.LoadHeadRepo(t.Context()))

		// use a user which have write access to the pr but not write permission to the head repository to do the rebase
		user40 := unittest.AssertExistsAndLoadBean(t, &user_model.User{ID: 40})
		err = repo_service.AddOrUpdateCollaborator(t.Context(), pr.BaseRepo, user40, perm.AccessModeWrite)
		assert.NoError(t, err)
		token40 := getUserToken(t, "user40", auth_model.AccessTokenScopeWriteRepository)

		req = NewRequestf(t, "POST", "/api/v1/repos/%s/%s/pulls/%d/update?style=rebase", pr.BaseRepo.OwnerName, pr.BaseRepo.Name, pr.Issue.Index).
			AddTokenAuth(token40)
		session.MakeRequest(t, req, http.StatusForbidden)

		err = repo_service.AddOrUpdateCollaborator(t.Context(), pr.HeadRepo, user40, perm.AccessModeWrite)
		assert.NoError(t, err)

		req = NewRequestf(t, "POST", "/api/v1/repos/%s/%s/pulls/%d/update?style=rebase", pr.BaseRepo.OwnerName, pr.BaseRepo.Name, pr.Issue.Index).
			AddTokenAuth(token40)
		session.MakeRequest(t, req, http.StatusOK)

		// Test GetDiverging after update
		diffCount, err = pull_service.GetDiverging(t.Context(), pr)
		assert.NoError(t, err)
		assert.Equal(t, 0, diffCount.Behind)
		assert.Equal(t, 1, diffCount.Ahead)
	})
}

func TestAPIPullUpdateByRebase2(t *testing.T) {
	onGiteaRun(t, func(t *testing.T, giteaURL *url.URL) {
		// Create PR to test
		user := unittest.AssertExistsAndLoadBean(t, &user_model.User{ID: 2})
		org26 := unittest.AssertExistsAndLoadBean(t, &user_model.User{ID: 26})
		pr := createOutdatedPR(t, user, org26)
		assert.NoError(t, pr.LoadBaseRepo(t.Context()))
		assert.NoError(t, pr.LoadIssue(t.Context()))

		enableRepoAllowUpdateWithRebase(t, pr.BaseRepo.ID, false)

		session := loginUser(t, "user2")
		token := getTokenForLoggedInUser(t, session, auth_model.AccessTokenScopeWriteRepository)
		req := NewRequestf(t, "POST", "/api/v1/repos/%s/%s/pulls/%d/update?style=rebase", pr.BaseRepo.OwnerName, pr.BaseRepo.Name, pr.Issue.Index).
			AddTokenAuth(token)
		session.MakeRequest(t, req, http.StatusForbidden)

		enableRepoAllowUpdateWithRebase(t, pr.BaseRepo.ID, true)
		assert.NoError(t, pr.LoadHeadRepo(t.Context()))

		// add a protected branch rule to the head branch to block rebase
		pb := git_model.ProtectedBranch{
			RepoID:       pr.HeadRepo.ID,
			RuleName:     pr.HeadBranch,
			CanPush:      false,
			CanForcePush: false,
		}
		err := git_model.UpdateProtectBranch(t.Context(), pr.HeadRepo, &pb, git_model.WhitelistOptions{})
		assert.NoError(t, err)
		req = NewRequestf(t, "POST", "/api/v1/repos/%s/%s/pulls/%d/update?style=rebase", pr.BaseRepo.OwnerName, pr.BaseRepo.Name, pr.Issue.Index).
			AddTokenAuth(token)
		session.MakeRequest(t, req, http.StatusForbidden)

		// remove the protected branch rule to allow rebase
		err = git_model.DeleteProtectedBranch(t.Context(), pr.HeadRepo, pb.ID)
		assert.NoError(t, err)

		req = NewRequestf(t, "POST", "/api/v1/repos/%s/%s/pulls/%d/update?style=rebase", pr.BaseRepo.OwnerName, pr.BaseRepo.Name, pr.Issue.Index).
			AddTokenAuth(token)
		session.MakeRequest(t, req, http.StatusOK)
	})
}

func createOutdatedPR(t *testing.T, actor, forkOrg *user_model.User) *issues_model.PullRequest {
	baseRepo, err := repo_service.CreateRepository(t.Context(), actor, actor, repo_service.CreateRepoOptions{
		Name:        "repo-pr-update",
		Description: "repo-tmp-pr-update description",
		AutoInit:    true,
		Gitignores:  "C,C++",
		License:     "MIT",
		Readme:      "Default",
		IsPrivate:   false,
	})
	assert.NoError(t, err)
	assert.NotEmpty(t, baseRepo)

	headRepo, err := repo_service.ForkRepository(t.Context(), actor, forkOrg, repo_service.ForkRepoOptions{
		BaseRepo:    baseRepo,
		Name:        "repo-pr-update",
		Description: "desc",
	})
	assert.NoError(t, err)
	assert.NotEmpty(t, headRepo)

	// create a commit on base Repo
	_, err = files_service.ChangeRepoFiles(t.Context(), baseRepo, actor, &files_service.ChangeRepoFilesOptions{
		Files: []*files_service.ChangeRepoFile{
			{
				Operation:     "create",
				TreePath:      "File_A",
				ContentReader: strings.NewReader("File A"),
			},
		},
		Message:   "Add File A",
		OldBranch: "master",
		NewBranch: "master",
		Author: &files_service.IdentityOptions{
			GitUserName:  actor.Name,
			GitUserEmail: actor.Email,
		},
		Committer: &files_service.IdentityOptions{
			GitUserName:  actor.Name,
			GitUserEmail: actor.Email,
		},
		Dates: &files_service.CommitDateOptions{
			Author:    time.Now(),
			Committer: time.Now(),
		},
	})
	assert.NoError(t, err)

	// create a commit on head Repo
	_, err = files_service.ChangeRepoFiles(t.Context(), headRepo, actor, &files_service.ChangeRepoFilesOptions{
		Files: []*files_service.ChangeRepoFile{
			{
				Operation:     "create",
				TreePath:      "File_B",
				ContentReader: strings.NewReader("File B"),
			},
		},
		Message:   "Add File on PR branch",
		OldBranch: "master",
		NewBranch: "newBranch",
		Author: &files_service.IdentityOptions{
			GitUserName:  actor.Name,
			GitUserEmail: actor.Email,
		},
		Committer: &files_service.IdentityOptions{
			GitUserName:  actor.Name,
			GitUserEmail: actor.Email,
		},
		Dates: &files_service.CommitDateOptions{
			Author:    time.Now(),
			Committer: time.Now(),
		},
	})
	assert.NoError(t, err)

	// create Pull
	pullIssue := &issues_model.Issue{
		RepoID:   baseRepo.ID,
		Title:    "Test Pull -to-update-",
		PosterID: actor.ID,
		Poster:   actor,
		IsPull:   true,
	}
	pullRequest := &issues_model.PullRequest{
		HeadRepoID: headRepo.ID,
		BaseRepoID: baseRepo.ID,
		HeadBranch: "newBranch",
		BaseBranch: "master",
		HeadRepo:   headRepo,
		BaseRepo:   baseRepo,
		Type:       issues_model.PullRequestGitea,
	}
	prOpts := &pull_service.NewPullRequestOptions{Repo: baseRepo, Issue: pullIssue, PullRequest: pullRequest}
	err = pull_service.NewPullRequest(t.Context(), prOpts)
	assert.NoError(t, err)

	issue := unittest.AssertExistsAndLoadBean(t, &issues_model.Issue{Title: "Test Pull -to-update-"})
	assert.NoError(t, issue.LoadPullRequest(t.Context()))

	return issue.PullRequest
}
