/* ************************************************************************
* Copyright (C) 2025 Advanced Micro Devices, Inc. All rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*
* ************************************************************************ */
#include "hipsparse.h"

#include <cuda_runtime_api.h>
#include <cusparse_v2.h>
#include <hip/hip_runtime_api.h>

#include "../utility.h"

#if CUDART_VERSION < 11000
hipsparseStatus_t hipsparseXcsrgemmNnz(hipsparseHandle_t         handle,
                                       hipsparseOperation_t      transA,
                                       hipsparseOperation_t      transB,
                                       int                       m,
                                       int                       n,
                                       int                       k,
                                       const hipsparseMatDescr_t descrA,
                                       int                       nnzA,
                                       const int*                csrRowPtrA,
                                       const int*                csrColIndA,
                                       const hipsparseMatDescr_t descrB,
                                       int                       nnzB,
                                       const int*                csrRowPtrB,
                                       const int*                csrColIndB,
                                       const hipsparseMatDescr_t descrC,
                                       int*                      csrRowPtrC,
                                       int*                      nnzTotalDevHostPtr)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseXcsrgemmNnz((cusparseHandle_t)handle,
                            hipsparse::hipOperationToCudaOperation(transA),
                            hipsparse::hipOperationToCudaOperation(transB),
                            m,
                            n,
                            k,
                            (const cusparseMatDescr_t)descrA,
                            nnzA,
                            csrRowPtrA,
                            csrColIndA,
                            (const cusparseMatDescr_t)descrB,
                            nnzB,
                            csrRowPtrB,
                            csrColIndB,
                            (const cusparseMatDescr_t)descrC,
                            csrRowPtrC,
                            nnzTotalDevHostPtr));
}

hipsparseStatus_t hipsparseScsrgemm(hipsparseHandle_t         handle,
                                    hipsparseOperation_t      transA,
                                    hipsparseOperation_t      transB,
                                    int                       m,
                                    int                       n,
                                    int                       k,
                                    const hipsparseMatDescr_t descrA,
                                    int                       nnzA,
                                    const float*              csrValA,
                                    const int*                csrRowPtrA,
                                    const int*                csrColIndA,
                                    const hipsparseMatDescr_t descrB,
                                    int                       nnzB,
                                    const float*              csrValB,
                                    const int*                csrRowPtrB,
                                    const int*                csrColIndB,
                                    const hipsparseMatDescr_t descrC,
                                    float*                    csrValC,
                                    const int*                csrRowPtrC,
                                    int*                      csrColIndC)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseScsrgemm((cusparseHandle_t)handle,
                         hipsparse::hipOperationToCudaOperation(transA),
                         hipsparse::hipOperationToCudaOperation(transB),
                         m,
                         n,
                         k,
                         (const cusparseMatDescr_t)descrA,
                         nnzA,
                         csrValA,
                         csrRowPtrA,
                         csrColIndA,
                         (const cusparseMatDescr_t)descrB,
                         nnzB,
                         csrValB,
                         csrRowPtrB,
                         csrColIndB,
                         (const cusparseMatDescr_t)descrC,
                         csrValC,
                         csrRowPtrC,
                         csrColIndC));
}

hipsparseStatus_t hipsparseDcsrgemm(hipsparseHandle_t         handle,
                                    hipsparseOperation_t      transA,
                                    hipsparseOperation_t      transB,
                                    int                       m,
                                    int                       n,
                                    int                       k,
                                    const hipsparseMatDescr_t descrA,
                                    int                       nnzA,
                                    const double*             csrValA,
                                    const int*                csrRowPtrA,
                                    const int*                csrColIndA,
                                    const hipsparseMatDescr_t descrB,
                                    int                       nnzB,
                                    const double*             csrValB,
                                    const int*                csrRowPtrB,
                                    const int*                csrColIndB,
                                    const hipsparseMatDescr_t descrC,
                                    double*                   csrValC,
                                    const int*                csrRowPtrC,
                                    int*                      csrColIndC)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseDcsrgemm((cusparseHandle_t)handle,
                         hipsparse::hipOperationToCudaOperation(transA),
                         hipsparse::hipOperationToCudaOperation(transB),
                         m,
                         n,
                         k,
                         (const cusparseMatDescr_t)descrA,
                         nnzA,
                         csrValA,
                         csrRowPtrA,
                         csrColIndA,
                         (const cusparseMatDescr_t)descrB,
                         nnzB,
                         csrValB,
                         csrRowPtrB,
                         csrColIndB,
                         (const cusparseMatDescr_t)descrC,
                         csrValC,
                         csrRowPtrC,
                         csrColIndC));
}

hipsparseStatus_t hipsparseCcsrgemm(hipsparseHandle_t         handle,
                                    hipsparseOperation_t      transA,
                                    hipsparseOperation_t      transB,
                                    int                       m,
                                    int                       n,
                                    int                       k,
                                    const hipsparseMatDescr_t descrA,
                                    int                       nnzA,
                                    const hipComplex*         csrValA,
                                    const int*                csrRowPtrA,
                                    const int*                csrColIndA,
                                    const hipsparseMatDescr_t descrB,
                                    int                       nnzB,
                                    const hipComplex*         csrValB,
                                    const int*                csrRowPtrB,
                                    const int*                csrColIndB,
                                    const hipsparseMatDescr_t descrC,
                                    hipComplex*               csrValC,
                                    const int*                csrRowPtrC,
                                    int*                      csrColIndC)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseCcsrgemm((cusparseHandle_t)handle,
                         hipsparse::hipOperationToCudaOperation(transA),
                         hipsparse::hipOperationToCudaOperation(transB),
                         m,
                         n,
                         k,
                         (const cusparseMatDescr_t)descrA,
                         nnzA,
                         (const cuComplex*)csrValA,
                         csrRowPtrA,
                         csrColIndA,
                         (const cusparseMatDescr_t)descrB,
                         nnzB,
                         (const cuComplex*)csrValB,
                         csrRowPtrB,
                         csrColIndB,
                         (const cusparseMatDescr_t)descrC,
                         (cuComplex*)csrValC,
                         csrRowPtrC,
                         csrColIndC));
}

hipsparseStatus_t hipsparseZcsrgemm(hipsparseHandle_t         handle,
                                    hipsparseOperation_t      transA,
                                    hipsparseOperation_t      transB,
                                    int                       m,
                                    int                       n,
                                    int                       k,
                                    const hipsparseMatDescr_t descrA,
                                    int                       nnzA,
                                    const hipDoubleComplex*   csrValA,
                                    const int*                csrRowPtrA,
                                    const int*                csrColIndA,
                                    const hipsparseMatDescr_t descrB,
                                    int                       nnzB,
                                    const hipDoubleComplex*   csrValB,
                                    const int*                csrRowPtrB,
                                    const int*                csrColIndB,
                                    const hipsparseMatDescr_t descrC,
                                    hipDoubleComplex*         csrValC,
                                    const int*                csrRowPtrC,
                                    int*                      csrColIndC)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseZcsrgemm((cusparseHandle_t)handle,
                         hipsparse::hipOperationToCudaOperation(transA),
                         hipsparse::hipOperationToCudaOperation(transB),
                         m,
                         n,
                         k,
                         (const cusparseMatDescr_t)descrA,
                         nnzA,
                         (const cuDoubleComplex*)csrValA,
                         csrRowPtrA,
                         csrColIndA,
                         (const cusparseMatDescr_t)descrB,
                         nnzB,
                         (const cuDoubleComplex*)csrValB,
                         csrRowPtrB,
                         csrColIndB,
                         (const cusparseMatDescr_t)descrC,
                         (cuDoubleComplex*)csrValC,
                         csrRowPtrC,
                         csrColIndC));
}
#endif

#if CUDART_VERSION < 12000
hipsparseStatus_t hipsparseScsrgemm2_bufferSizeExt(hipsparseHandle_t         handle,
                                                   int                       m,
                                                   int                       n,
                                                   int                       k,
                                                   const float*              alpha,
                                                   const hipsparseMatDescr_t descrA,
                                                   int                       nnzA,
                                                   const int*                csrRowPtrA,
                                                   const int*                csrColIndA,
                                                   const hipsparseMatDescr_t descrB,
                                                   int                       nnzB,
                                                   const int*                csrRowPtrB,
                                                   const int*                csrColIndB,
                                                   const float*              beta,
                                                   const hipsparseMatDescr_t descrD,
                                                   int                       nnzD,
                                                   const int*                csrRowPtrD,
                                                   const int*                csrColIndD,
                                                   csrgemm2Info_t            info,
                                                   size_t*                   pBufferSizeInBytes)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseScsrgemm2_bufferSizeExt((cusparseHandle_t)handle,
                                        m,
                                        n,
                                        k,
                                        alpha,
                                        (const cusparseMatDescr_t)descrA,
                                        nnzA,
                                        csrRowPtrA,
                                        csrColIndA,
                                        (const cusparseMatDescr_t)descrB,
                                        nnzB,
                                        csrRowPtrB,
                                        csrColIndB,
                                        beta,
                                        (const cusparseMatDescr_t)descrD,
                                        nnzD,
                                        csrRowPtrD,
                                        csrColIndD,
                                        (csrgemm2Info_t)info,
                                        pBufferSizeInBytes));
}

hipsparseStatus_t hipsparseDcsrgemm2_bufferSizeExt(hipsparseHandle_t         handle,
                                                   int                       m,
                                                   int                       n,
                                                   int                       k,
                                                   const double*             alpha,
                                                   const hipsparseMatDescr_t descrA,
                                                   int                       nnzA,
                                                   const int*                csrRowPtrA,
                                                   const int*                csrColIndA,
                                                   const hipsparseMatDescr_t descrB,
                                                   int                       nnzB,
                                                   const int*                csrRowPtrB,
                                                   const int*                csrColIndB,
                                                   const double*             beta,
                                                   const hipsparseMatDescr_t descrD,
                                                   int                       nnzD,
                                                   const int*                csrRowPtrD,
                                                   const int*                csrColIndD,
                                                   csrgemm2Info_t            info,
                                                   size_t*                   pBufferSizeInBytes)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseDcsrgemm2_bufferSizeExt((cusparseHandle_t)handle,
                                        m,
                                        n,
                                        k,
                                        alpha,
                                        (const cusparseMatDescr_t)descrA,
                                        nnzA,
                                        csrRowPtrA,
                                        csrColIndA,
                                        (const cusparseMatDescr_t)descrB,
                                        nnzB,
                                        csrRowPtrB,
                                        csrColIndB,
                                        beta,
                                        (const cusparseMatDescr_t)descrD,
                                        nnzD,
                                        csrRowPtrD,
                                        csrColIndD,
                                        (csrgemm2Info_t)info,
                                        pBufferSizeInBytes));
}

hipsparseStatus_t hipsparseCcsrgemm2_bufferSizeExt(hipsparseHandle_t         handle,
                                                   int                       m,
                                                   int                       n,
                                                   int                       k,
                                                   const hipComplex*         alpha,
                                                   const hipsparseMatDescr_t descrA,
                                                   int                       nnzA,
                                                   const int*                csrRowPtrA,
                                                   const int*                csrColIndA,
                                                   const hipsparseMatDescr_t descrB,
                                                   int                       nnzB,
                                                   const int*                csrRowPtrB,
                                                   const int*                csrColIndB,
                                                   const hipComplex*         beta,
                                                   const hipsparseMatDescr_t descrD,
                                                   int                       nnzD,
                                                   const int*                csrRowPtrD,
                                                   const int*                csrColIndD,
                                                   csrgemm2Info_t            info,
                                                   size_t*                   pBufferSizeInBytes)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseCcsrgemm2_bufferSizeExt((cusparseHandle_t)handle,
                                        m,
                                        n,
                                        k,
                                        (const cuComplex*)alpha,
                                        (const cusparseMatDescr_t)descrA,
                                        nnzA,
                                        csrRowPtrA,
                                        csrColIndA,
                                        (const cusparseMatDescr_t)descrB,
                                        nnzB,
                                        csrRowPtrB,
                                        csrColIndB,
                                        (const cuComplex*)beta,
                                        (const cusparseMatDescr_t)descrD,
                                        nnzD,
                                        csrRowPtrD,
                                        csrColIndD,
                                        (csrgemm2Info_t)info,
                                        pBufferSizeInBytes));
}

hipsparseStatus_t hipsparseZcsrgemm2_bufferSizeExt(hipsparseHandle_t         handle,
                                                   int                       m,
                                                   int                       n,
                                                   int                       k,
                                                   const hipDoubleComplex*   alpha,
                                                   const hipsparseMatDescr_t descrA,
                                                   int                       nnzA,
                                                   const int*                csrRowPtrA,
                                                   const int*                csrColIndA,
                                                   const hipsparseMatDescr_t descrB,
                                                   int                       nnzB,
                                                   const int*                csrRowPtrB,
                                                   const int*                csrColIndB,
                                                   const hipDoubleComplex*   beta,
                                                   const hipsparseMatDescr_t descrD,
                                                   int                       nnzD,
                                                   const int*                csrRowPtrD,
                                                   const int*                csrColIndD,
                                                   csrgemm2Info_t            info,
                                                   size_t*                   pBufferSizeInBytes)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseZcsrgemm2_bufferSizeExt((cusparseHandle_t)handle,
                                        m,
                                        n,
                                        k,
                                        (const cuDoubleComplex*)alpha,
                                        (const cusparseMatDescr_t)descrA,
                                        nnzA,
                                        csrRowPtrA,
                                        csrColIndA,
                                        (const cusparseMatDescr_t)descrB,
                                        nnzB,
                                        csrRowPtrB,
                                        csrColIndB,
                                        (const cuDoubleComplex*)beta,
                                        (const cusparseMatDescr_t)descrD,
                                        nnzD,
                                        csrRowPtrD,
                                        csrColIndD,
                                        (csrgemm2Info_t)info,
                                        pBufferSizeInBytes));
}

hipsparseStatus_t hipsparseXcsrgemm2Nnz(hipsparseHandle_t         handle,
                                        int                       m,
                                        int                       n,
                                        int                       k,
                                        const hipsparseMatDescr_t descrA,
                                        int                       nnzA,
                                        const int*                csrRowPtrA,
                                        const int*                csrColIndA,
                                        const hipsparseMatDescr_t descrB,
                                        int                       nnzB,
                                        const int*                csrRowPtrB,
                                        const int*                csrColIndB,
                                        const hipsparseMatDescr_t descrD,
                                        int                       nnzD,
                                        const int*                csrRowPtrD,
                                        const int*                csrColIndD,
                                        const hipsparseMatDescr_t descrC,
                                        int*                      csrRowPtrC,
                                        int*                      nnzTotalDevHostPtr,
                                        const csrgemm2Info_t      info,
                                        void*                     pBuffer)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseXcsrgemm2Nnz((cusparseHandle_t)handle,
                             m,
                             n,
                             k,
                             (const cusparseMatDescr_t)descrA,
                             nnzA,
                             csrRowPtrA,
                             csrColIndA,
                             (const cusparseMatDescr_t)descrB,
                             nnzB,
                             csrRowPtrB,
                             csrColIndB,
                             (const cusparseMatDescr_t)descrD,
                             nnzD,
                             csrRowPtrD,
                             csrColIndD,
                             (const cusparseMatDescr_t)descrC,
                             csrRowPtrC,
                             nnzTotalDevHostPtr,
                             (const csrgemm2Info_t)info,
                             pBuffer));
}

hipsparseStatus_t hipsparseScsrgemm2(hipsparseHandle_t         handle,
                                     int                       m,
                                     int                       n,
                                     int                       k,
                                     const float*              alpha,
                                     const hipsparseMatDescr_t descrA,
                                     int                       nnzA,
                                     const float*              csrValA,
                                     const int*                csrRowPtrA,
                                     const int*                csrColIndA,
                                     const hipsparseMatDescr_t descrB,
                                     int                       nnzB,
                                     const float*              csrValB,
                                     const int*                csrRowPtrB,
                                     const int*                csrColIndB,
                                     const float*              beta,
                                     const hipsparseMatDescr_t descrD,
                                     int                       nnzD,
                                     const float*              csrValD,
                                     const int*                csrRowPtrD,
                                     const int*                csrColIndD,
                                     const hipsparseMatDescr_t descrC,
                                     float*                    csrValC,
                                     const int*                csrRowPtrC,
                                     int*                      csrColIndC,
                                     const csrgemm2Info_t      info,
                                     void*                     pBuffer)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseScsrgemm2((cusparseHandle_t)handle,
                          m,
                          n,
                          k,
                          alpha,
                          (const cusparseMatDescr_t)descrA,
                          nnzA,
                          csrValA,
                          csrRowPtrA,
                          csrColIndA,
                          (const cusparseMatDescr_t)descrB,
                          nnzB,
                          csrValB,
                          csrRowPtrB,
                          csrColIndB,
                          beta,
                          (const cusparseMatDescr_t)descrD,
                          nnzD,
                          csrValD,
                          csrRowPtrD,
                          csrColIndD,
                          (const cusparseMatDescr_t)descrC,
                          csrValC,
                          csrRowPtrC,
                          csrColIndC,
                          (const csrgemm2Info_t)info,
                          pBuffer));
}

hipsparseStatus_t hipsparseDcsrgemm2(hipsparseHandle_t         handle,
                                     int                       m,
                                     int                       n,
                                     int                       k,
                                     const double*             alpha,
                                     const hipsparseMatDescr_t descrA,
                                     int                       nnzA,
                                     const double*             csrValA,
                                     const int*                csrRowPtrA,
                                     const int*                csrColIndA,
                                     const hipsparseMatDescr_t descrB,
                                     int                       nnzB,
                                     const double*             csrValB,
                                     const int*                csrRowPtrB,
                                     const int*                csrColIndB,
                                     const double*             beta,
                                     const hipsparseMatDescr_t descrD,
                                     int                       nnzD,
                                     const double*             csrValD,
                                     const int*                csrRowPtrD,
                                     const int*                csrColIndD,
                                     const hipsparseMatDescr_t descrC,
                                     double*                   csrValC,
                                     const int*                csrRowPtrC,
                                     int*                      csrColIndC,
                                     const csrgemm2Info_t      info,
                                     void*                     pBuffer)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseDcsrgemm2((cusparseHandle_t)handle,
                          m,
                          n,
                          k,
                          alpha,
                          (const cusparseMatDescr_t)descrA,
                          nnzA,
                          csrValA,
                          csrRowPtrA,
                          csrColIndA,
                          (const cusparseMatDescr_t)descrB,
                          nnzB,
                          csrValB,
                          csrRowPtrB,
                          csrColIndB,
                          beta,
                          (const cusparseMatDescr_t)descrD,
                          nnzD,
                          csrValD,
                          csrRowPtrD,
                          csrColIndD,
                          (const cusparseMatDescr_t)descrC,
                          csrValC,
                          csrRowPtrC,
                          csrColIndC,
                          (const csrgemm2Info_t)info,
                          pBuffer));
}

hipsparseStatus_t hipsparseCcsrgemm2(hipsparseHandle_t         handle,
                                     int                       m,
                                     int                       n,
                                     int                       k,
                                     const hipComplex*         alpha,
                                     const hipsparseMatDescr_t descrA,
                                     int                       nnzA,
                                     const hipComplex*         csrValA,
                                     const int*                csrRowPtrA,
                                     const int*                csrColIndA,
                                     const hipsparseMatDescr_t descrB,
                                     int                       nnzB,
                                     const hipComplex*         csrValB,
                                     const int*                csrRowPtrB,
                                     const int*                csrColIndB,
                                     const hipComplex*         beta,
                                     const hipsparseMatDescr_t descrD,
                                     int                       nnzD,
                                     const hipComplex*         csrValD,
                                     const int*                csrRowPtrD,
                                     const int*                csrColIndD,
                                     const hipsparseMatDescr_t descrC,
                                     hipComplex*               csrValC,
                                     const int*                csrRowPtrC,
                                     int*                      csrColIndC,
                                     const csrgemm2Info_t      info,
                                     void*                     pBuffer)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseCcsrgemm2((cusparseHandle_t)handle,
                          m,
                          n,
                          k,
                          (const cuComplex*)alpha,
                          (const cusparseMatDescr_t)descrA,
                          nnzA,
                          (const cuComplex*)csrValA,
                          csrRowPtrA,
                          csrColIndA,
                          (const cusparseMatDescr_t)descrB,
                          nnzB,
                          (const cuComplex*)csrValB,
                          csrRowPtrB,
                          csrColIndB,
                          (const cuComplex*)beta,
                          (const cusparseMatDescr_t)descrD,
                          nnzD,
                          (const cuComplex*)csrValD,
                          csrRowPtrD,
                          csrColIndD,
                          (const cusparseMatDescr_t)descrC,
                          (cuComplex*)csrValC,
                          csrRowPtrC,
                          csrColIndC,
                          (const csrgemm2Info_t)info,
                          pBuffer));
}

hipsparseStatus_t hipsparseZcsrgemm2(hipsparseHandle_t         handle,
                                     int                       m,
                                     int                       n,
                                     int                       k,
                                     const hipDoubleComplex*   alpha,
                                     const hipsparseMatDescr_t descrA,
                                     int                       nnzA,
                                     const hipDoubleComplex*   csrValA,
                                     const int*                csrRowPtrA,
                                     const int*                csrColIndA,
                                     const hipsparseMatDescr_t descrB,
                                     int                       nnzB,
                                     const hipDoubleComplex*   csrValB,
                                     const int*                csrRowPtrB,
                                     const int*                csrColIndB,
                                     const hipDoubleComplex*   beta,
                                     const hipsparseMatDescr_t descrD,
                                     int                       nnzD,
                                     const hipDoubleComplex*   csrValD,
                                     const int*                csrRowPtrD,
                                     const int*                csrColIndD,
                                     const hipsparseMatDescr_t descrC,
                                     hipDoubleComplex*         csrValC,
                                     const int*                csrRowPtrC,
                                     int*                      csrColIndC,
                                     const csrgemm2Info_t      info,
                                     void*                     pBuffer)
{
    return hipsparse::hipCUSPARSEStatusToHIPStatus(
        cusparseZcsrgemm2((cusparseHandle_t)handle,
                          m,
                          n,
                          k,
                          (const cuDoubleComplex*)alpha,
                          (const cusparseMatDescr_t)descrA,
                          nnzA,
                          (const cuDoubleComplex*)csrValA,
                          csrRowPtrA,
                          csrColIndA,
                          (const cusparseMatDescr_t)descrB,
                          nnzB,
                          (const cuDoubleComplex*)csrValB,
                          csrRowPtrB,
                          csrColIndB,
                          (const cuDoubleComplex*)beta,
                          (const cusparseMatDescr_t)descrD,
                          nnzD,
                          (const cuDoubleComplex*)csrValD,
                          csrRowPtrD,
                          csrColIndD,
                          (const cusparseMatDescr_t)descrC,
                          (cuDoubleComplex*)csrValC,
                          csrRowPtrC,
                          csrColIndC,
                          (const csrgemm2Info_t)info,
                          pBuffer));
}
#endif
