/*
 * Argus Software.  Argus files - Input processing
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * ArgusSource.c - Argus packet source routines.
 *
 * written by Carter Bullard
 * QoSient, LLC
 * Tue Aug  8 08:13:36 EDT 2000
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */

/*
#define ARGUS_NEW_INTERFACE_STRATEGY   1   
*/

#if !defined(ArgusSource)
#define ArgusSource
#endif

#if defined(__APPLE_CC__) || defined(__APPLE__)
#define PCAP_DONT_INCLUDE_PCAP_BPF_H
#include <sys/ioctl.h>
#include <net/bpf.h>
#endif

#include <compat.h>
#include <ArgusModeler.h>

#include <sys/mman.h>
#include <net/ppp.h>
#if !defined(PPP_HDRLEN)
#define PPP_HDRLEN      4       /* length of PPP header */
#endif


void ArgusGetInterfaceStatus (struct ArgusSourceStruct *src);


struct ArgusSourceStruct *
ArgusNewSource(struct ArgusModelerStruct *model)
{
   struct ArgusSourceStruct *retn =  NULL;
 
   if ((retn = (struct ArgusSourceStruct *) ArgusCalloc (1, sizeof (struct ArgusSourceStruct))) == NULL)
      ArgusLog (LOG_ERR, "ArgusNewSource: ArgusCalloc error %s\n", strerror(errno));

   retn->ArgusModel = model;
   retn->sNflag = -1;
   retn->eNflag = -1;
   retn->ArgusSnapLen = ARGUS_MINSNAPLEN;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusNewSource() returning 0x%x\n", retn);
#endif

   return (retn);
}


#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

void ArgusOpenInterface(struct ArgusSourceStruct *, struct ArgusInterfaceStruct *);

void
ArgusOpenInterface(struct ArgusSourceStruct *src, struct ArgusInterfaceStruct *inf)
{
   struct ArgusDeviceStruct *device = inf->ArgusDevice;
   char errbuf[PCAP_ERRBUF_SIZE];
   extern int ArgusShutDownFlag;
   int type;

   if (ArgusShutDownFlag) {
      ArgusShutDown(0);
      return;
   }

   if ((device == NULL) || (device->name == NULL))
      return;

   if ((inf->ArgusPd = pcap_open_live(device->name, src->ArgusSnapLen, !src->Arguspflag, 100, errbuf)) != NULL) {
      pcap_setnonblock(inf->ArgusPd, 1, errbuf);

#if defined(__APPLE_CC__) || defined(__APPLE__)
      int v = 1;
      ioctl(pcap_fileno(inf->ArgusPd), BIOCIMMEDIATE, &v);
#endif
#ifdef ARGUSDEBUG
      ArgusDebug (1, "ArgusOpenInterface() pcap_open_live(%s) returned 0x%x\n", device->name, inf->ArgusPd);
#endif
      src->ArgusInputPacketFileType = ARGUSLIBPPKTFILE;
      inf->ArgusInterfaceType = ARGUSLIBPPKTFILE;
      memset((char *)&inf->ifr, 0, sizeof(inf->ifr));
      strncpy(inf->ifr.ifr_name, device->name, sizeof(inf->ifr.ifr_name));
      if (!((pcap_lookupnet (device->name, (u_int *)&inf->ArgusLocalNet,
                                           (u_int *)&inf->ArgusNetMask, errbuf)) < 0)) {
#if defined(_LITTLE_ENDIAN)
         inf->ArgusLocalNet = ntohl(inf->ArgusLocalNet);
         inf->ArgusNetMask  = ntohl(inf->ArgusNetMask);
#endif
      }

      type = pcap_datalink(inf->ArgusPd);

      if ((inf->ArgusCallBack = Arguslookup_pcap_callback(type)) == NULL)
         ArgusLog (LOG_ERR, "unsupported device type %d\n", type);

   } else {
      if (!(strstr(errbuf, "Network is down")))
         ArgusLog (LOG_ERR, "ArgusOpenInterface: pcap_open_live %s\n", errbuf);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusOpenInterface(0x%x, '%s') returning\n", src, inf->ArgusDevice->name);
#endif
}

#if defined(CYGWIN)
#include <pcap.h>
#endif

void
ArgusInitSource (struct ArgusSourceStruct *src)
{
   char errbuf[PCAP_ERRBUF_SIZE];
   char *cmdbuf = NULL;
   int i = 0;

   src->ArgusInterfaces = 0;
   for (i = 0; i < ARGUS_MAXINTERFACE; i++)
      bzero ((char *)&src->ArgusInterface[i], sizeof(struct ArgusInterfaceStruct));
   i = 0;
   if (src->ArgusRfileList != NULL) {
      setuid(getuid());
      bzero(errbuf, sizeof(errbuf));

      if (ArgusOpenInputPacketFile(src, errbuf) == 0)
         ArgusLog (LOG_ERR, "ArgusInitSource: %s", errbuf);

   } else {
      if (src->ArgusDeviceList == NULL) {
         pcap_if_t *d;

         if (pcap_findalldevs(&src->ArgusPacketDevices, errbuf) == -1)
            ArgusLog (LOG_ERR, "ArgusInitSource: pcap_findalldevs_ex %s\n", errbuf);

         for (d = src->ArgusPacketDevices; d != NULL; d = d->next) {
#if defined(CYGWIN)
            printf ("%d. %s", ++i, d->name);
            if (d->description)
               printf (" (%s)\n", d->description);
            else
               printf ("\n");
#else
            i++;
#if defined(ARGUS_NEW_INTERFACE_STRATEGY)
            setArgusDevice (src, d->name);
#endif
#endif
         }
#if defined(ARGUS_NEW_INTERFACE_STRATEGY)
         if (i == 0) 
            ArgusLog (LOG_ERR, "ArgusInitSource: no interfaces\n");
#endif

         pcap_freealldevs(src->ArgusPacketDevices);

#if defined(CYGWIN)
         exit(1);
#else
#if !defined(ARGUS_NEW_INTERFACE_STRATEGY)
         setArgusDevice (src, pcap_lookupdev (errbuf));
#endif
#endif
      }

      if (src->ArgusDeviceList) {
         int count = src->ArgusDeviceList->count;

         for (i = 0; i < count; i++) {
            struct ArgusDeviceStruct *device = (struct ArgusDeviceStruct *) ArgusPopFrontList(src->ArgusDeviceList, ARGUS_LOCK);

            if (device != NULL) {
               src->ArgusInterface[i].ArgusDevice = device;
               ArgusOpenInterface(src, &src->ArgusInterface[i]);
               src->ArgusInterfaces++;
               ArgusPushBackList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_LOCK);
            }
         }
      }

      setuid(getuid());
   }

/*
   if (src->ArgusInterface[0].ArgusPd == NULL)
      ArgusLog (LOG_ERR, "ArgusInitSource: no input source\n");
*/ 

   cmdbuf = ArgusCopyArgv(&src->ArgusArgv[src->ArgusOptind]);

   if (cmdbuf) {
      if (src->ArgusInputFilter)
         ArgusFree(src->ArgusInputFilter);

      src->ArgusInputFilter = cmdbuf;
   }

   bzero ((char *) &src->ArgusInterface[0].ArgusFilter, sizeof (struct bpf_program));

   if (src->ArgusInterface[0].ArgusPd) {
      if (pcap_compile (src->ArgusInterface[0].ArgusPd, &src->ArgusInterface[0].ArgusFilter, src->ArgusInputFilter, getArgusOflag(src), src->ArgusInterface[0].ArgusNetMask) < 0)
         ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));

      if (src->Argusbpflag) {
         Argusbpf_dump (&src->ArgusInterface[0].ArgusFilter, src->Argusbpflag);
         exit(0);
      }

      for (i = 0; i < src->ArgusInterfaces; i++) {
         if (src->ArgusInterface[i].ArgusPd) {
            if (src->ArgusInputPacketFileType == ARGUSLIBPPKTFILE) {
               if (src->ArgusInputFilter != NULL) {
                  if (pcap_setfilter (src->ArgusInterface[i].ArgusPd, &src->ArgusInterface[0].ArgusFilter) < 0)
                     ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[i].ArgusPd));
               }
            }
         }
      }
   }

   if (src->ArgusWriteOutPacketFile) {
      if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
         ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusInitSource() returning\n");
#endif
}


int
ArgusCloseSource(struct ArgusSourceStruct *src)
{

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusCloseSource(0x%x) starting\n", src);
#endif

   if (src) {
      if (src->ArgusPcapOutFile)
         pcap_dump_close(src->ArgusPcapOutFile);

      if (src->ArgusInputFilter)
         ArgusFree (src->ArgusInputFilter);

      if (src->ArgusDeviceList)
         ArgusDeleteList(src->ArgusDeviceList, ARGUS_DEVICE_LIST);

      if (src->ArgusRfileList != NULL)
         ArgusDeleteList (src->ArgusRfileList, ARGUS_RFILE_LIST);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusCloseSource(0x%x) deleting source\n", src);
#endif
   return (0);
}


unsigned char
getArgusInterfaceStatus(struct ArgusSourceStruct *src) {
   return (src->ArgusInterfaceStatus);
}
 
void
setArgusInterfaceStatus(struct ArgusSourceStruct *src, unsigned char value)
{
   src->ArgusInterfaceStatus = value;
 
#ifdef ARGUSDEBUG
   ArgusDebug (1, "setArgusInterfaceStatus(%d)\n", value);
#endif
}

 
int
getArgusSnapLen(struct ArgusSourceStruct *src)
{
   return (src->ArgusSnapLen);
}
 
void
setArgusSnapLen(struct ArgusSourceStruct *src, int value)
{
   src->ArgusSnapLen = value;
}

int
getArgusbpflag(struct ArgusSourceStruct *src)
{
   return (src->Argusbpflag);
}

int
getArguspflag(struct ArgusSourceStruct *src)
{
   return (src->Arguspflag);
}

int
getArgusOflag(struct ArgusSourceStruct *src)
{
   return (src->ArgusOflag);
}

void
setArgusbpflag(struct ArgusSourceStruct *src, int value)
{
   src->Argusbpflag = value;
}

void
setArguspflag(struct ArgusSourceStruct *src, int value)
{
   src->Arguspflag = value;
}

void
setArgusOflag(struct ArgusSourceStruct *src, int value)
{
   src->ArgusOflag = value;
}

void
setArgusArgv(struct ArgusSourceStruct *src, char **value)
{
   src->ArgusArgv = value;
}

void
setArgusOptind (struct ArgusSourceStruct *src, int value)
{
   src->ArgusOptind = value;
}

char *
getArgusDevice (struct ArgusSourceStruct *src)
{
   struct ArgusDeviceStruct *device = NULL;
   char *retn = NULL;

   if (src->ArgusDeviceList != NULL)
      if ((device = (struct ArgusDeviceStruct *) ArgusPopFrontList(src->ArgusDeviceList, ARGUS_LOCK)) != NULL)
         ArgusPushFrontList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_LOCK);

   if (device != NULL)
      retn = device->name;
#ifdef ARGUSDEBUG
   ArgusDebug (3, "getArgusDevice() returning %s\n", retn);
#endif
   return (retn);
}


void
setArgusDevice (struct ArgusSourceStruct *src, char *value)
{
   if (src->ArgusDeviceList == NULL)
      src->ArgusDeviceList = ArgusNewList();

   if (value) {
      struct ArgusDeviceStruct *device = NULL;
      if ((device = (struct ArgusDeviceStruct *) ArgusCalloc(1, sizeof(*device))) == NULL) {
         ArgusLog (LOG_ERR, "setArgusDevice ArgusCalloc %s\n", strerror(errno));
      } else {

#if defined(CYGWIN)
         int i = 0, num, ref = 0;
         pcap_if_t *alldevs, *d;
         char *ptr = NULL;
         char errbuf[PCAP_ERRBUF_SIZE];

         if (pcap_findalldevs(&alldevs, errbuf) == -1)
            ArgusLog (LOG_ERR, "ArgusInitSource: pcap_findalldevs %s\n", errbuf);
         num = (int)strtol(value, (char **)&ptr, 10);
         if (ptr != value)
            ref = 1;
            
         for (d = alldevs; d != NULL; d = d->next) {
            i++;
            if (ref) {
               if (i == num) {
                  device->name = strdup(d->name);
                  break;
               }
            } else {
               if (!strcmp(value, d->name)) {
                  device->name = strdup(d->name);
                  break;
               }
            }
         }
         if (i == 0) 
            ArgusLog (LOG_ERR, "ArgusInitSource: no interfaces\n");
         pcap_freealldevs(alldevs);
#else
         device->name = strdup(value);
#endif
         ArgusPushFrontList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_LOCK);
      }
   } else {
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "setArgusDevice(%s) returning\n", value);
#endif
}

void
clearArgusDevice (struct ArgusSourceStruct *src)
{
   if (src->ArgusDeviceList != NULL) {
      ArgusDeleteList(src->ArgusDeviceList, ARGUS_DEVICE_LIST);
      src->ArgusDeviceList = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "clearArgusDevice(0x%x) returning\n", src);
#endif
}


char *
getArgusrfile (struct ArgusSourceStruct *src)
{
   struct ArgusRfileStruct *rfile = NULL;
   char *retn = NULL;

   if (src->ArgusRfileList != NULL) {
      rfile = (struct ArgusRfileStruct *) src->ArgusRfileList->start;
      retn = rfile->name;
   }
   return (retn);
}


void ArgusSortFileList (struct ArgusListStruct *);

void
setArgusrfile (struct ArgusSourceStruct *src, char *value)
{
   if (src->ArgusRfileList == NULL)
      src->ArgusRfileList = ArgusNewList();

   if (value) {
      struct ArgusRfileStruct *rfile;
      struct stat statbuf;
      char *tok, *ptr = value;

      while ((tok = strtok (ptr, " \t")) != NULL) {
         if (strcmp("-", tok))
            if (stat(tok, &statbuf) < 0)
               ArgusLog (LOG_ERR, "input file '%s': %s", tok, strerror(errno));

         if ((rfile = (struct ArgusRfileStruct *) ArgusCalloc(1, sizeof(*rfile))) == NULL)
            ArgusLog (LOG_ERR, "setArgusrfile ArgusCalloc %s\n", strerror(errno));

         rfile->name = strdup(tok);
         ArgusPushBackList(src->ArgusRfileList, (struct ArgusListRecord *) rfile, ARGUS_LOCK);
         ptr = NULL;
      }

      ArgusSortFileList (src->ArgusRfileList);
   }
}

int
getArgusMoatTshFile (struct ArgusSourceStruct *src)
{
   return(src->Argustflag);
}

void
setArgusMoatTshFile (struct ArgusSourceStruct *src, int value)
{
   src->Argustflag = value;
}

float
getArgusRealTime (struct ArgusSourceStruct *src)
{
   return(src->Tflag);
}


void
setArgusRealTime (struct ArgusSourceStruct *src, float value)
{
   src->Tflag = value;
}


void
setArgusWriteOutPacketFile (struct ArgusSourceStruct *src, char *file)
{
   src->ArgusWriteOutPacketFile = strdup(file);
}


#define ARGUSMOATLEN      44
#define ARGUSMOATTSHTCPLEN   40

int ArgusMoatTshRead (struct ArgusSourceStruct *);

int
ArgusMoatTshRead (struct ArgusSourceStruct *src)
{
   struct ArgusMoatTshPktHdr MoatTshBuffer[2], *ArgusMoatPktHdr = &MoatTshBuffer[0];
   int retn = 0, length = 0;
   struct ip *iphdr = NULL;

   bzero (ArgusMoatPktHdr, sizeof(MoatTshBuffer));
 
   if ((retn = read(pcap_fileno(src->ArgusInterface[0].ArgusPd), ArgusMoatPktHdr, ARGUSMOATLEN)) == ARGUSMOATLEN) {
      ArgusMoatPktHdr->interface = 0;
#if defined(_LITTLE_ENDIAN)
      src->ArgusModel->ArgusGlobalTime.tv_sec  = ntohl(ArgusMoatPktHdr->sec);
      src->ArgusModel->ArgusGlobalTime.tv_usec = ntohl(*((int *)&ArgusMoatPktHdr->interface));
#else
      src->ArgusModel->ArgusGlobalTime.tv_sec  = ArgusMoatPktHdr->sec;
      src->ArgusModel->ArgusGlobalTime.tv_usec = *((int *)&ArgusMoatPktHdr->interface);
#endif

      iphdr = &ArgusMoatPktHdr->ip;

#if defined(_LITTLE_ENDIAN)
      length = ntohs(iphdr->ip_len);
#else
      length = iphdr->ip_len;
#endif
      src->ArgusThisLength  = length;

      switch (iphdr->ip_p) {
         case IPPROTO_ICMP:
         case IPPROTO_TCP:
         default:
            src->ArgusSnapLength  = ARGUSMOATTSHTCPLEN;
            break;
      }

      src->ArgusThisSnapEnd = (((unsigned char *)iphdr) + src->ArgusSnapLength);

      if ((src->ArgusInputFilter == NULL) ||
           (bpf_filter(src->ArgusInterface[0].ArgusFilter.bf_insns, (u_char *)iphdr, src->ArgusSnapLength, src->ArgusSnapLen))) {

         ArgusProcessIpPacket (src->ArgusModel, iphdr, length, &src->ArgusModel->ArgusGlobalTime);
      }

   } else
      close(pcap_fileno(src->ArgusInterface[0].ArgusPd));

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusMoatTshRead() returning %d\n", retn);
#endif

   return (retn);
}


int
ArgusSnoopRead (struct ArgusSourceStruct *src)
{
   int retn = 0, len = 0;
   struct pcap_pkthdr pcap_pkthdr;
   struct ArgusSnoopPktHdr SnoopPktHdr;
   unsigned char ArgusPacketBuf[1540];

   if ((retn = read(pcap_fileno(src->ArgusInterface[0].ArgusPd), &SnoopPktHdr, sizeof(SnoopPktHdr))) == sizeof(SnoopPktHdr)) {
#if defined(_LITTLE_ENDIAN)
      SnoopPktHdr.len            = ntohl(SnoopPktHdr.len);
      SnoopPktHdr.tlen           = ntohl(SnoopPktHdr.tlen);
      SnoopPktHdr.argtvp.tv_sec  = ntohl(SnoopPktHdr.argtvp.tv_sec);
      SnoopPktHdr.argtvp.tv_usec = ntohl(SnoopPktHdr.argtvp.tv_usec);
#endif
      if ((len = ((SnoopPktHdr.tlen + 3) & 0xFFFFFFC)) < 1500) {
         if ((retn = read(pcap_fileno(src->ArgusInterface[0].ArgusPd), ArgusPacketBuf, len)) == len) {
            pcap_pkthdr.ts.tv_sec  = SnoopPktHdr.argtvp.tv_sec;
            pcap_pkthdr.ts.tv_usec = SnoopPktHdr.argtvp.tv_usec;
            pcap_pkthdr.caplen = SnoopPktHdr.tlen;
            pcap_pkthdr.len    = SnoopPktHdr.len;

            if ((src->ArgusInputFilter == NULL) ||
               (bpf_filter(src->ArgusInterface[0].ArgusFilter.bf_insns, ArgusPacketBuf, SnoopPktHdr.tlen, src->ArgusSnapLen))) {
 
               src->ArgusInterface[0].ArgusCallBack (NULL, &pcap_pkthdr, ArgusPacketBuf);
            }
         }
      }

   } else
      close(pcap_fileno(src->ArgusInterface[0].ArgusPd));

#ifdef ARGUSDEBUG
   ArgusDebug (5, "ArgusSnoopRead() returning %d\n", retn);
#endif

   return (retn);
}


pcap_handler
Arguslookup_pcap_callback (int type)
{
   pcap_handler retn = NULL;
   struct callback *callback;
 
   for (callback = ArgusSourceCallbacks; callback->function; ++callback)
      if (type == callback->type) {
         retn = callback->function;
         break;
      }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "Arguslookup_pcap_callback(%d) returning 0x%x\n", type, retn);
#endif

   return (retn);
}

int ArgusProcessPacket (struct ArgusSourceStruct *, char *, int, struct timeval *, int);

void
ArgusEtherPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   unsigned int caplen = h->caplen;
   unsigned int length = h->len;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   struct stat statbuf;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (p != NULL) {
      unsigned int ind = src->ArgusThisIndex;

#define ARGUS_TIME_THRESHOLD   300

      if (src->lasttime.tv_sec) {
         if ((tvp->tv_sec + ARGUS_TIME_THRESHOLD) < src->lasttime.tv_sec) {
            ArgusLog (LOG_WARNING, "ArgusInterface timestamps wayyy out of order: now %d then %d\n",
                          tvp->tv_sec, src->lasttime.tv_sec);
         }
      }
         
      src->ArgusModel->ArgusGlobalTime  = *tvp;
      src->lasttime = *tvp;

      if (src->ArgusWriteOutPacketFile) {
         if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
            if (src->ArgusPcapOutFile != NULL) {
               pcap_dump_close(src->ArgusPcapOutFile);
               src->ArgusPcapOutFile = NULL;
            }
   
            if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
               ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
         }

         if (src->ArgusPcapOutFile != NULL)
            pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
      }

      if (p && (length >= sizeof(struct ether_header))) {
         struct ether_header *ep;
 
         src->ArgusInterface[ind].ArgusTotalPkts++;
         src->ArgusInterface[ind].ArgusTotalBytes += length;
         src->ArgusInterface[ind].ArgusPacketBuffer = (u_char *) p;
         ep = (struct ether_header *) p;

         src->ArgusModel->ArgusThisLength  = length;
         src->ArgusModel->ArgusSnapLength  = caplen;
         src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;
         src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_ETHER;

         ArgusProcessPacket (src, (char *)ep, length, tvp, ARGUS_ETHER_HDR);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusEtherPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

#define TYPE_LEGACY       0
#define TYPE_HDLC_POS     1
#define TYPE_ETH          2
#define TYPE_ATM          3
#define TYPE_AAL5         4

void
ArgusTokenPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusTokenPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


#include <net/arcnet.h>

void
ArgusArcnetPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct ether_header *ep = (struct ether_header *) src->ArgusInterface[src->ArgusThisIndex].ArgusPacketBufferBuffer;
   struct arc_header *ap = (struct arc_header *) p;
   u_char arc_type = ap->arc_type;
   struct stat statbuf;
   int archdrlen = 0;

   struct timeval tvpbuf, *tvp = &tvpbuf;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   if (src->ArgusModel->ArgusGlobalTime.tv_sec < 0) {
      ArgusLog (LOG_ERR, "ArgusArcnetPacket (0x%x, 0x%x, 0x%x) libpcap timestamp out of range %d.%d\n",
              user, h, p, src->ArgusModel->ArgusGlobalTime.tv_sec, src->ArgusModel->ArgusGlobalTime.tv_usec);
   }

   src->ArgusInterface[src->ArgusThisIndex].ArgusPacketBuffer = src->ArgusInterface[src->ArgusThisIndex].ArgusPacketBufferBuffer; 

   switch (arc_type) {
      case ARCTYPE_IP_OLD:
      case ARCTYPE_ARP_OLD:
      case ARCTYPE_DIAGNOSE:
         archdrlen = ARC_HDRLEN;
         break;

      default:
         if (ap->arc_flag == 0xff) {
            archdrlen = ARC_HDRNEWLEN_EXC;
         } else {
            archdrlen = ARC_HDRNEWLEN;
         }
         break;
   }

   length -= archdrlen;
   caplen -= archdrlen;
   p += archdrlen;
   
   bcopy (p, (char *)ep, caplen);

   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;
   src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_ARCNET;

   ArgusProcessPacket (src, (char *)ep, length, tvp, ARGUS_ETHER_HDR);

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusArcnetPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusAtmClipPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusAtmClipPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusLoopPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusLoopPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusHdlcPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   int ind = src->ArgusThisIndex;
   struct stat statbuf;

   struct timeval tvpbuf, *tvp = &tvpbuf;
   unsigned int caplen = h->caplen;
   unsigned int length = h->len;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusInterface[ind].ArgusTotalPkts++;
   src->ArgusInterface[ind].ArgusTotalBytes += length;
   src->ArgusInterface[ind].ArgusPacketBuffer = (u_char *) p;

   src->ArgusModel->ArgusGlobalTime = *tvp;

   if (p && length) {
      unsigned short proto = 0;
      unsigned short value;
      int offset = 0;

#define CHDLC_UNICAST           0x0f
#define CHDLC_BCAST             0x8f
#define CHDLC_TYPE_SLARP        0x8035
#define CHDLC_TYPE_CDP          0x2000

#if defined(_LITTLE_ENDIAN)
      value = ntohs(*(unsigned short *)p);
#else
      value = (*(unsigned short *)p);
#endif
      switch (value) {
         case CHDLC_UNICAST:
         case CHDLC_BCAST:
         case CHDLC_TYPE_SLARP:
         case CHDLC_TYPE_CDP:
         default:
            src->ArgusModel->ArgusThisEncaps = ARGUS_ENCAPS_CHDLC;
            break;
      }

#if defined(_LITTLE_ENDIAN)
      proto = ntohs(*(u_short *)&p[2]);
#else
      proto = *(u_short *)&p[2];
#endif

#if !defined(CHDLC_HDRLEN)
#define CHDLC_HDRLEN      4
#endif
      offset = CHDLC_HDRLEN;
      p = (unsigned char *) (src->ArgusInterface[ind].ArgusPacketBuffer + offset);
      length -= offset;

      src->ArgusModel->ArgusThisLength  = length;
      src->ArgusModel->ArgusSnapLength  = caplen;
      src->ArgusModel->ArgusThisSnapEnd = (src->ArgusInterface[ind].ArgusPacketBuffer + caplen);

      switch (proto) {
         case ETHERTYPE_IP:
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_IP);
            break;

         case ETHERTYPE_IPV6:
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_IPV6);
            break;

         case ETHERTYPE_MPLS:
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_MPLS);
            break;

         case ETHERTYPE_MPLS_MULTI:
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_MPLS_MULTI);
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusHdlcPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusPppHdlcPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   int ind = src->ArgusThisIndex;
   struct stat statbuf;

   struct timeval tvpbuf, *tvp = &tvpbuf;
   unsigned int caplen = h->caplen;
   unsigned int length = h->len;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusInterface[ind].ArgusTotalPkts++;
   src->ArgusInterface[ind].ArgusTotalBytes += length;
   src->ArgusInterface[ind].ArgusPacketBuffer = (u_char *) p;

   src->ArgusModel->ArgusGlobalTime = *tvp;
   src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_HDLC;

   if (p && length) {
      unsigned short proto = 0;
      unsigned short value;
      int offset = 0;

#if defined(_LITTLE_ENDIAN)
      value = ntohs(*(unsigned short *)p);
#else
      value = (*(unsigned short *)p);
#endif
      if (value == 0xFF03) {
         src->ArgusModel->ArgusThisEncaps  |= ARGUS_ENCAPS_PPP;
         if ((p[0] == PPP_ADDRESS) && (p[1] == PPP_CONTROL)) {
            p += 2; length -= 2;
         }
         if (*p & 01) {
            proto = *p; p++; length -= 1;
         } else {
#if defined(_LITTLE_ENDIAN)
            proto = ntohs(*(u_short *)p);
#else
            proto = *(u_short *)p;
#endif
            p += 2; length -= 2;
         }
      } else {
#if defined(_LITTLE_ENDIAN)
         proto = ntohs(*(u_short *)&p[2]);
#else
         proto = *(u_short *)p;
#endif

#if !defined(CHDLC_HDRLEN)
#define CHDLC_HDRLEN      4
#endif
         src->ArgusModel->ArgusThisEncaps = ARGUS_ENCAPS_CHDLC;
         offset = CHDLC_HDRLEN;
         p = (unsigned char *) (src->ArgusInterface[ind].ArgusPacketBuffer + offset);
         length -= offset;
      }

      src->ArgusModel->ArgusThisLength  = length;
      src->ArgusModel->ArgusSnapLength  = caplen;
      src->ArgusModel->ArgusThisSnapEnd = (src->ArgusInterface[ind].ArgusPacketBuffer + caplen);

      switch (proto) {
         case ETHERTYPE_IP:      /*XXX*/
         case PPP_IP:
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_IP);
            break;

         case PPP_IPV6:
         case ETHERTYPE_IPV6:    /*XXX*/
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_IPV6);
            break;

         case PPP_MPLS_UCAST:
         case ETHERTYPE_MPLS:
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_MPLS);
            break;

         case PPP_MPLS_MCAST:
         case ETHERTYPE_MPLS_MULTI:
            ArgusProcessPacket (src, (char *)p, length, tvp, ETHERTYPE_MPLS_MULTI);
            break;

         case PPP_OSI:
         case PPP_NS:
         case PPP_DECNET:
         case PPP_APPLE:
         case PPP_IPX:
         case PPP_VJC:
         case PPP_VJNC:
         case PPP_BRPDU:
         case PPP_STII:
         case PPP_VINES:
         case PPP_COMP:
         case PPP_HELLO:
         case PPP_LUXCOM:
         case PPP_SNS:
         case PPP_IPCP:
         case PPP_OSICP:
         case PPP_NSCP:
         case PPP_DECNETCP:
         case PPP_APPLECP:
         case PPP_IPXCP:
         case PPP_STIICP:
         case PPP_VINESCP:
         case PPP_IPV6CP:
         case PPP_CCP:
         case PPP_PAP:
         case PPP_LQM:
         case PPP_CHAP:
         case PPP_BACP:
         case PPP_BAP:
         case PPP_MP:

         default:
            break;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppHdlcPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusPppEtherPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppEtherPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

#include <ieee802_11.h>

int ArgusExtract802_11HeaderLength(u_int16_t);

int
ArgusExtract802_11HeaderLength(u_int16_t fc)
{
        switch (FC_TYPE(fc)) {
        case T_MGMT:
                return MGMT_HDRLEN;
        case T_CTRL:
                switch (FC_SUBTYPE(fc)) {
                case CTRL_PS_POLL:
                        return CTRL_PS_POLL_HDRLEN;
                case CTRL_RTS:
                        return CTRL_RTS_HDRLEN;
                case CTRL_CTS:
                        return CTRL_CTS_HDRLEN;
                case CTRL_ACK:
                        return CTRL_ACK_HDRLEN;
                case CTRL_CF_END:
                        return CTRL_END_HDRLEN;
                case CTRL_END_ACK:
                        return CTRL_END_ACK_HDRLEN;
                default:
                        return 0;
                }
        case T_DATA:
                return (FC_TO_DS(fc) && FC_FROM_DS(fc)) ? 30 : 24;
        default:
                printf("unknown IEEE802.11 frame type (%d)", FC_TYPE(fc));
                return 0;
        }
}


void
Argus802_11Packet (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   struct stat statbuf;
   u_int16_t fc;
   u_int hdrlen;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;
   
   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   fc = EXTRACT_LE_16BITS(p);
   hdrlen = ArgusExtract802_11HeaderLength(fc);

   if (caplen > hdrlen) {
      switch (FC_TYPE(fc)) {
         case T_MGMT:
         case T_CTRL:
            break;
         case T_DATA: {
            src->ArgusModel->ArgusGlobalTime  = *tvp;
            src->ArgusModel->ArgusSnapLength  = caplen;
            src->ArgusModel->ArgusThisSnapEnd = (u_char *)(p + caplen);
            src->ArgusModel->ArgusThisEncaps  = 0;

            src->ArgusModel->ArgusThisLength  = length;

            if (FC_WEP(fc)) {
            } else {
               src->ArgusModel->ArgusThisEncaps = ARGUS_ENCAPS_802_11;
               ArgusProcessPacket (src, (char *)p, length, tvp, ARGUS_802_11_HDR);
            }
         }
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "Argus802_11Packet (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

void
ArgusLtalkPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusLtalkPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

#define PRISM_HDR_LEN      144

#define WLANCAP_MAGIC_COOKIE_V1   0x80211001

#define DIDmsg_lnxind_wlansniffrm		0x0041
#define DIDmsg_lnxind_wlansniffrm_hosttime	0x1041
#define DIDmsg_lnxind_wlansniffrm_mactime	0x2041
#define DIDmsg_lnxind_wlansniffrm_channel	0x3041
#define DIDmsg_lnxind_wlansniffrm_rssi		0x4041
#define DIDmsg_lnxind_wlansniffrm_sq		0x5041
#define DIDmsg_lnxind_wlansniffrm_signal	0x6041
#define DIDmsg_lnxind_wlansniffrm_noise		0x7041
#define DIDmsg_lnxind_wlansniffrm_rate		0x8041
#define DIDmsg_lnxind_wlansniffrm_istx		0x9041
#define DIDmsg_lnxind_wlansniffrm_frmlen	0xA041

struct prism_value {
   u_int32_t did;
   u_int16_t status, len;
   u_int32_t data;
};

struct prism_header {
   u_int32_t msgcode, msglen;
   u_char devname[16];
   struct prism_value hosttime;
   struct prism_value mactime;
   struct prism_value channel;
   struct prism_value rssi;
   struct prism_value sq;
   struct prism_value signal;
   struct prism_value noise;
   struct prism_value rate;
   struct prism_value istx;
   struct prism_value frmlen;
};


/*
 * For DLT_PRISM_HEADER; like DLT_IEEE802_11, but with an extra header,
 * containing information such as radio information, which we
 * currently ignore.
 *
 * If, however, the packet begins with WLANCAP_MAGIC_COOKIE_V1, it's
 * really DLT_IEEE802_11_RADIO (currently, on Linux, there's no
 * ARPHRD_ type for DLT_IEEE802_11_RADIO, as there is a
 * ARPHRD_IEEE80211_PRISM for DLT_PRISM_HEADER, so
 * ARPHRD_IEEE80211_PRISM is used for DLT_IEEE802_11_RADIO, and
 * the first 4 bytes of the header are used to indicate which it is).
 */

void
ArgusPrismPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct prism_header *phdr = (struct prism_header *) p;
   u_int caplen = h->caplen, phdrlen = sizeof (*phdr);

   if (caplen < 4) 
      return;

   if (EXTRACT_32BITS(p) == WLANCAP_MAGIC_COOKIE_V1)
      Argus802_11RadioPacket (user, h, p + PRISM_HDR_LEN);

   if (caplen < PRISM_HDR_LEN)
      return;

/*
   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;
   src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_PRISM;
*/

   if (phdr->hosttime.len != 4) {

#define swapl(n)	(((((unsigned long)(n) & 0xFF)) << 24) | \
			((((unsigned long)(n) & 0xFF00)) << 8) | \
			((((unsigned long)(n) & 0xFF0000)) >> 8) | \
			((((unsigned long)(n) & 0xFF000000)) >> 24))
#define swaps(n)	(((((unsigned short)(n) & 0xFF)) << 8) | (((unsigned short)(n) & 0xFF00) >> 8))

      phdr->msgcode = swapl(phdr->msgcode);
      phdr->msglen  = swapl(phdr->msglen);
      phdr->hosttime.did    = swapl(phdr->hosttime.did);
      phdr->hosttime.status = swaps(phdr->hosttime.status);
      phdr->hosttime.len    = swaps(phdr->hosttime.len);
      phdr->hosttime.data   = swapl(phdr->hosttime.data);
      phdr->mactime.did     = swapl(phdr->mactime.did);
      phdr->mactime.status  = swaps(phdr->mactime.status);
      phdr->mactime.len     = swaps(phdr->mactime.len);
      phdr->mactime.data    = swapl(phdr->mactime.data);
      phdr->channel.did     = swapl(phdr->channel.did);
      phdr->channel.status  = swaps(phdr->channel.status);
      phdr->channel.len     = swaps(phdr->channel.len);
      phdr->channel.data    = swapl(phdr->channel.data);
      phdr->rssi.did        = swapl(phdr->rssi.did);
      phdr->rssi.status     = swaps(phdr->rssi.status);
      phdr->rssi.len        = swaps(phdr->rssi.len);
      phdr->rssi.data       = swapl(phdr->rssi.data);
      phdr->sq.did          = swapl(phdr->sq.did);
      phdr->sq.status       = swaps(phdr->sq.status);
      phdr->sq.len          = swaps(phdr->sq.len);
      phdr->sq.data         = swapl(phdr->sq.data);
      phdr->signal.did      = swapl(phdr->signal.did);
      phdr->signal.status   = swaps(phdr->signal.status);
      phdr->signal.len      = swaps(phdr->signal.len);
      phdr->signal.data     = swapl(phdr->signal.data);
      phdr->noise.did       = swapl(phdr->noise.did);
      phdr->noise.status    = swaps(phdr->noise.status);
      phdr->noise.len       = swaps(phdr->noise.len);
      phdr->noise.data      = swapl(phdr->noise.data);
      phdr->rate.did        = swapl(phdr->rate.did);
      phdr->rate.status     = swaps(phdr->rate.status);
      phdr->rate.len        = swaps(phdr->rate.len);
      phdr->rate.data       = swapl(phdr->rate.data);
      phdr->istx.did        = swapl(phdr->istx.did);
      phdr->istx.status     = swaps(phdr->istx.status);
      phdr->istx.len        = swaps(phdr->istx.len);
      phdr->istx.data       = swapl(phdr->istx.data);
      phdr->frmlen.did      = swapl(phdr->frmlen.did);
      phdr->frmlen.status   = swaps(phdr->frmlen.status);
      phdr->frmlen.len      = swaps(phdr->frmlen.len);
      phdr->frmlen.data     = swapl(phdr->frmlen.data);
   }

   if ((phdrlen == PRISM_HDR_LEN) && ((void *)(phdr + 1) == (p + PRISM_HDR_LEN)))
      Argus802_11Packet (user, h, p + PRISM_HDR_LEN);

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPrismPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


struct avs_header {
   u_int32_t version, length;
   u_int64_t mactime, hosttime;
   u_int32_t phytype, channel;
   u_int32_t datarate, antenna;
   u_int32_t priority, ssi_type;
   u_int32_t ssi_signal, ssi_noise;
   u_int32_t preamble, encoding;
};


void
Argus802_11RadioAvsPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct avs_header *ahdr = (struct avs_header *) p;
   u_int32_t caplen = h->caplen, caphdr_len;

   caphdr_len = EXTRACT_32BITS(p + 4);
   if (caphdr_len < 8)
      return;
   
   if (caplen < caphdr_len)
      return;
   
   Argus802_11Packet (user, h, (u_char *)(ahdr + 1));

#ifdef ARGUSDEBUG 
   ArgusDebug (8, "Argus802_11RadioAvsPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


void
Argus802_11RadioPacket (u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{

#ifdef ARGUSDEBUG
   ArgusDebug (8, "Argus802_11RadioPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


void
ArgusIpPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   struct ip *ip = (struct ip *) p;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   struct stat statbuf;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *) ip) + caplen;
   src->ArgusModel->ArgusThisEncaps  = 0;

   if (p) {
      src->ArgusModel->ArgusThisLength  = length;
      ArgusProcessIpPacket (src->ArgusModel, ip, length, tvp);
   }


#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusIpPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}



#if defined(ultrix) || defined(__alpha)
static int   fddi_bitswap = 0;
#else
static int   fddi_bitswap = 1;
#endif

int fddipad = FDDIPAD;

#define FDDI_HDRLEN (sizeof(struct fddi_header))

static u_char fddi_bit_swap[] = {
   0x00, 0x80, 0x40, 0xc0, 0x20, 0xa0, 0x60, 0xe0,
   0x10, 0x90, 0x50, 0xd0, 0x30, 0xb0, 0x70, 0xf0,
   0x08, 0x88, 0x48, 0xc8, 0x28, 0xa8, 0x68, 0xe8,
   0x18, 0x98, 0x58, 0xd8, 0x38, 0xb8, 0x78, 0xf8,
   0x04, 0x84, 0x44, 0xc4, 0x24, 0xa4, 0x64, 0xe4,
   0x14, 0x94, 0x54, 0xd4, 0x34, 0xb4, 0x74, 0xf4,
   0x0c, 0x8c, 0x4c, 0xcc, 0x2c, 0xac, 0x6c, 0xec,
   0x1c, 0x9c, 0x5c, 0xdc, 0x3c, 0xbc, 0x7c, 0xfc,
   0x02, 0x82, 0x42, 0xc2, 0x22, 0xa2, 0x62, 0xe2,
   0x12, 0x92, 0x52, 0xd2, 0x32, 0xb2, 0x72, 0xf2,
   0x0a, 0x8a, 0x4a, 0xca, 0x2a, 0xaa, 0x6a, 0xea,
   0x1a, 0x9a, 0x5a, 0xda, 0x3a, 0xba, 0x7a, 0xfa,
   0x06, 0x86, 0x46, 0xc6, 0x26, 0xa6, 0x66, 0xe6,
   0x16, 0x96, 0x56, 0xd6, 0x36, 0xb6, 0x76, 0xf6,
   0x0e, 0x8e, 0x4e, 0xce, 0x2e, 0xae, 0x6e, 0xee,
   0x1e, 0x9e, 0x5e, 0xde, 0x3e, 0xbe, 0x7e, 0xfe,
   0x01, 0x81, 0x41, 0xc1, 0x21, 0xa1, 0x61, 0xe1,
   0x11, 0x91, 0x51, 0xd1, 0x31, 0xb1, 0x71, 0xf1,
   0x09, 0x89, 0x49, 0xc9, 0x29, 0xa9, 0x69, 0xe9,
   0x19, 0x99, 0x59, 0xd9, 0x39, 0xb9, 0x79, 0xf9,
   0x05, 0x85, 0x45, 0xc5, 0x25, 0xa5, 0x65, 0xe5,
   0x15, 0x95, 0x55, 0xd5, 0x35, 0xb5, 0x75, 0xf5,
   0x0d, 0x8d, 0x4d, 0xcd, 0x2d, 0xad, 0x6d, 0xed,
   0x1d, 0x9d, 0x5d, 0xdd, 0x3d, 0xbd, 0x7d, 0xfd,
   0x03, 0x83, 0x43, 0xc3, 0x23, 0xa3, 0x63, 0xe3,
   0x13, 0x93, 0x53, 0xd3, 0x33, 0xb3, 0x73, 0xf3,
   0x0b, 0x8b, 0x4b, 0xcb, 0x2b, 0xab, 0x6b, 0xeb,
   0x1b, 0x9b, 0x5b, 0xdb, 0x3b, 0xbb, 0x7b, 0xfb,
   0x07, 0x87, 0x47, 0xc7, 0x27, 0xa7, 0x67, 0xe7,
   0x17, 0x97, 0x57, 0xd7, 0x37, 0xb7, 0x77, 0xf7,
   0x0f, 0x8f, 0x4f, 0xcf, 0x2f, 0xaf, 0x6f, 0xef,
   0x1f, 0x9f, 0x5f, 0xdf, 0x3f, 0xbf, 0x7f, 0xff,
};

static inline void
Argusextract_fddi_addrs(const struct fddi_header *fp, struct ether_header *ehdr)
{
   char *fsrc = (char *)&ehdr->ether_shost;
   char *fdst = (char *)&ehdr->ether_dhost;
   int i;

   if (fddi_bitswap) {
      for (i = 0; i < 6; ++i)
         fdst[i] = fddi_bit_swap[fp->fddi_dhost[i]];
      for (i = 0; i < 6; ++i)
         fsrc[i] = fddi_bit_swap[fp->fddi_shost[i]];
   }
   else {
      bcopy ((char *) fp->fddi_dhost, fdst, 6);
      bcopy ((char *) fp->fddi_shost, fsrc, 6);
   }
}

int
ArgusCreatePktFromFddi(const struct fddi_header *fp, struct ether_header *ep, int length)
{
   unsigned char *ptr;
   unsigned int retn = 0;
   struct llc *llc;
 
   if ((fp->fddi_fc & FDDIFC_CLFF) == FDDIFC_LLC_ASYNC) {
      Argusextract_fddi_addrs (fp, ep);

      llc = (struct llc *)(fp + 1);
 
      if (llc->ssap == LLCSAP_SNAP && llc->dsap == LLCSAP_SNAP && llc->llcui == LLC_UI) {
         ((struct ether_header *) ep)->ether_type = EXTRACT_16BITS(&llc->ethertype[0]);
         ptr = (unsigned char *)(llc + 1);
         length -= (sizeof(struct fddi_header) + sizeof(struct llc));
         bcopy ((char *)ptr, (char *)(ep + 1), length);
         retn = length + sizeof(struct ether_header);
      }
   }

   return (retn);
}

void
ArgusFddiPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   const struct fddi_header *fp = (struct fddi_header *)p;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   int ind = src->ArgusThisIndex;
   struct ether_header *ep;
   struct stat statbuf;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   ep = (struct ether_header *) src->ArgusInterface[ind].ArgusPacketBuffer;

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;
   src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_FDDI;

   src->ArgusInterface[ind].ArgusPacketBuffer = src->ArgusInterface[ind].ArgusPacketBufferBuffer;
   if (p && (length = ArgusCreatePktFromFddi(fp, ep, length))) {
      if (p && length)
         ArgusProcessPacket (src, (char *)ep, length, tvp, ARGUS_ETHER_HDR);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusFddiPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


#define ARGUS_802_6_MAC_HDR_LEN      20
#define ARGUS_ATM_HDR_OFFSET      8

void
ArgusATMPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   unsigned int caplen = h->caplen;
   unsigned int length = h->len;
   int ind = src->ArgusThisIndex;
   struct ether_header *ep;
   struct stat statbuf;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   ep = (struct ether_header *) src->ArgusInterface[ind].ArgusPacketBuffer;

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusInterface[ind].ArgusPacketBuffer = src->ArgusInterface[ind].ArgusPacketBufferBuffer;

   if (caplen > 8) {
      if (p[0] != 0xaa || p[1] != 0xaa || p[2] != 0x03) {
         if (caplen > 28) {
            p += ARGUS_802_6_MAC_HDR_LEN;
            length -= ARGUS_802_6_MAC_HDR_LEN;
            caplen -= ARGUS_802_6_MAC_HDR_LEN;
         } else
            return;
      }
   } else
      return;
   
   ep->ether_type = ((p[6] << 8) | p[7]);
   length -= ARGUS_ATM_HDR_OFFSET;
   caplen -= ARGUS_ATM_HDR_OFFSET;
   p += ARGUS_ATM_HDR_OFFSET;
   
   bcopy (p, (char *)(ep + 1), caplen);

   length += sizeof(*ep);
   caplen += sizeof(*ep);

   src->ArgusModel->ArgusThisLength  = length;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ep) + caplen;
   src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_ATM;

   ArgusProcessPacket (src, (char *)ep, length, tvp, ARGUS_ETHER_HDR);

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusATMPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


void
ArgusPppPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   struct ip *ip = (struct ip *) (p + PPP_HDRLEN);
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   struct stat statbuf;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   if (p && (length > PPP_HDRLEN)) {
      src->ArgusModel->ArgusGlobalTime  = *tvp;
      src->ArgusModel->ArgusSnapLength  = caplen;
      src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ip) + (caplen - PPP_HDRLEN);
      src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_PPP;

      length -= PPP_HDRLEN;

      src->ArgusModel->ArgusThisLength  = length;

      ArgusProcessIpPacket (src->ArgusModel, ip, length, tvp);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}



#define ARGUS_PPPBSDOS_HDR_LEN       24


void
ArgusPppBsdosPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   unsigned int length = h->len, hdrlen = 0;
   unsigned int caplen = h->caplen;
   unsigned short ptype = 0;
   struct stat statbuf;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = (u_char *) p + caplen;
   src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_PPP;

   if (p[0] == PPP_ADDRESS && p[1] == PPP_CONTROL) {
      p += 2;
      hdrlen = 2;
   }

   if (*p & 01) {                  /* Retrieve the protocol type */
      ptype = *p;                  /* Compressed protocol field */
      p++;
      hdrlen += 1;
   } else {
#if defined(_LITTLE_ENDIAN)
      ptype = ntohs(*(u_short *)p);
#else
      ptype = *(u_short *)p;
#endif
      p += 2;
      hdrlen += 2;
   }

   length -= hdrlen;
   if (ptype == PPP_IP)
      if (p && (length > 0)) {
         src->ArgusModel->ArgusThisLength  = length;
         ArgusProcessIpPacket (src->ArgusModel, (struct ip *) p, length, tvp);
      }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusPppBsdosPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

#if defined(__NetBSD__) || defined(__FreeBSD__)
#include <sys/param.h>
#include <sys/mbuf.h>
#endif

#include <net/slcompress.h>
#include <net/slip.h>


/* XXX BSD/OS 2.1 compatibility */

#if !defined(ARGUS_SLIP_HDR_LEN) && defined(SLC_BPFHDR)
#define SLIP_HDRLEN SLC_BPFHDR
#define SLX_DIR 0
#define SLX_CHDR (SLC_BPFHDRLEN - 1)
#define CHDR_LEN (SLC_BPFHDR - SLC_BPFHDRLEN)
#else

#endif


void
ArgusSlipPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct timeval tvpbuf, *tvp = &tvpbuf;
   struct ip *ip = (struct ip *) (p + SLIP_HDRLEN);
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   struct stat statbuf;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   src->ArgusModel->ArgusGlobalTime  = *tvp;
   src->ArgusModel->ArgusSnapLength  = caplen;
   src->ArgusModel->ArgusThisSnapEnd = ((u_char *)ip) + (caplen - SLIP_HDRLEN);
   src->ArgusModel->ArgusThisEncaps  = ARGUS_ENCAPS_SLIP;

   if (p && (length > SLIP_HDRLEN)) {
      length -= SLIP_HDRLEN;

      src->ArgusModel->ArgusThisLength  = length;

      ArgusProcessIpPacket (src->ArgusModel, ip, length, tvp);
   }

 
#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusSlipPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}


#include <sll.h>

#if !defined(ETHER_ADDR_LEN)
#define ETHER_ADDR_LEN  6
#endif


void
ArgusSllPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct ArgusModelerStruct *model = src->ArgusModel;
   unsigned int length = h->len;
   unsigned int caplen = h->caplen;
   const struct sll_header *sllp = NULL;
   struct timeval tvpbuf, *tvp = &tvpbuf;

   unsigned char buf[2048];
   struct ether_header *ep = (struct ether_header *)buf;
   struct stat statbuf;
   u_short pkttype;

   tvp->tv_sec  = h->ts.tv_sec;
   tvp->tv_usec = h->ts.tv_usec;


   model->ArgusGlobalTime  = *tvp;

   if (src->ArgusWriteOutPacketFile) {
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   sllp = (const struct sll_header *)p;
   memcpy((void *)&ep->ether_shost, sllp->sll_addr, ETHER_ADDR_LEN);

#if defined(_LITTLE_ENDIAN)
   pkttype = ntohs(sllp->sll_pkttype);
#else
   pkttype = sllp->sll_pkttype;
#endif

   if (pkttype != LINUX_SLL_OUTGOING) {
      if (pkttype == LINUX_SLL_BROADCAST)
         memset((void *)&ep->ether_dhost, 0xFF, ETHER_ADDR_LEN);
      else {
         memset((void *)&ep->ether_dhost, 0, ETHER_ADDR_LEN);
         if (pkttype == LINUX_SLL_MULTICAST)
#if defined(HAVE_SOLARIS)
            ep->ether_dhost.ether_addr_octet[0] = 0x01;
#else
            ep->ether_dhost[0] = 0x01;
#endif
         else
#if defined(HAVE_SOLARIS)
            ep->ether_dhost.ether_addr_octet[ETHER_ADDR_LEN-1] = 0x01;
#else
            ep->ether_dhost[ETHER_ADDR_LEN-1] = 0x01;
#endif
      }
   } else {
      /*
       * We sent this packet; we don't know whether it's
       * broadcast, multicast, or unicast, so just make
       * the destination address all 0's.
       */
      memset((void *)&ep->ether_dhost, 0, ETHER_ADDR_LEN);
   }

   length -= SLL_HDR_LEN;
   caplen -= SLL_HDR_LEN;
   p += SLL_HDR_LEN;
 
   ep->ether_type = sllp->sll_protocol;
 
   memcpy((ep + 1), p, caplen);

   model->ArgusThisSnapEnd = (unsigned char *)(ep + caplen);
   model->ArgusThisLength  = length;
   model->ArgusSnapLength  = caplen;
   model->ArgusThisEncaps  = ARGUS_ENCAPS_SLL;

   ArgusProcessPacket (src, (char *)ep, length, tvp, ARGUS_ETHER_HDR);

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusSllPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

/*
 * Byte-swap a 32-bit number.
 * ("htonl()" or "ntohl()" won't work - we want to byte-swap even on
 * big-endian platforms.)
 */
#define SWAPLONG(y) \
((((y)&0xff)<<24) | (((y)&0xff00)<<8) | (((y)&0xff0000)>>8) | (((y)>>24)&0xff))

#define NULL_HDRLEN 4

#define BSD_AF_INET             2
#define BSD_AF_NS               6               /* XEROX NS protocols */
#define BSD_AF_ISO              7
#define BSD_AF_APPLETALK        16
#define BSD_AF_IPX              23
#define BSD_AF_INET6_BSD        24      /* OpenBSD (and probably NetBSD), BSD/OS */
#define BSD_AF_INET6_FREEBSD    28
#define BSD_AF_INET6_DARWIN     30

void
ArgusNullPacket(u_char *user, const struct pcap_pkthdr *h, const u_char *p)
{
   struct ArgusSourceStruct *src = (struct ArgusSourceStruct *) user;
   struct pcap_pkthdr hbuf;
   unsigned int family;

   if (src->ArgusWriteOutPacketFile) {
      struct stat statbuf;
      if (stat(src->ArgusWriteOutPacketFile, &statbuf) < 0) {
         if (src->ArgusPcapOutFile != NULL) {
            pcap_dump_close(src->ArgusPcapOutFile);
            src->ArgusPcapOutFile = NULL;
         }

         if ((src->ArgusPcapOutFile = pcap_dump_open(src->ArgusInterface[0].ArgusPd, src->ArgusWriteOutPacketFile)) == NULL)
            ArgusLog (LOG_ERR, "%s\n", pcap_geterr (src->ArgusInterface[0].ArgusPd));
      }

      if (src->ArgusPcapOutFile != NULL)
         pcap_dump((u_char *)src->ArgusPcapOutFile, h, p);
   }

   memcpy((char *)&family, (char *)p, sizeof(family));
   memcpy((char *)&hbuf, (char *)h, sizeof(*h));

   if ((family & 0xFFFF0000) != 0)
      family = SWAPLONG(family);

   hbuf.len    -= NULL_HDRLEN;
   hbuf.caplen -= NULL_HDRLEN;
   p           += NULL_HDRLEN;
 
   switch (family) {
      case BSD_AF_INET:
      case BSD_AF_INET6_BSD:     
      case BSD_AF_INET6_FREEBSD: 
      case BSD_AF_INET6_DARWIN:
         ArgusIpPacket (user, &hbuf, p);
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (8, "ArgusNullPacket (0x%x, 0x%x, 0x%x) returning\n", user, h, p);
#endif
}

#include <sys/ioctl.h>

#if defined(HAVE_SOLARIS)
#include <sys/sockio.h>
#endif



void
ArgusGetPackets (struct ArgusSourceStruct *src)
{
   fd_set ArgusReadMask, ArgusWriteMask, ArgusExceptMask;
   int tmp, i, width = 0, noerror = 1, fd;
   int found = 0, up = 0, notselectable = 0;
   int fds[ARGUS_MAXINTERFACE];
   struct timeval wait;

   if (src != NULL) {
#ifdef ARGUSDEBUG
      ArgusDebug (4, "ArgusGetPackets (0x%x) starting\n", src);
#endif

      for (i = 0; i < ARGUS_MAXINTERFACE; i++)
         fds[i] = -1;

#if defined(HAVE_SOLARIS)
      sigignore (SIGPIPE);
#else
      signal (SIGPIPE, SIG_IGN);
#endif
      FD_ZERO(&ArgusReadMask);
      FD_ZERO(&ArgusWriteMask);
      FD_ZERO(&ArgusExceptMask);

      wait.tv_sec = 0; wait.tv_usec = 200000;

      ArgusGetInterfaceStatus(src);
      gettimeofday (&src->ArgusStartTime, 0L);

      for (i = 0; i < src->ArgusInterfaces; i++) {
         if (src->ArgusInterface[i].ArgusPd && (fd = pcap_fileno(src->ArgusInterface[i].ArgusPd) >= 0)) {
            found++;
#if !defined(CYGWIN) && !defined(WIN32) && !defined(MSDOS)
            if (pcap_get_selectable_fd(src->ArgusInterface[i].ArgusPd) < 0)
               notselectable++;
#endif
            if ((src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) != 0) {
               up++;
               fds[i] = fd;
               setArgusInterfaceStatus(src, 1);
               FD_SET(pcap_fileno(src->ArgusInterface[i].ArgusPd), &ArgusReadMask);
               if (width < fd)
                  width = fd;
            } else
               setArgusInterfaceStatus(src, 0);
         } else
            setArgusInterfaceStatus(src, 0);
      }

      if (!(src->ArgusReadingOffLine)) {
         do {
#if defined(CYGWIN)
            struct pcap_pkthdr *header;
            const u_char *pkt_data;

            if ((tmp = pcap_next_ex(src->ArgusInterface[0].ArgusPd, &header, &pkt_data)) >= 0) {
               if ( tmp > 0) {
                  src->ArgusThisIndex = 0;
                  src->ArgusInterface[0].ArgusCallBack((char *)src, header, pkt_data);
               } else
                  gettimeofday (&src->ArgusModel->ArgusGlobalTime, NULL);

            } else {
               gettimeofday (&src->ArgusModel->ArgusGlobalTime, NULL);
               noerror = 0;
            }
#else
            if (notselectable) {
               struct pcap_pkthdr *header;
               const u_char *pkt_data;
               int x, pkts = 0;

               for (i = 0; i < src->ArgusInterfaces; i++) {
                  int cnt = 0;
                  if ((fd = fds[i]) != -1) {
                     for (x = 0; x < 5; x++) {
                        if ((cnt = pcap_next_ex(src->ArgusInterface[i].ArgusPd, &header, &pkt_data)) >= 0) {
                           if (cnt > 0) {
                              pkts++;
                              src->ArgusThisIndex = i;
                              src->ArgusInterface[i].ArgusCallBack((u_char *)src, header, pkt_data);
                           } else
                              break;
                        } else
                           noerror = 0;
                     }
                     found++;
                  }
               }

               if (pkts == 0) {
                  struct timespec tsbuf = {0, 25000000}, *ts = &tsbuf;
                  gettimeofday (&src->ArgusModel->ArgusGlobalTime, NULL);
                  nanosleep(ts, NULL);
               }

            } else {
               if (up && ((tmp = select (width + 1, &ArgusReadMask, NULL, NULL, &wait)) >= 0)) {
                  found = 0;
#ifdef ARGUSDEBUG
                  ArgusDebug (10, "ArgusGetPackets: select() returned %d\n", tmp);
#endif
                  if (tmp > 0) {
                     for (i = 0; i < src->ArgusInterfaces; i++) {
                        if ((fd = fds[i]) != -1) {
                           if (FD_ISSET(fd, &ArgusReadMask)) {
                              found++;
                              src->ArgusThisIndex = i;
                              switch (src->ArgusInterface[i].ArgusInterfaceType) {
                                 case ARGUSLIBPPKTFILE:
                                    if ((pcap_dispatch (src->ArgusInterface[i].ArgusPd, 1, src->ArgusInterface[i].ArgusCallBack, (u_char *)src)) < 0) {
                                       if (!(strncmp (pcap_geterr(src->ArgusInterface[i].ArgusPd), "recvfrom", 8))) {
#ifdef ARGUSDEBUG
                                          ArgusDebug (3, "ArgusGetPackets: pcap_dispatch() returned %s\n", pcap_geterr(src->ArgusInterface[i].ArgusPd));
#endif

                                       } else
                                          noerror = 0;
                                    } else {
#ifdef ARGUSDEBUG
                                       ArgusDebug (9, "ArgusGetPackets: pcap_dispatch() interface %s %d up\n", src->ArgusInterface[i].ArgusDevice, up);
#endif
                                    }
                                    break;
       
                                 case ARGUSSNOOPKTFILE:
                                    if (ArgusSnoopRead (src) < 0)
                                       noerror = 0;
                                    break;

                                 case ARGUSMOATTSHPKTFILE:
                                    if (ArgusMoatTshRead (src) < 0)
                                       noerror = 0;
                                    break;

                              }
                              if (src->eNflag > 0)
                                 src->eNflag--;
                           }
                        }
                     }

                  } else {
/* libpcap workaround */
                     struct pcap_pkthdr *header;
                     const u_char *pkt_data;
                     int pkts = 0, cnt, retn;

                     do {
                        cnt = 0;
                        for (i = 0; i < src->ArgusInterfaces; i++) {
                           if ((fd = fds[i]) != -1) {
                              if ((retn = pcap_next_ex(src->ArgusInterface[i].ArgusPd, &header, &pkt_data)) > 0) {
                                 pkts++;
                                 cnt += retn;
                                 src->ArgusThisIndex = i;
                                 src->ArgusInterface[i].ArgusCallBack((u_char *)src, header, pkt_data);
                              } else
                                 if (retn < 0)
                                    noerror = 0;
                           }
                        }
                     } while (cnt > 0);

                     if (!pkts) {
                        gettimeofday (&src->ArgusModel->ArgusGlobalTime, NULL);
#ifdef ARGUSDEBUG
                        ArgusDebug (9, "ArgusGetPackets: select() timeout %d up interfaces\n", up);
#endif
                     }
                  }

               } else {
                  gettimeofday (&src->ArgusModel->ArgusGlobalTime, NULL);
                  if (up) {
#ifdef ARGUSDEBUG
                     ArgusDebug (3, "ArgusGetPackets: select() returned %s\n", strerror(errno));
#endif
                  } else {
                     struct timespec tsbuf = {0, 150000000}, *ts = &tsbuf;
#ifdef ARGUSDEBUG
                     ArgusDebug (5, "ArgusGetPackets: no interfaces up: sleeping\n");
#endif
                     nanosleep(ts, NULL);
                  }
               }
               width = 0;
               found = 0;
               up = 0;
               FD_ZERO(&ArgusReadMask);

               for (i = 0; i < src->ArgusInterfaces; i++) {
                  if (src->ArgusInterface[i].ArgusPd && ((fd = pcap_fileno(src->ArgusInterface[i].ArgusPd)) >= 0)) {
                     found++;
                     fds[i] = fd;

                     if (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) {
                        up++;
                        FD_SET(pcap_fileno(src->ArgusInterface[i].ArgusPd), &ArgusReadMask);
                        if (width < pcap_fileno(src->ArgusInterface[i].ArgusPd))
                           width = pcap_fileno(src->ArgusInterface[i].ArgusPd);
                     }

                  } else {
                     fds[i] = -1;
                     ArgusOpenInterface(src, &src->ArgusInterface[i]);
                     found++;
                  }
               }

               wait.tv_sec = 0; wait.tv_usec = 200000;
            }
 #endif  
            if (!found)
               break;

            if (ArgusUpdateTime (src->ArgusModel)) {
               ArgusGetInterfaceStatus(src);
               ArgusQueueManager(ArgusModel);
#if !defined(ARGUS_THREADS)
               ArgusOutputProcess(ArgusOutputTask);
#endif
            }

         } while (noerror && (src->eNflag != 0) && (!(ArgusShutDownStarted)));
      
      } else {
         for (i = 0; i < src->ArgusInterfaces; i++) {
            src->ArgusThisIndex = i;
            pcap_offline_read (src->ArgusInterface[i].ArgusPd, src->eNflag,
                       src->ArgusInterface[i].ArgusCallBack, (u_char *) src);
         }
      }

      setArgusFarReportInterval (ArgusModel, "0");
      ArgusQueueManager(ArgusModel);
#if !defined(ARGUS_THREADS)
      ArgusOutputProcess(ArgusOutputTask);
#endif
   }

   gettimeofday (&src->ArgusEndTime, 0L);

#ifdef ARGUSDEBUG
   ArgusDebug (4, "ArgusGetPackets () returning\n");
#endif
}


void
Argusbpf_dump(struct bpf_program *p, int option)
{
   struct bpf_insn *insn;
   int i, n = p->bf_len;

   insn = p->bf_insns;
   if (option > 2) {
      fprintf(stdout, "%d\n", n);
      for (i = 0; i < n; ++insn, ++i) {
         fprintf(stdout, "%lu %lu %lu %lu\n", (long) insn->code,
                (long) insn->jt, (long) insn->jf, (long) insn->k);
      }
      return;
   }
   if (option > 1) {
      for (i = 0; i < n; ++insn, ++i) {
         fprintf(stdout, "{ 0x%x, %d, %d, 0x%08x },\n",
                insn->code, insn->jt, insn->jf, (int) insn->k);
      }
      return;
   }

   for (i = 0; i < n; ++insn, ++i)
      fprintf (stdout, "%s\n", bpf_image(insn, i));
}




#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
 
#define ARGUSSNOOPTAG  "snoop"
 
int
ArgusOpenInputPacketFile(struct ArgusSourceStruct *src, char *errbuf)
{
   int retn = 0, i = src->ArgusInterfaces;
   struct ArgusRfileStruct *rfile;
   int ch, rlen, type;
   char readbuf[256];

   if (src->ArgusRfileList != NULL) {
      if ((src->ArgusRfileList->count + i) > ARGUS_MAXINTERFACE)
         ArgusLog(LOG_ERR, "ArgusOpenInputPacketFile: too many inputs max is %d\n", ARGUS_MAXINTERFACE);

      rfile = (struct ArgusRfileStruct *) src->ArgusRfileList->start;

      while (rfile) {
         if ((src->ArgusInterface[i].ArgusPd = pcap_open_offline(rfile->name, errbuf)) != NULL) {
            src->ArgusInputPacketFileType = ARGUSLIBPPKTFILE;
            src->ArgusInterface[i].ArgusInterfaceType = ARGUSLIBPPKTFILE;
            src->ArgusInterface[i].ArgusDevice = (struct ArgusDeviceStruct *) rfile;
            type = pcap_datalink(src->ArgusInterface[i].ArgusPd);

            if ((src->ArgusInterface[i].ArgusCallBack = Arguslookup_pcap_callback(type)) == NULL)
               ArgusLog(LOG_ERR, "ArgusOpenInputPacketFile(%s) unsupported device type %d\n", rfile->name, type);

            src->ArgusInterface[i].ArgusLocalNet = 0;
            src->ArgusInterface[i].ArgusNetMask = 0;
            src->ArgusReadingOffLine++;
            retn++;

         } else {
#ifdef ARGUSDEBUG
            ArgusDebug (3, "ArgusOpenInputPacketFile(0x%x) pcap_open_offline %s\n", src, errbuf);
#endif
            if (strcmp(rfile->name, "-")) {
               if ((src->ArgusPacketInput = fopen(rfile->name, "r")) == NULL) {
                  snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "%s: %s\n", rfile->name, strerror(errno));
               }
            } else
               src->ArgusPacketInput = stdin;
    
            if (src->ArgusPacketInput) {
               if (getArgusMoatTshFile(src)) {
                  if (src->ArgusPacketInput == stdin) {
                     src->ArgusInterface[i].ArgusPcap.fd = 0;
                     src->ArgusInterface[i].ArgusPcap.snapshot = 1500;
                     src->ArgusInterface[i].ArgusPcap.linktype = DLT_EN10MB;
                     src->ArgusInterface[i].ArgusInterfaceType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInputPacketFileType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInterface[i].ArgusDevice = (struct ArgusDeviceStruct *) rfile;
                     src->ArgusReadingOffLine++;
                     retn++;
                  } else
                  if ((src->ArgusInterface[i].ArgusPcap.fd = open (rfile->name, O_RDONLY, NULL)) >= 0) {
                     src->ArgusInterface[i].ArgusPcap.snapshot = 1500;
                     src->ArgusInterface[i].ArgusPcap.linktype = DLT_EN10MB;
                     src->ArgusInterface[i].ArgusInterfaceType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInputPacketFileType = ARGUSMOATTSHPKTFILE;
                     src->ArgusInterface[i].ArgusDevice = (struct ArgusDeviceStruct *) rfile;
                     src->ArgusReadingOffLine++;
                     retn++;
   
                  } else
                     ArgusLog(LOG_ERR, "ArgusOpenInputPacketFile(%s) error. %s\n", rfile->name, strerror(errno));
               } else
               if ((ch = fgetc(src->ArgusPacketInput)) != EOF) {
                  ungetc(ch, src->ArgusPacketInput);
                  if ((rlen = fread ((char *)readbuf, 1, sizeof(ARGUSSNOOPTAG),
                                           src->ArgusPacketInput)) == sizeof(ARGUSSNOOPTAG)) {
                     if ((strncmp((char *)readbuf, ARGUSSNOOPTAG, sizeof(ARGUSSNOOPTAG)) == 0)) {
                        fclose(src->ArgusPacketInput);
                        if ((src->ArgusInterface[i].ArgusPcap.fd = open (rfile->name, O_RDONLY, NULL)) >= 0) {
                           lseek(src->ArgusInterface[i].ArgusPcap.fd, 16, SEEK_SET);
                           src->ArgusInterface[i].ArgusPcap.snapshot = 1500;
                           src->ArgusInterface[i].ArgusPcap.linktype = DLT_EN10MB;
                           src->ArgusInputPacketFileType = ARGUSSNOOPKTFILE;
                           src->ArgusReadingOffLine++;
                           retn++;
                        }
         
                     } else {
                        snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "%s, unknown packet file format", rfile->name);
                     }
                  } else {
                     snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "Error reading %s. Read %d bytes", rfile->name, rlen);
                  }
               } else {
                  snprintf (errbuf, PCAP_ERRBUF_SIZE - 1, "Error reading %s. stream empty", rfile->name);
               }

               src->ArgusInterface[i].ArgusLocalNet = 0;
               src->ArgusInterface[i].ArgusNetMask = 0;
            }
         }

         rfile = (struct ArgusRfileStruct *) rfile->nxt;
         i++;
      }
      src->ArgusInterfaces = i;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusOpenInputPacketFile(0x%x) returning %d\n", errbuf, retn);
#endif

   return (retn);
}


char *
ArgusCopyArgv (char **argv)
{
   char **p;
   int len = 0;
   char *buf = NULL, *src, *dst;
 
   p = argv;
   if (*p == 0) return 0;
 
   while (*p) len += (int) strlen (*p++) + 1;
 
   if ((buf = (char *) ArgusMalloc (len)) != NULL) {
      p = argv;
      dst = buf;
      while ((src = *p++) != NULL) {
         if (*src != '-') {
            while ((*dst++ = *src++) != '\0') ;
            dst[-1] = ' ';
         }
      }
 
      dst[-1] = '\0';
   }

#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusCopyArgv(0x%x) returning 0x%x\n", argv, buf);
#endif
 
   return buf;
}


#include <sys/stat.h>
#include <fcntl.h>
#include <net/if.h>
#include <sys/ioctl.h>

int ArgusGetInterfaceFD = -1;

void
ArgusGetInterfaceStatus (struct ArgusSourceStruct *src)
{
   struct ArgusDeviceStruct *device = NULL;
   char errbuf[PCAP_ERRBUF_SIZE];
   extern int ArgusShutDownFlag;
   struct ifreq ifr;
   int fd, i;

   if (ArgusShutDownFlag) {
      ArgusShutDown(0);
      return;
   }

   if (src && src->ArgusDeviceList)
      if ((device = (struct ArgusDeviceStruct *) ArgusPopFrontList(src->ArgusDeviceList, ARGUS_LOCK)) != NULL)
         ArgusPushFrontList(src->ArgusDeviceList, (struct ArgusListRecord *) device, ARGUS_LOCK);

   if (device == NULL)
      return;

   if (strstr(device->name, "dag")) {
      for (i = 0; i < src->ArgusInterfaces; i++) {
         if (src->ArgusInterface[i].ArgusPd && (pcap_fileno(src->ArgusInterface[i].ArgusPd) > 0))
            bzero ((char *)&src->ArgusInterface[i].ifr, sizeof(ifr));

         src->ArgusInterface[i].ifr.ifr_flags |= IFF_UP;
         setArgusInterfaceStatus(src, 1);
      }
      return;
   }

   if (ArgusGetInterfaceFD < 0)
      if ((ArgusGetInterfaceFD = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
         ArgusLog(LOG_ERR, "ArgusGetInterfaceStatus: socket %s", strerror(errno));

   fd = ArgusGetInterfaceFD;

   for (i = 0; i < src->ArgusInterfaces; i++) {
      if (src->ArgusInterface[i].ArgusPd && (pcap_fileno(src->ArgusInterface[i].ArgusPd) > 0)) {
         memcpy ((char *)&ifr, (char *)&src->ArgusInterface[i].ifr, sizeof(ifr));

         if ((ioctl(fd, SIOCGIFFLAGS, (char *)&src->ArgusInterface[i].ifr)) < 0) {
            if (src->ArgusInterface[i].ArgusPd) {
               pcap_close(src->ArgusInterface[i].ArgusPd);
               src->ArgusInterface[i].ArgusPd = NULL;
               return;
            }
         }

         if ((ifr.ifr_flags & IFF_UP) != (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP)) {
            setArgusInterfaceStatus(src, (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) ? 1 : 0);
 
            if (!((pcap_lookupnet (src->ArgusInterface[i].ArgusDevice->name, 
                         (u_int *)&src->ArgusInterface[i].ArgusLocalNet,
                         (u_int *)&src->ArgusInterface[i].ArgusNetMask, errbuf)) < 0)) {
#if defined(_LITTLE_ENDIAN)
               src->ArgusInterface[i].ArgusLocalNet = ntohl(src->ArgusInterface[i].ArgusLocalNet);
               src->ArgusInterface[i].ArgusNetMask  = ntohl(src->ArgusInterface[i].ArgusNetMask);
#endif
            }
            ArgusLog (LOG_WARNING, "ArgusGetInterfaceStatus: interface %s is %s\n", src->ArgusInterface[i].ifr.ifr_name,
               (src->ArgusInterface[i].ifr.ifr_flags & IFF_UP) ? "up" : "down");
         }

         pcap_stats (src->ArgusInterface[i].ArgusPd, &src->ArgusInterface[i].ArgusStat);
      } else {
      }
   }

   return;
}

int RaSortFileList (const void *, const void *);

int
RaSortFileList (const void *item1, const void *item2)
{
   struct ArgusRfileStruct *file1 = *(struct ArgusRfileStruct **) item1;
   struct ArgusRfileStruct *file2 = *(struct ArgusRfileStruct **) item2;

   return (strcmp (file1->name, file2->name));
}



void
ArgusSortFileList (struct ArgusListStruct *list)
{
   struct ArgusRfileStruct *rfile;
   void **array = NULL;
   int i = 0, count = 0;

   if ((list != NULL) && (count = list->count)) {
      if ((array = ArgusCalloc (count, sizeof(rfile))) == NULL)
         ArgusLog (LOG_ERR, "ArgusSortFileList: ArgusCalloc %s", strerror(errno));

      while ((rfile = (void *)ArgusPopFrontList(list, ARGUS_LOCK)) != NULL)
         array[i++] = rfile;

      if (i != count)
         ArgusLog (LOG_ERR, "ArgusSortFileList: integrity failure");

      qsort (array, i, sizeof(rfile), RaSortFileList);

      for (i = 0; i < count; i++)
         ArgusPushFrontList(list, (struct ArgusListRecord *) array[i], ARGUS_LOCK);

      ArgusFree (array);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (3, "ArgusSortFileList(0x%x)\n", list);
#endif
}

