/* $Id: ArkConfig.cpp,v 1.38 2003/03/23 20:09:25 mrq Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <iostream>
#include <sstream>
#include <vector>
#include <map>

#include <Ark/Ark.h>
#include <Ark/ArkString.h>
#include <Ark/ArkConfig.h>
#include <Ark/ArkLexer.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkFileSys.h>


namespace Ark
{

   struct Function
   {
	 Config::CfgCb m_Callback;
	 int m_nArgs;
	 void *m_Misc;

      public:
	 Function (Config::CfgCb cb, int n, void* m)
	    : m_Callback(cb), m_nArgs(n), m_Misc(m) {}

   };
   

   struct VariableContext
   {
	 String m_Name;
	 String m_File;
	 int m_Line;

	 VariableContext(const String& name, const String& file, int line)
	    : m_Name(name), m_File(file), m_Line(line) {}
   };
   

   typedef std::map<String, Function* > FunctionList;
   typedef FunctionList::iterator FunctionLIterator;
   
   typedef std::map<String, Variable* > VariableList;
   typedef VariableList::iterator VariableLIterator;

   class ConfigP
   {
	 FunctionList m_Functions;
	 VariableList m_Variables;

      public:
	 ConfigP ()
	 {
	 }

	 ~ConfigP ()
	 {
	    Clear();
	 }

	 /**
	  * Adds a variable in the map.
	  */
	 void AddVariable(const String &name, Variable* v)
	 {
	    m_Variables[name] = v;
	    //m_Variables.insert( VariableList::value_type(name, v) );
	 }

	 /**
	  * Adds a function in the map.
	  */
	 void AddFunction(const String &name, Function* f)
	 {
	    m_Functions.insert( FunctionList::value_type(name, f) );
	 }

	 /**
	  * Find a variable in the map. If it can't be found, create a new
	  * one, with an undefined type.
	  */
	 Variable* FindVariable (const String &name)
	 {
	    VariableLIterator v = m_Variables.find( name );

	    return (v == m_Variables.end()) ? 0 : v->second;
	 }

	 /** Find a function in the function list. Return NULL if it doesn't
	  * exist.
	  */
	 Function* FindFunction (const String &name)
	 {
	    FunctionLIterator v = m_Functions.find( name );

	    return (v == m_Functions.end()) ? 0 : v->second;
	 }

	 /** Destroy all variables contained in the configuration. Clear the
	  * variable list and the function list.
	  */
	 void Clear()
	 {
	    for (VariableLIterator vi = m_Variables.begin() ; 
		 vi != m_Variables.end() ; 
		 ++vi )
	    {
	       Variable* v = vi->second;
	       delete v;
	    }
	    m_Variables.clear();

	    for (FunctionLIterator fi = m_Functions.begin() ; 
		 fi != m_Functions.end() ; 
		 ++fi )
	    {
	       Function* f = fi->second;
	       delete f;
	    }
	    m_Functions.clear();
	 }

	 void Write (std::ostream & stream) const
	 {
	    VariableList::const_iterator v;
	    for (v = m_Variables.begin(); v != m_Variables.end(); ++v)
	    {
	       stream << v->first << " = "
		      << v->second->GetWritableString() << ";\n";
	    }
	 }
   };

   /*------------------------------------------- CALLBACKS ----- */
   static bool cbInclude (void *misc, Config *self, int nargs, String *args)
   {
      assert (self != NULL);
      assert (nargs >= 1);

      if (nargs == 2 && args[2] == "sys")
	 self->LoadSystem (args[0]);
      else
	 self->Load (args[0]);
      return true;
   }

   /*------------------------------------------- CONFIG -------- */
   Config::Config ()
   {
      m_Cfg = new ConfigP ();
      AddFunction ("Include", -1, &cbInclude);
   }
   
   Config::~Config ()
   {
      delete m_Cfg;
   }
   

   // ======
   // Find a variable of the given type (Str, Int, Scalar). If its type
   // is not the right, emit a warning. If it didn't exist, use the
   // default value (def argument)
   // ======

   String
   Config::GetStr (const String &name, const String& def) const
   {
      Variable* v = m_Cfg->FindVariable( name );
      return ( v ) ? v->GetAsString() : def;
   }
   
   int
   Config::GetInt (const String &name, int def) const 
   {
      Variable* v = m_Cfg->FindVariable( name );
      return ( v ) ? v->GetAsInteger() : def;
   }


   scalar 
   Config::GetScalar(const String &name, scalar def) const
   {
      Variable* v = m_Cfg->FindVariable( name );
      return ( v ) ? v->GetAsScalar() : def;
   }

   // Write all config variables in a file. 
   void
   Config::Write (const String &file) const
   {
      String realname = Sys()->FS()->GetFileName(file);
      std::ofstream str (realname.c_str());
      m_Cfg->Write (str);
   }
   
   // Load a configuration file.
   bool
   Config::Load (const String& filename)
   {
      const String appends[] = {"",
				Ark::String(".") + Sys()->GetEnv("ARK_LANG")};
      bool res = false;

      for (size_t i = 0; i < 2; ++i)
      {
	 String fn = filename + appends[i];
	 AutoReadStream str(fn);
	 std::ifstream& is = str.Get();
	 
	 if (is.is_open())
	    res = true;
	 
	 Load(fn, is);
      }

      return res;
   }

   /** Load a "system" configuration, meaning it will be searched for
    *  (in this order) in the system configuration directory, in the
    *  user's home directory and in the current directory.
    */
   bool
   Config:: LoadSystem (const String &config)
   {
      static const char * const str[] = {"{system}/",
					 "{game}/config/",
					 "{home}/.arkhart/",
					 "./", NULL};
      bool found = false;
      int i;
      
      for (i = 0; str[i]; i++)
      {
	 String fname = String (str[i]) + config;

	 if (Sys()->FS()->IsFile(fname) && Load (fname))
	 {found = true;}
      }

      if (found) return true;

      std::ostringstream os;
      os << "Cannot find any instance of a config file called '"
	 << config
	 << "'\nin any of the search directories. Searched path : \n";

      for (i = 0; str[i]; i++)
	 os << "\t"
	    << Sys()->FS()->GetFileName(String(str[i]) + config)
	    << "\n";
      os << '\n';

      Sys()->Error (os.str().c_str());
      return false;
   }
   
   /**
    * Replaces all references to variables in the source string by
    * the actual value of these variables. A variable reference is
    * of the form "%{variable_name}".
    * 
    * \return The string with variable content. If a variable reference
    * cant be resolved, then it is not changed at all.
    */
   String
   Config::Expand (const String& src)
   {
      const String openVar = "%{";
      const String closeVar = "}";

      // gets the first delimeter
      String::size_type pos = src.find( openVar );

      // sets the string up to the first variable
      String ret(src, 0, pos);

      // there is at least one variable
      while (pos != String::npos)
      {
	 // push the pointer 2 after %{
	 pos += 2;
	   
	 String::size_type end = src.find( closeVar, pos );
	   
	 // extract variable
	 String name(src, pos, end - pos);
	   
	 // expand
	 ret += GetStr(name, String());

	 // search for next variable 
	 pos = src.find( openVar, ++end );

	 // copy until there
	 ret += src.substr( end, pos - end );
      }

      return ret;
   }
   
   // Load a configuration file.
   void
   Config::Load (const String& name, Stream &stream)
   {
      if (stream.eof())
	 return;

      Lexer lexer (name, stream);
      bool unexptok = true;

      while (true)
      {
	 String full;

	 while (true)
	 {
	    String tok = lexer.GetToken();

	    if (tok.empty() || (!isalpha (tok[0]) && (tok != ":")))
	    {
	       lexer.UngetToken();
	       break;
	    }

	    full += tok;
	 }

	 if (full.empty())
	 {
	    /* No error */
	    unexptok = false;
	    break;
	 }

	 String sym = lexer.GetToken (Lexer::SYMBOL);
	 if (sym == "=")
	 {
	    VariableContext vc(full, lexer.Name(), lexer.Line());
	   
	    String value = lexer.GetToken ();

	    if (value[0] == '"')
	    {
	       value = UnquoteString(value);
	      
	       // comment is needed here
	       while (true)
	       {
		  String multi = lexer.GetToken();
		 
		  if (multi[0] == '"')
		  {
		     value += UnquoteString(multi);
		  }
		  else
		  {
		     lexer.UngetToken();
		     break;
		  }
	       }
	      
	       m_Cfg->AddVariable(full, new StringVariable( value ));
	      
	       lexer.CheckToken (";");
	    }
	    else if (isdigit (value[0]) || (value[0] == '-'))
	    {
	       if (value.find('.') != String::npos)
	       {
		  StringVariable temp( value );
		  m_Cfg->AddVariable(full, new ScalarVariable( temp.GetAsScalar() ));
	       }
	       else
	       {
		  StringVariable temp( value );
		  m_Cfg->AddVariable(full, new IntegerVariable( temp.GetAsInteger() ));
	       }

	       lexer.CheckToken (";");
	    }
	    else
	       break;
	 }
	 else if (sym == "(")
	 {
	    String function_name = full;
	    Function* function = m_Cfg->FindFunction (function_name);

	    if (!function || !function->m_Callback)
	    {
	       // Cannot find function : emit a warning
	       String msg("Cannot find function \"" + function_name + "\"");
	       lexer.Error (msg);

	       // Skip to the next closing parenthesis
	       String tok;
	       do
	       {
		  tok = lexer.GetToken();
	       } while ((!tok.empty()) && tok != ")");

	       continue;
	    }

	    bool variable = function->m_nArgs == -1;
	    std::vector<String> args;
	    int i = 0;

	    for (;; i++)
	    {
	       String str = lexer.GetToken (Lexer::STRING);

	       if (variable || i < function->m_nArgs)
	       {
		  str = Expand (UnquoteString(str));

		  args.push_back(str);
	       }

	       str = lexer.GetToken (Lexer::SYMBOL);

	       if (str == ")")
		  break;
	       else if (str != ",")
		  lexer.Error ("Missing comma or closing parenthesis.");
	    }

	    i++;
	    if (variable == false && i > function->m_nArgs)
	    {
	       std::ostringstream os;

	       os << "Too much arguments in call to '" << function_name << "' ("
		  << i << " instead of " << function->m_nArgs << ").";

	       lexer.Error( os.str() );
	    }
	    else if (variable == false && i < function->m_nArgs)
	    {
	       std::ostringstream os;

	       os << "Too few arguments in call to '" << function_name << "' ("
		  << i << " instead of " << function->m_nArgs << ").";

	       lexer.Error( os.str() );

	       // Fill unread strings with blanks.
	       for (; i < function->m_nArgs; i++)
		  args[i] = String();
	    }
	    lexer.CheckToken (";");

	    // Call the function and destroy the args
	    function->m_Callback (function->m_Misc, this,
				  variable ? i : function->m_nArgs, &args [0]);
	 }
	 else
	    break;

      }

      if (unexptok)
	 lexer.Error ("Unexpected token");

   }

   // Add a function to this configuration object.
   void
   Config::AddFunction
   (const Ark::String &name, int nargs, CfgCb cb, void *misc)
   {
      Function* f = new Function(cb,  nargs, misc);
      
      m_Cfg->AddFunction(name, f);
   }

}
