#!/bin/sh
# CVE-2011-3667
set -e

echo "> $0 $*"

cd "$1" && patch -p1 < "$0"

exit 0

Description: CVE-2011-3667
Origin: http://bzr.mozilla.org/bugzilla/3.6/revision/7267
Bug: https://bugzilla.mozilla.org/show_bug.cgi?id=711714

--- a/Bugzilla/User.pm	2010-02-18 00:16:31 +0000
+++ b/Bugzilla/User.pm	2011-12-28 22:18:06 +0000
@@ -1694,6 +1694,32 @@
     return 1;
 }
 
+sub check_account_creation_enabled {
+    my $self = shift;
+
+    # If we're using e.g. LDAP for login, then we can't create a new account.
+    $self->authorizer->user_can_create_account
+      || ThrowUserError('auth_cant_create_account');
+
+    Bugzilla->params->{'createemailregexp'}
+      || ThrowUserError('account_creation_disabled');
+}
+
+sub check_and_send_account_creation_confirmation {
+    my ($self, $login) = @_;
+
+    $login = $self->check_login_name_for_creation($login);
+    my $creation_regexp = Bugzilla->params->{'createemailregexp'};
+
+    if ($login !~ /$creation_regexp/i) {
+        ThrowUserError('account_creation_restricted');
+    }
+
+    # Create and send a token for this new account.
+    require Bugzilla::Token;
+    Bugzilla::Token::issue_new_user_account_token($login);
+}
+
 sub login_to_id {
     my ($login, $throw_error) = @_;
     my $dbh = Bugzilla->dbh;
@@ -2196,6 +2222,17 @@
 Takes a username as its only argument. Throws an error if there is no
 user with that username. Returns a C<Bugzilla::User> object.
 
+=item C<check_account_creation_enabled>
+
+Checks that users can create new user accounts, and throws an error
+if user creation is disabled.
+
+=item C<check_and_send_account_creation_confirmation($login)>
+
+If the user request for a new account passes validation checks, an email
+is sent to this user for confirmation. Otherwise an error is thrown
+indicating why the request has been rejected.
+
 =item C<is_available_username>
 
 Returns a boolean indicating whether or not the supplied username is

--- a/Bugzilla/WebService/Constants.pm	2010-10-14 00:43:05 +0000
+++ b/Bugzilla/WebService/Constants.pm	2011-12-28 22:18:06 +0000
@@ -110,6 +110,7 @@
     # User errors are 500-600.
     account_exists        => 500,
     illegal_email_address => 501,
+    auth_cant_create_account    => 501,
     account_creation_disabled   => 501,
     account_creation_restricted => 501,
     password_too_short    => 502,

--- a/Bugzilla/WebService/User.pm	2011-02-14 07:43:51 +0000
+++ b/Bugzilla/WebService/User.pm	2011-12-28 22:18:06 +0000
@@ -27,7 +27,6 @@
 use Bugzilla::Error;
 use Bugzilla::User;
 use Bugzilla::Util qw(trim);
-use Bugzilla::Token;
 use Bugzilla::WebService::Util qw(filter validate);
 
 # Don't need auth to login
@@ -86,18 +85,8 @@
     my $email = trim($params->{email})
         || ThrowCodeError('param_required', { param => 'email' });
 
-    my $createexp = Bugzilla->params->{'createemailregexp'};
-    if (!$createexp) {
-        ThrowUserError("account_creation_disabled");
-    }
-    elsif ($email !~ /$createexp/) {
-        ThrowUserError("account_creation_restricted");
-    }
-
-    $email = Bugzilla::User->check_login_name_for_creation($email);
-
-    # Create and send a token for this new account.
-    Bugzilla::Token::issue_new_user_account_token($email);
+    Bugzilla->user->check_account_creation_enabled;
+    Bugzilla->user->check_and_send_account_creation_confirmation($email);
 
     return undef;
 }
@@ -365,15 +354,15 @@
 
 =over
 
-=item 500 (Illegal Email Address)
+=item 500 (Account Already Exists)
+
+An account with that email address already exists in Bugzilla.
+
+=item 501 (Illegal Email Address)
 
 This Bugzilla does not allow you to create accounts with the format of
 email address you specified. Account creation may be entirely disabled.
 
-=item 501 (Account Already Exists)
-
-An account with that email address already exists in Bugzilla.
-
 =back
 
 =back

--- a/createaccount.cgi	2007-11-12 04:03:16 +0000
+++ b/createaccount.cgi	2011-12-28 22:18:06 +0000
@@ -31,47 +31,24 @@
 use Bugzilla;
 use Bugzilla::Constants;
 use Bugzilla::Error;
-use Bugzilla::User;
-use Bugzilla::BugMail;
-use Bugzilla::Util;
 
 # Just in case someone already has an account, let them get the correct footer
 # on an error message. The user is logged out just after the account is
 # actually created.
-Bugzilla->login(LOGIN_OPTIONAL);
-
-my $dbh = Bugzilla->dbh;
+my $user = Bugzilla->login(LOGIN_OPTIONAL);
 my $cgi = Bugzilla->cgi;
 my $template = Bugzilla->template;
-my $vars = {};
-
-$vars->{'doc_section'} = 'myaccount.html';
+my $vars = { doc_section => 'myaccount.html' };
 
 print $cgi->header();
 
-# If we're using LDAP for login, then we can't create a new account here.
-unless (Bugzilla->user->authorizer->user_can_create_account) {
-    ThrowUserError("auth_cant_create_account");
-}
-
-my $createexp = Bugzilla->params->{'createemailregexp'};
-unless ($createexp) {
-    ThrowUserError("account_creation_disabled");
-}
-
+$user->check_account_creation_enabled;
 my $login = $cgi->param('login');
 
 if (defined($login)) {
-    $login = Bugzilla::User->check_login_name_for_creation($login);
+    $user->check_and_send_account_creation_confirmation($login);
     $vars->{'login'} = $login;
 
-    if ($login !~ /$createexp/) {
-        ThrowUserError("account_creation_restricted");
-    }
-
-    # Create and send a token for this new account.
-    Bugzilla::Token::issue_new_user_account_token($login);
-
     $template->process("account/created.html.tmpl", $vars)
       || ThrowTemplateError($template->error());
     exit;

--- a/token.cgi	2009-10-09 04:31:08 +0000
+++ b/token.cgi	2011-12-28 22:18:06 +0000
@@ -355,6 +355,7 @@
 sub request_create_account {
     my $token = shift;
 
+    Bugzilla->user->check_account_creation_enabled;
     my (undef, $date, $login_name) = Bugzilla::Token::GetTokenData($token);
     $vars->{'token'} = $token;
     $vars->{'email'} = $login_name . Bugzilla->params->{'emailsuffix'};
@@ -368,6 +369,7 @@
 sub confirm_create_account {
     my $token = shift;
 
+    Bugzilla->user->check_account_creation_enabled;
     my (undef, undef, $login_name) = Bugzilla::Token::GetTokenData($token);
 
     my $password = $cgi->param('passwd1') || '';

