// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"

template< typename T, int Size, typename MatrixType, typename VectorType >
void helper_checkEuclidean( const char *strType )
{
    int i, j;
    VectorType v1(Size), v2(Size), v3(Size);
    MatrixType m1(Size), m2(Size), m3(Size);
    v1.loadRandomUnit();
    QVERIFY( test_isApprox( v1.norm(), static_cast<T>(1) ) );
    do v1.loadRandom(); while( v1.norm() == static_cast<T>(0) );
    QVERIFY( test_isApprox( v1.norm() * v1.norm(), v1.norm2() ) );
    v1.normalize();
    QVERIFY( test_isApprox( v1.norm(), static_cast<T>(1) ) );
    do v1.loadRandom(); while( v1.norm() == static_cast<T>(0) );
    v2.loadRandom();
    QVERIFY( test_isApprox( Util::conj( dot( v1, v2 ) ), dot( v2, v1 ) ) );
    v1.loadRandom();
    v3 = v1.ortho();
    QVERIFY( test_isNegligible( dot( v1, v3 ), static_cast<T>(1) ) );
    QVERIFY( test_isApprox( v3.norm(), static_cast<T>(1) ) );
    QVERIFY( test_isApprox( ( v1 + v3).norm2(), v1.norm2() + v3.norm2() ) );
    v1.makeOrthoVector( & v3 );
    QVERIFY( test_isNegligible( dot( v1, v3 ), static_cast<T>(1) ) );
    QVERIFY( test_isApprox( v3.norm(), static_cast<T>(1) ) );
    QVERIFY( test_isApprox( ( v1 + v3).norm2(), v1.norm2() + v3.norm2() ) );
    v3.loadOrtho( v1 );
    QVERIFY( test_isNegligible( dot( v1, v3 ), static_cast<T>(1) ) );
    QVERIFY( test_isApprox( v3.norm(), static_cast<T>(1) ) );
    QVERIFY( test_isApprox( ( v1 + v3).norm2(), v1.norm2() + v3.norm2() ) );
    m1.loadOrthoBasis(v1);
    QVERIFY( compareVectors( v1, m1.column(0), strType ) );
    for( i = 0; i < Size; i++ )
        for( j = 0; j < i; j++ )
            QVERIFY( test_isNegligible( dot( m1.column(i),
                                             m1.column(j) ),
                                        static_cast<T>(1) ) );
    v1.normalize();
    m1.loadOrthoBasis(v1);
    QVERIFY( compareVectors( v1, m1.column(0), strType ) );
    m2 = m1 * m1.adjoint();
    m3.loadIdentity();
    QVERIFY( compareMatrices( m2, m3, strType ) );
    v1.loadRandomUnit();
    v2 = v1.normalized();
    QVERIFY( compareVectors( v1, v2, strType ) );
}

template< typename T, typename MatrixType, typename VectorType >
void helper_checkCrossProduct( const char *strType )
{
    (void) strType;
    VectorType u(3), v(3), w(3);
    u.loadRandomUnit();
    v.loadRandomUnit();
    T n2 = static_cast<T>(1) - dot(u,v) * dot(u,v);
    u.cross(v, &w);
    QVERIFY( test_isApprox( w.norm2(), n2 ) );
    w = u.cross(v);
    QVERIFY( test_isApprox( w.norm2(), n2 ) );
    w = cross(u, v);
    QVERIFY( test_isApprox( w.norm2(), n2 ) );
}

void MainTest::checkEuclidean()
{
#define HELPER_CHECKEUCLIDEAN_FIX( T, Size ) \
    helper_checkEuclidean< T, Size, \
        Matrix< T, Size >, Vector< T, Size > >\
            ("fixed<"__STRING(T)","__STRING(Size)">")

#define HELPER_CHECKEUCLIDEAN_DYN( T, Size ) \
    helper_checkEuclidean< T, Size, \
        MatrixX<T>, VectorX<T> >\
            ("dynamic<"__STRING(T)">("__STRING(Size)")")

#define HELPER_CHECKCROSSPRODUCT_FIX( T ) \
    helper_checkCrossProduct< T, \
        Matrix< T, 3 >, Vector< T, 3 > >\
            ("fixed<"__STRING(T)",3>")

#define HELPER_CHECKCROSSPRODUCT_DYN( T ) \
    helper_checkCrossProduct< T, \
        MatrixX< T >, VectorX< T > >\
            ("dynamic<"__STRING(T)",3>")


    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        HELPER_CHECKEUCLIDEAN_FIX( double, 2 );
        HELPER_CHECKEUCLIDEAN_FIX( float, 3 );
        HELPER_CHECKEUCLIDEAN_FIX( double, 4 );
        HELPER_CHECKEUCLIDEAN_FIX( double, 7 );
        HELPER_CHECKEUCLIDEAN_FIX( float, 8 );
        HELPER_CHECKEUCLIDEAN_FIX( double, 12 );
        HELPER_CHECKEUCLIDEAN_FIX( complex<double>, 7 );
        HELPER_CHECKEUCLIDEAN_FIX( complex<float>, 8 );
        HELPER_CHECKEUCLIDEAN_FIX( complex<double>, 12 );
    
        HELPER_CHECKEUCLIDEAN_DYN( double, 2 );
        HELPER_CHECKEUCLIDEAN_DYN( float, 3 );
        HELPER_CHECKEUCLIDEAN_DYN( double, 4 );
        HELPER_CHECKEUCLIDEAN_DYN( double, 7 );
        HELPER_CHECKEUCLIDEAN_DYN( float, 8 );
        HELPER_CHECKEUCLIDEAN_DYN( double, 12 );
        HELPER_CHECKEUCLIDEAN_DYN( complex<double>, 7 );
        HELPER_CHECKEUCLIDEAN_DYN( complex<float>, 8 );
        HELPER_CHECKEUCLIDEAN_DYN( complex<double>, 12 );

        HELPER_CHECKCROSSPRODUCT_FIX( float );
        HELPER_CHECKCROSSPRODUCT_FIX( double );
        HELPER_CHECKCROSSPRODUCT_DYN( float );
        HELPER_CHECKCROSSPRODUCT_DYN( double );
    }
}
