/*
 * Copyright (C) 2009, 2010 Hermann Meyer, James Warden, Andreas Degert
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * --------------------------------------------------------------------------
 *
 *  This is the JConv settings handler class used by guitarix
 *  There can only be one instance at all time. To get a pointer
 *  to the unique instance, simply call
 *
 *  GxJConvSettings* jcset = GxJConvSettings::instance();
 *
 *  throughout the entire program.
 *
 *  This class belongs to the gx_jconv namespace
 *
 * --------------------------------------------------------------------------
 */
#include <libgen.h>
#include "guitarix.h"

using namespace gx_system;
using namespace gx_sndfile;
using namespace gx_jack;
using namespace gx_child_process;
using namespace gx_preset;

namespace gx_jconv
{

// --------------- static vars
float GxJConvSettings::checkbutton7 = 0.;

// ---------------  constructor
GxJConvSettings::GxJConvSettings()
{
	// default parameters
	fIRDir      = getenv("HOME");
	fIRFile     = "nofile";

	fGain       = 0.2;
	flGain       = 0.2;
	fMem        = 8000;
	fMode       = kJConvCopy;
	fBufferSize = gx_jack::jack_bs;
	fOffset     = 0;
	fLength     = 0;
	fDelay      = 0;
	flDelay      = 0;
	if (gx_jack::jack_bs == 0) {
		fBufferSize = 128;
	}
	// invalidate due to no IR
	invalidate();
}

// --------------- reset internal setting
void GxJConvSettings::resetSetting()
{
	// default parameters
	fIRDir      = getenv("HOME");
	fIRFile     = "nofile";

	fGain       = 0.2;
	flGain       = 0.2;
	fMem        = 8000;
	fMode       = kJConvCopy;
	fBufferSize = gx_jack::jack_bs;
	fOffset     = 0;
	fLength     = 0;
	fDelay      = 0;
	flDelay      = 0;
	if (gx_jack::jack_bs == 0) {
		fBufferSize = 128;
	}
	// invalidate due to no IR
	invalidate();
}

// --------------- attempt to validate the settings
// Note: for now, simply check that the IR file is a wav file
void GxJConvSettings::validate()
{
	gx_engine::Audiofile audio;
	fValidSettings = (audio.open_read(getFullIRPath()) == 0 &&
	                  audio.type() == gx_engine::Audiofile::TYPE_WAV);
}


// --------------- invalidate the settings via IR file
inline void GxJConvSettings::invalidate()
{
	fIRFile = "nofile";
	fValidSettings = false;
}

void GxJConvSettings::writeJSON(JsonWriter& w)
{
	w.begin_object(true);
	w.write_key("jconv.IRFile"); w.write(fIRFile, true);
	w.write_key("jconv.IRDir"); w.write(fIRDir, true);
	w.write_key("jconv.Gain"); w.write(fGain, true);
	w.write_key("jconv.lGain"); w.write(flGain, true);
	w.write_key("jconv.Mem"); w.write(fMem, true);
	w.write_key("jconv.Mode"); w.write(fMode, true);
	w.write_key("jconv.BufferSize"); w.write(fBufferSize, true);
	w.write_key("jconv.Offset"); w.write(fOffset, true);
	w.write_key("jconv.Length"); w.write(fLength, true);
	w.write_key("jconv.Delay"); w.write(fDelay, true);
	w.write_key("jconv.lDelay"); w.write(flDelay, true);
	w.end_object(true);
}

GxJConvSettings::GxJConvSettings(JsonParser& jp)
{
	jp.next(JsonParser::begin_object);
	do {
		jp.next(JsonParser::value_key);
		if (jp.current_value() == "jconv.IRFile") {
			jp.next(JsonParser::value_string);
			fIRFile = jp.current_value();
		} else if (jp.current_value() == "jconv.IRDir") {
			jp.next(JsonParser::value_string);
			fIRDir = jp.current_value();
		} else if (jp.current_value() == "jconv.Gain") {
			jp.next(JsonParser::value_number);
			fGain = jp.current_value_float();
		} else if (jp.current_value() == "jconv.lGain") {
			jp.next(JsonParser::value_number);
			flGain = jp.current_value_float();
		} else if (jp.current_value() == "jconv.Mem") {
			jp.next(JsonParser::value_number);
			fMem = jp.current_value_int();
		} else if (jp.current_value() == "jconv.Mode") {
			jp.next(JsonParser::value_number);
			fMode = (GxJConvMode)jp.current_value_int(); //FIXME check
		} else if (jp.current_value() == "jconv.BufferSize") {
			jp.next(JsonParser::value_number);
			fBufferSize = jp.current_value_int();
		} else if (jp.current_value() == "jconv.Offset") {
			jp.next(JsonParser::value_number);
			fOffset = jp.current_value_int();
		} else if (jp.current_value() == "jconv.Length") {
			jp.next(JsonParser::value_number);
			fLength = jp.current_value_int();
		} else if (jp.current_value() == "jconv.Delay") {
			jp.next(JsonParser::value_number);
			fDelay = jp.current_value_int();
		} else if (jp.current_value() == "jconv.lDelay") {
			jp.next(JsonParser::value_number);
			flDelay = jp.current_value_int();
		} else {
			jp.skip_object();
			gx_print_warning("jconv settings", "unknown key: " + jp.current_value());
		}
	} while (jp.peek() == JsonParser::value_key);
	jp.next(JsonParser::end_object);
}

// --------------------- dump parameters on demand to a string
void GxJConvSettings::dumpParameters()
{
	static string tab = "  ";
	static ostringstream dump;
	dump.str("");

	dump << "--- JConv is using IR " << endl
	     << getFullIRPath()          << endl
	     << " in mode " << (fMode == kJConvCopy ? "'copy'" : "'read'") << endl

	     << tab << "buffer size: " << fBufferSize << tab
	     << tab << "memory: "      << fMem  << endl

	     << tab << "gain: "   << fGain   << tab
	     << tab << "left gain: "   << flGain   << tab
	     << tab << "delay: "  << fDelay  << tab
	     << tab << "left delay: "  << flDelay  << tab
	     << tab << "offset: " << fOffset << tab
	     << tab << "length: " << fLength << tab;

	gx_print_info("Saving JConv parameters to file", dump.str().c_str());
}

//----- show the jconv settings widget
void gx_show_jconv_dialog_gui(GtkWidget *widget, gpointer data)
{
	gtk_widget_show_all(gx_gui::jc_dialog);

}

//----- disable the right gain and delay when run in copy mode
void gx_set_sensitive(GtkWidget *gwidget, gpointer obj)
{
	GtkWidget *widget = (GtkWidget *)obj;
	GxJConvSettings* jcset = GxJConvSettings::instance();
	switch (jcset->getMode())
	{
	case kJConvRead:
		gtk_widget_set_sensitive(GTK_WIDGET(widget),TRUE);
		break;

	case kJConvCopy:
	default:
		gtk_widget_set_sensitive(GTK_WIDGET(widget),FALSE);
		break;
	}


}

//----- refresh jcconv settings GUI
void gx_reload_jcgui()
{
	gtk_widget_destroy(gx_gui::jc_dialog);

	gx_setting_jconv_dialog_gui(gx_gui::jc_dialog,NULL);

}


//----- hide the jconv settings widget
gboolean gx_delete_event( GtkWidget *widget, gpointer   data )
{
	gtk_widget_hide(gx_gui::jc_dialog);
	return TRUE;
}

//------- idle thread to reset the file chooser filter
gboolean gx_set_file_filter(gpointer data)
{
	GxJConvSettings* jcset = GxJConvSettings::instance();
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(gx_gui::fbutton),
	                                    jcset->getIRDir().c_str());
	return false;

}

//----- show value to a label
void gx_set_value(GtkObject * widget, gpointer obj)
{
	GtkWidget *lw = (GtkWidget *)obj;
	float	v = GTK_ADJUSTMENT (widget)->value;
	int fPrecision = GTK_ADJUSTMENT (widget)->step_increment;
	char s[64];
	if (fPrecision >= 1)
		snprintf(s, 63, "%d", int(v));

	else if (fPrecision == 0)
	{
		const char* format[] = {"%.1f", "%.2f", "%.3f"};
		snprintf(s, 63, format[2-1], v);
	}
	gtk_label_set_text(GTK_LABEL(lw), s);
}

//----- create knobs/sliders with labels
GtkWidget * gx_knob(const char* label,int mode, float init, float min, float max, float step)
{

	GtkObject* adj = gtk_adjustment_new(init, min, max, step, 10*step, 0);
	GtkWidget* lw = gtk_label_new("");
	GtkWidget* lwl = gtk_label_new(label);
	gtk_widget_set_name (lw,"value_label");
	gtk_widget_set_name (lwl,"effekt_label");

	GtkStyle *style = gtk_widget_get_style(lw);
	pango_font_description_set_size(style->font_desc, 8*PANGO_SCALE);
	pango_font_description_set_weight(style->font_desc, PANGO_WEIGHT_BOLD);
	gtk_widget_modify_font(lw, style->font_desc);
	gtk_widget_modify_font(lwl, style->font_desc);
	GtkWidget* slider;

	GtkRegler myGtkRegler;
	if(!mode) slider = myGtkRegler.gtk_regler_new_with_adjustment(GTK_ADJUSTMENT(adj));
	else slider = myGtkRegler.gtk_vslider_new_with_adjustment(GTK_ADJUSTMENT(adj));
	gtk_range_set_inverted (GTK_RANGE(slider), TRUE);
	gx_set_value(GTK_OBJECT(adj), lw);
	// connect label with value
	g_signal_connect(GTK_OBJECT(adj), "value-changed",
	                 G_CALLBACK(gx_set_value),
	                 (gpointer)lw);

	GtkWidget* box     = gtk_vbox_new (FALSE,  1);
	gtk_container_set_border_width (GTK_CONTAINER (box), 1);

	gtk_container_add (GTK_CONTAINER (box),    lwl);
	gtk_container_add (GTK_CONTAINER (box),    slider);
	gtk_container_add (GTK_CONTAINER (box),    lw);

	gtk_widget_show(lwl);
	gtk_widget_show(slider);
	gtk_widget_show(lw);
	gtk_widget_show(box);

	return slider;

}

// ---------------------  The JConv setting dialog GUI
void gx_setting_jconv_dialog_gui(GtkWidget *widget, gpointer data)
{
	GxJConvSettings* jcset = GxJConvSettings::instance();

	// -- main window
	gx_gui::jc_dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_decorated(GTK_WINDOW(gx_gui::jc_dialog), TRUE);
	gtk_window_set_resizable(GTK_WINDOW(gx_gui::jc_dialog), FALSE);
	gtk_window_set_gravity(GTK_WINDOW(gx_gui::jc_dialog), GDK_GRAVITY_SOUTH);
	gtk_window_set_transient_for (GTK_WINDOW(gx_gui::jc_dialog), GTK_WINDOW(gx_gui::fWindow));
	gtk_window_set_position (GTK_WINDOW(gx_gui::jc_dialog), GTK_WIN_POS_MOUSE);
	gtk_window_set_keep_below (GTK_WINDOW(gx_gui::jc_dialog), FALSE);
	gtk_window_set_title (GTK_WINDOW (gx_gui::jc_dialog), "Convolver Settings");
	gtk_window_set_destroy_with_parent(GTK_WINDOW(gx_gui::jc_dialog), TRUE);

	// -- labels
	GtkWidget* label  = gtk_label_new ("\n         Convolver  Settings      \n"
	                                   "\n");
	gx_gui::label1    = gtk_label_new (" \n");
	gtk_widget_set_name (gx_gui::label1,"effekt_label");

	GtkWidget* label2 = gtk_label_new (" partition size");
	GtkWidget* label5 = gtk_label_new (" mode ");

	GdkColor colorGreen;
	gdk_color_parse("#969292", &colorGreen);
	gtk_widget_modify_fg (label2, GTK_STATE_NORMAL, &colorGreen);
	gtk_widget_modify_fg (label5, GTK_STATE_NORMAL, &colorGreen);
	GtkStyle *style = gtk_widget_get_style(label2);
	pango_font_description_set_size(style->font_desc, 8*PANGO_SCALE);
	pango_font_description_set_weight(style->font_desc, PANGO_WEIGHT_BOLD);
	gtk_widget_modify_font(label2, style->font_desc);
	gtk_widget_modify_font(label5, style->font_desc);
	gtk_widget_modify_font(label, style->font_desc);

	// -- OK button
	GtkWidget* ok_button = gtk_button_new_with_label("OK");


	// ----- setting GUI stuff with current initial values

	ostringstream lab; // label text

	// -- wave file info
	int chans      = 0; // channels
	int sr         = 0; // sample rate
	int framecount = 0; // number of frames
	SNDFILE* sf    = NULL;

	// try to open IR file
	jcset->validate();

	if (jcset->isValid())
	{
		sf = openInputSoundFile(jcset->getFullIRPath().c_str(),
		                        &chans, &sr, &framecount);
		closeSoundFile(sf);

		// display IR file info
		lab.str("");
		lab << "IR file info: " << endl
		    << chans            << " channel(s) "
		    << sr               << " Sample rate "
		    << framecount       << " Samples ";

		gtk_label_set_text(GTK_LABEL(gx_gui::label1), lab.str().c_str());
	}
	if (chans == 1) jcset->setMode ( kJConvCopy);

	// -- Delay
	GtkWidget* dslider = gx_knob ("left delay ",0,jcset->getDelay(), 0, 6000, 1);
	GtkAdjustment *dadj = gtk_range_get_adjustment(GTK_RANGE(dslider));

	GtkWidget * dslider_box = gtk_widget_get_parent(GTK_WIDGET(dslider));
	jcset->setDelay(gtk_adjustment_get_value(GTK_ADJUSTMENT(dadj)));
	// -- Delay
	GtkWidget* dlslider = gx_knob ("right delay ",0,jcset->getlDelay(), 0, 6000, 1);
	GtkAdjustment *dladj = gtk_range_get_adjustment(GTK_RANGE(dlslider));

	GtkWidget * dlslider_box = gtk_widget_get_parent(GTK_WIDGET(dlslider));
	jcset->setlDelay(gtk_adjustment_get_value(GTK_ADJUSTMENT(dladj)));

	// -- GAIN
	GtkWidget* gslider = gx_knob ("left gain",0,jcset->getGain(), 0.0, 5.0, 0.01);
	GtkAdjustment *gadj = gtk_range_get_adjustment(GTK_RANGE(gslider));

	GtkWidget * gslider_box = gtk_widget_get_parent(GTK_WIDGET(gslider));
	jcset->setGain(gtk_adjustment_get_value(GTK_ADJUSTMENT(gadj)));
	// -- GAIN
	GtkWidget* glslider = gx_knob ("right gain",0,jcset->getlGain(), 0.0, 5.0, 0.01);
	GtkAdjustment *gladj = gtk_range_get_adjustment(GTK_RANGE(glslider));

	GtkWidget * glslider_box = gtk_widget_get_parent(GTK_WIDGET(glslider));
	jcset->setlGain(gtk_adjustment_get_value(GTK_ADJUSTMENT(gladj)));

	// -- MEMORY
	mslider = gx_knob ("max mem",1,jcset->getMem(), 32, 400000, 1000);
	GtkAdjustment *madj = gtk_range_get_adjustment(GTK_RANGE(mslider));
	gtk_range_set_value(GTK_RANGE(mslider), framecount );

	GtkWidget * mslider_box = gtk_widget_get_parent(GTK_WIDGET(mslider));
	jcset->setMem(gtk_adjustment_get_value(GTK_ADJUSTMENT(madj)));

	// -- READ MODE
	GtkWidget* jcmode_combo = gtk_combo_box_new_text();
	string buffer = "/impulse/copy";
	gtk_combo_box_append_text(GTK_COMBO_BOX(jcmode_combo), buffer.c_str());

	buffer = "/impulse/read";
	gtk_combo_box_append_text(GTK_COMBO_BOX(jcmode_combo), buffer.c_str());

	// turn on default
	gtk_combo_box_set_active(GTK_COMBO_BOX(jcmode_combo), (gint)jcset->getMode());

	// -- BUFFER SIZE
	GtkWidget* bs_combo = gtk_combo_box_new_text();
	const int min_pow = 6;  // 2**6  = 64
	const int max_pow = 12; // 2**12 = 4096

	for (int i = min_pow; i <= max_pow; i++)
	{
		ostringstream s;
		jack_nframes_t bs = (jack_nframes_t)pow(2.,i);
		s << bs;
		gtk_combo_box_append_text(GTK_COMBO_BOX(bs_combo), s.str().c_str());

		if (bs == jcset->getBufferSize())
			gtk_combo_box_set_active(GTK_COMBO_BOX(bs_combo), i - min_pow);
	}

	// -- IR File Chooser
	gx_gui::fbutton =
		gtk_file_chooser_button_new ("Select a *.wav file", GTK_FILE_CHOOSER_ACTION_OPEN);

	gtk_file_chooser_set_local_only(GTK_FILE_CHOOSER (gx_gui::fbutton), true);
	gtk_file_chooser_set_show_hidden (GTK_FILE_CHOOSER (gx_gui::fbutton),false);

	GtkFileFilter* filter = gtk_file_filter_new ();
	gtk_file_filter_add_pattern (GTK_FILE_FILTER(filter), "*.wav");
	gtk_file_filter_add_pattern (GTK_FILE_FILTER(filter), "*.WAV");
	gtk_file_filter_add_pattern (GTK_FILE_FILTER(filter), "*.Wav");
	gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (gx_gui::fbutton), GTK_FILE_FILTER(filter));

	// label for display the filename
	gx_gui::label6 = gtk_label_new(jcset->getIRFile().c_str());

	// display wave view of IR file
	GtkWidget* waveview = gx_wave_view();
	gtk_widget_set_size_request (GTK_WIDGET(waveview), 300.0, 200.0);

	// scrolled window
	GtkWidget* scrlwd = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrlwd),
	                               GTK_POLICY_ALWAYS,GTK_POLICY_NEVER);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrlwd),
	                                    GTK_SHADOW_IN);

	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrlwd), waveview);
	gtk_widget_set_size_request (GTK_WIDGET(scrlwd), 304.0, -1);

	//----- arrange widgets
	GtkWidget* box     = gtk_vbox_new (FALSE,  4);
	GtkWidget* box1    = gtk_vbox_new (TRUE,  4);
	GtkWidget* box2    = gtk_hbox_new (FALSE,  0);
	GtkWidget* box3    = gtk_vbox_new (FALSE,  4);
	GtkWidget* box4    = gtk_vbox_new (FALSE, 4);
	GtkWidget* box5    = gtk_hbox_new (FALSE, 4);
	GtkWidget* box6    = gtk_hbox_new (TRUE, 4);
	GtkWidget* box7    = gtk_hbox_new (FALSE, 8);
	GtkWidget* box8    = gtk_vbox_new (FALSE, 0);
	GtkWidget* box9    = gtk_hbox_new (FALSE, 8);
	GtkWidget* box10    = gtk_vbox_new (FALSE, 0);
	GtkWidget* box11    = gtk_vbox_new (TRUE, 0);
	// GtkWidget* box12    = gtk_vbox_new (FALSE, 4);
	GtkWidget* viewbox = gtk_event_box_new ();

	gtk_container_set_border_width (GTK_CONTAINER (box7), 8);
	gtk_container_set_border_width (GTK_CONTAINER (viewbox), 8);
	gtk_container_set_border_width (GTK_CONTAINER (box9), 14);
	gtk_container_set_border_width (GTK_CONTAINER (box6), 4);
	gtk_container_set_border_width (GTK_CONTAINER (box2), 2);

	gtk_container_add (GTK_CONTAINER (box4),   label);
	gtk_container_add (GTK_CONTAINER (box4),    viewbox);
	gtk_container_add (GTK_CONTAINER (viewbox), scrlwd);
	gtk_container_add (GTK_CONTAINER (gx_gui::jc_dialog), box7);
	gtk_container_add (GTK_CONTAINER (box7),   box9);
	gtk_container_add (GTK_CONTAINER (box9),   box4);
	gtk_container_add (GTK_CONTAINER (box5),   gx_gui::label6);
	gtk_container_add (GTK_CONTAINER (box5),   gx_gui::fbutton);
	gtk_container_add (GTK_CONTAINER (box4),   box5);
	gtk_container_add (GTK_CONTAINER (box4),   gx_gui::label1);
	gtk_container_add (GTK_CONTAINER (box4),   box6);
	gtk_container_add (GTK_CONTAINER (box4),   box2);
	gtk_container_add (GTK_CONTAINER (box2),   box8);
	gtk_container_add (GTK_CONTAINER (box8),   gslider_box);
	gtk_container_add (GTK_CONTAINER (box8),   glslider_box);
	gtk_container_add (GTK_CONTAINER (box2),   box10);
	gtk_container_add (GTK_CONTAINER (box10),   dslider_box);
	gtk_container_add (GTK_CONTAINER (box10),   dlslider_box);
	gtk_container_add (GTK_CONTAINER (box2),   box11);
	gtk_container_add (GTK_CONTAINER (box11),   mslider_box);
	//gtk_container_add (GTK_CONTAINER (box11),   box12);
	gtk_container_add (GTK_CONTAINER (box2),   box1);
	//gtk_container_add (GTK_CONTAINER (box1),   box3); //FIXME mode (read/copy) remove after jconvolver transition
	gtk_container_add (GTK_CONTAINER (box3),   label5);
	gtk_container_add (GTK_CONTAINER (box3),   jcmode_combo);
	gtk_container_add (GTK_CONTAINER (box1),   box);
	gtk_box_pack_start(GTK_BOX(box), label2, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(box), bs_combo, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (box4),   ok_button);

	//----- connect signals to callback cairo funcs
	g_signal_connect(box9, "expose-event", G_CALLBACK(gx_cairo::conv_widget_expose), NULL);
	g_signal_connect(box7, "expose-event", G_CALLBACK(gx_cairo::rectangle_expose), NULL);
	g_signal_connect(box2, "expose-event", G_CALLBACK(gx_cairo::rectangle_skin_color_expose), NULL);
	g_signal_connect(gx_gui::label6, "expose-event", G_CALLBACK(gx_cairo::rectangle_skin_color_expose), NULL);
	g_signal_connect(label, "expose-event", G_CALLBACK(gx_cairo::convolver_icon_expose), NULL);

	// IR file select
	g_signal_connect (gx_gui::fbutton, "file-set",
	                  G_CALLBACK(gx_select_and_draw_jconv_ir), (gpointer) jcmode_combo);

	// confirm and save setting
	g_signal_connect_swapped(ok_button, "clicked",
	                         G_CALLBACK(gx_delete_event), gx_gui::jc_dialog);
	g_signal_connect_swapped (G_OBJECT (gx_gui::jc_dialog), "delete_event",
	                          G_CALLBACK (gx_delete_event), (gpointer) ok_button);


	// gain setting
	g_signal_connect(GTK_OBJECT(gadj), "value-changed",
	                 G_CALLBACK(gx_acquire_jconv_value),
	                 (gpointer)kJConvGain);
	// gain setting
	g_signal_connect(GTK_OBJECT(gladj), "value-changed",
	                 G_CALLBACK(gx_acquire_jconv_value),
	                 (gpointer)kJConvlGain);

	// memory
	g_signal_connect(GTK_OBJECT(madj), "value-changed",
	                 G_CALLBACK(gx_acquire_jconv_value),
	                 (gpointer)kJConvMem);

	// buffer size
	g_signal_connect(GTK_OBJECT (bs_combo), "changed",
	                 G_CALLBACK(gx_acquire_jconv_value),
	                 (gpointer)kJConvBuffersize);

	// read mode
	g_signal_connect(GTK_OBJECT (jcmode_combo), "changed",
	                 G_CALLBACK(gx_acquire_jconv_value),
	                 (gpointer)kJConvMode);
	g_signal_connect(GTK_OBJECT (jcmode_combo), "changed",
	                 G_CALLBACK(gx_set_sensitive),
	                 (gpointer)glslider);
	g_signal_connect(GTK_OBJECT (jcmode_combo), "changed",
	                 G_CALLBACK(gx_set_sensitive),
	                 (gpointer)dlslider);

	// delay setting
	g_signal_connect(GTK_OBJECT(dadj), "value-changed",
	                 G_CALLBACK(gx_acquire_jconv_value),
	                 (gpointer)kJConvDelay);
	// delay setting
	g_signal_connect(GTK_OBJECT(dladj), "value-changed",
	                 G_CALLBACK(gx_acquire_jconv_value),
	                 (gpointer)kJConvlDelay);

	// set courent folder to file chooser
	g_idle_add(gx_set_file_filter,NULL);

	//----- load file to wave view
	gx_waveview_set_value(GTK_WIDGET(gx_gui::fbutton),NULL);
}


// ---------------  retrieve a setting value
void gx_acquire_jconv_value(GtkWidget *widget, gpointer data )
{
	GxJConvSettings* jcset = GxJConvSettings::instance();

	GxJConvParamType ptype = (GxJConvParamType)GPOINTER_TO_INT(data);
	string s;

	switch (ptype)
	{
	case kJConvGain:
		jcset->setGain(gtk_adjustment_get_value(GTK_ADJUSTMENT(widget)));
		break;

	case kJConvlGain:
		jcset->setlGain(gtk_adjustment_get_value(GTK_ADJUSTMENT(widget)));
		break;

	case kJConvMem:
		jcset->setMem((guint)gtk_adjustment_get_value(GTK_ADJUSTMENT(widget)));
		break;

	case kJConvBuffersize:
		s = gtk_combo_box_get_active_text(GTK_COMBO_BOX(widget));
		jcset->setBufferSize((jack_nframes_t)atoi(s.c_str()));
		break;

	case kJConvMode:
		s = gtk_combo_box_get_active_text(GTK_COMBO_BOX(widget));
		jcset->setMode(((int)s.find("copy") != -1 ? kJConvCopy : kJConvRead));
		break;

	case kJConvDelay:
		jcset->setDelay(gtk_adjustment_get_value(GTK_ADJUSTMENT(widget)));
		break;

	case kJConvlDelay:
		jcset->setlDelay(gtk_adjustment_get_value(GTK_ADJUSTMENT(widget)));
		break;

	case kJConvOffset:
	case kJConvLength:

	default:  // do nothing
		break;
	}
}


// --------------- IR file processing
void gx_select_and_draw_jconv_ir(GtkWidget* widget, gpointer obj)
{
	// get the chosen file from the file chooser
	gchar * _file   = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(widget));
	// return when file is empty
	if (!_file) {
		g_free(_file);
		return;
	}
	GxJConvSettings* jcset = GxJConvSettings::instance();
	GtkWidget *combo = (GtkWidget *)obj;

	string file = _file;
	g_free(_file);
	string folder = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(widget));

	if (!file.empty()) {
		// let's save it into the JConv settings
		string tmp_file = basename((char*)file.c_str());
		jcset->setIRFile(tmp_file);
		jcset->setIRDir(folder);

		// let's validate it
		jcset->validate();

		if (jcset->isValid()) {
			g_idle_add(gx_set_file_filter,NULL);

			//----- get name from used wav file
			gtk_label_set_text(GTK_LABEL(gx_gui::label6),
			                   jcset->getIRFile().c_str());

			//----- get snd info from IR file
			int chans, sr, framescount;

			SNDFILE* sf = openInputSoundFile(file.c_str(),
			                                 &chans, &sr, &framescount);
			closeSoundFile(sf);

			/* We set the sndfile info here to the label, if resampling is needed/done. */
			//----- set label text
			ostringstream lab;
			lab.str("");
			lab << "IR file info: " << endl
			    << chans            << " channel(s) "
			    << sr << " Sample rate "
			    << framescount      << " Samples ";
			gtk_label_set_text(GTK_LABEL(gx_gui::label1), lab.str().c_str());

			/* Note, resample the file didn't change the samples, so we could draw it here
			   already, independend from the result of resampling. Also a no resampled file could be used
			   if the user wish so.*/

			//----- draw the wave file
			gx_waveview_set_value(widget, NULL);
			gtk_range_set_value(GTK_RANGE(mslider), framescount );

			//----- set mode chooser to copy and disable it when a 1 channel file is loaded
			if(chans == 1) {
				jcset->setMode ( kJConvCopy);
				gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);
				gtk_widget_set_sensitive(GTK_WIDGET(combo),FALSE);
			} else if (chans == 2) {
				gtk_widget_set_sensitive(GTK_WIDGET(combo),TRUE);
			} else if (chans > 2) {
				ostringstream msg;
				msg << "Error opening file " << jcset->getIRFile() << "\n have more then 2 Channels";
				(void)gx_gui::gx_message_popup(msg.str().c_str());

				jcset->invalidate();
				// force the wave view to draw a red X
				gx_waveview_set_value(widget, NULL);
				// set label text
				gtk_label_set_text(GTK_LABEL(gx_gui::label1), "IR file empty\n");
				gtk_label_set_text(GTK_LABEL(gx_gui::label6), "empty");
				return;
			}
		} else { // end of if (file is wave audio)
			jcset->invalidate();
			gx_print_error("IR File Processing",
			               jcset->getIRFile() +
			               string(" is not a WAVE Audio file! Invalidating ... "));
			// force the wave view to draw a red X
			gx_waveview_set_value(widget, NULL);
			// set label text
			gtk_label_set_text(GTK_LABEL(gx_gui::label1), "IR file empty\n");
			gtk_label_set_text(GTK_LABEL(gx_gui::label6), "empty");
		}
	} else { // end of if (!file.empty())
		gx_print_error("IR File Processing",
		               "Filename empty, you probably need to re-pick a file");
		gx_waveview_set_value(widget, NULL);
		gtk_label_set_text(GTK_LABEL(gx_gui::label1), "No file name\n");
		gtk_label_set_text(GTK_LABEL(gx_gui::label6), "empty");
	}
}

} /* end of gx_jconv namespace */
