/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icontourfilter.h"


#include <vtkContourFilter.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPolyDataNormals.h>
#include <vtkToolkits.h>

#if defined(IVTK_4) && !defined(VTK_USE_PATENTED)
#define vtkMarchingContourFilter vtkContourFilter
#else
#include <vtkMarchingContourFilter.h>
#endif


//
//  Templates
//
#include "igenericfiltertemplate.h"


iContourFilter::iContourFilter(iViewSubject *vo) : iGenericFilter<vtkDataSetToPolyDataFilter,vtkDataSet,vtkPolyData>(vo,1,true,false)
{
	mMethod = 0;
	mCurVar = 0;

	vtkContourFilter *w0 = vtkContourFilter::New(); IERROR_ASSERT(w0);
#ifndef IVTK_4
	w0->UseScalarTreeOn();
#endif
	w0->ComputeNormalsOn();
	w0->ComputeScalarsOff();
	w0->ComputeGradientsOff();
	if(vo->IsCreatingMainPipeline())
	{
		w0->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetAbortRenderEventObserver());
	}
	else
	{
		w0->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetSlaveAbortRenderEventObserver());
	}

	vtkMarchingContourFilter *w1 = vtkMarchingContourFilter::New(); IERROR_ASSERT(w1);
#if !defined(IVTK_4) || defined(VTK_USE_PATENTED)
	w1->UseScalarTreeOn();
#endif
	w1->ComputeNormalsOn();
	w1->ComputeScalarsOff();
	w1->ComputeGradientsOff();
	if(vo->IsCreatingMainPipeline())
	{
		w1->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetAbortRenderEventObserver());
	}
	else
	{
		w1->AddObserver(vtkCommand::ProgressEvent,vo->GetViewModule()->GetSlaveAbortRenderEventObserver());
	}

	mWorkers[0] = w0;
	mWorkers[1] = w1;
}


iContourFilter::~iContourFilter()
{
	mWorkers[0]->Delete();
	mWorkers[1]->Delete();
}


void iContourFilter::SetMethod(int n)
{ 
	if(n>=0 && n<2 && n!=mMethod)
	{
		mWorkers[mMethod]->GetOutput()->Initialize();  // erase old data
		mMethod = n;
		this->Modified();
	}
}


void iContourFilter::SetCurrentVar(int n)
{ 
	if(n>=0 && n!=mCurVar)
	{
		mCurVar = n;
		this->Modified();
	}
}


float iContourFilter::GetLevel() const
{
	return iRequiredCast<vtkContourFilter>(INFO,mWorkers[0])->GetValue(0);
}


void iContourFilter::SetLevel(float v)
{
	iRequiredCast<vtkContourFilter>(INFO,mWorkers[0])->SetValue(0,v);
	iRequiredCast<vtkMarchingContourFilter>(INFO,mWorkers[1])->SetValue(0,v);
	this->Modified();
}


void iContourFilter::ProduceOutput()
{
	vtkDataSet *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	//
	//  VTK contouring filters always make the isosurface of the first component. We simple shift the data by mCurVar values
	//  so that the current value is always the 0 component.
	//
	int numComp;
	if(input->GetPointData()==0 || input->GetPointData()->GetScalars()==0 || (numComp=input->GetPointData()->GetScalars()->GetNumberOfComponents())<1 || mCurVar<0 || mCurVar>=numComp)
	{
		output->Initialize();
		return; 
    }

	vtkDataSet *newInput = input->NewInstance(); IERROR_ASSERT(newInput);
	newInput->ShallowCopy(input);

	vtkFloatArray *scalars = iRequiredCast<vtkFloatArray>(INFO,input->GetPointData()->GetScalars());
	vtkFloatArray *newScalars = vtkFloatArray::New(); IERROR_ASSERT(newScalars);
	newScalars->SetNumberOfComponents(numComp);
	newScalars->SetArray(scalars->GetPointer(mCurVar),scalars->GetSize(),1);
	newInput->GetPointData()->SetScalars(newScalars);
	newScalars->Delete();

	mWorkers[mMethod]->SetInput(newInput);
	newInput->Delete();

	mWorkers[mMethod]->Update();

	//
	//  Are there normals? Not all combinations of inputs/filters/VTK versions create those
	//
	if(mWorkers[mMethod]->GetOutput()->GetPointData()->GetNormals()==0 || mWorkers[mMethod]->GetOutput()->GetPointData()->GetNormals()->GetNumberOfTuples()==0)
	{
		vtkPolyDataNormals *nf = vtkPolyDataNormals::New(); IERROR_ASSERT(nf);
		nf->ConsistencyOn();
		nf->ComputePointNormalsOn(); 
		nf->ComputeCellNormalsOff();
		nf->SetInput(mWorkers[mMethod]->GetOutput());
		nf->Update();
		mWorkers[mMethod]->GetOutput()->GetPointData()->SetNormals(nf->GetOutput()->GetPointData()->GetNormals());
		nf->Delete();
	}

	output->ShallowCopy(mWorkers[mMethod]->GetOutput());
}

