/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iimagecomposer.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iimagecomposerwindows.h"
#include "imath.h"
#include "ishellfactory.h"
#include "istereoimagearray.h"
#include "iviewmodule.h"

//
//  Templates
//
#include "iarraytemplate.h"


//
//  Main class
//
IOBJECT_DEFINE_TYPE(iImageComposer,ImageComposer,ic,iObjectType::_Module);

IOBJECT_DEFINE_KEY(iImageComposer,BorderColor,bc,Color,1);
IOBJECT_DEFINE_KEY(iImageComposer,BackgroundWindowViewModule,bgv,OffsetInt,0);
IOBJECT_DEFINE_KEY(iImageComposer,BackgroundWindowViewModule2,bgv2,OffsetInt,0);
IOBJECT_DEFINE_KEY(iImageComposer,BackgroundWindowViewModule3,bgv3,OffsetInt,0);
IOBJECT_DEFINE_KEY(iImageComposer,BackgroundWindowWallpaperFile,bgw,String,0);
IOBJECT_DEFINE_KEY(iImageComposer,BorderWidth,bw,Int,1);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowBorderColor,fgc,Color,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowScale,fgs,Float,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowViewModule,fgv,OffsetInt,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowViewModule2,fgv2,OffsetInt,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowViewModule3,fgv3,OffsetInt,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowBorderWidth,fgw,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowPositionX,fgx,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowPositionY,fgy,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowZoomSource,fgz,Int,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowZoom4Line,fgz4,Bool,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowZoomFactor,fgzf,Float,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowZoomX,fgzx,Float,0);
IOBJECT_DEFINE_KEY(iImageComposer,ForegroundWindowZoomY,fgzy,Float,0);
IOBJECT_DEFINE_KEY(iImageComposer,ImageHeight,h,Int,1);
IOBJECT_DEFINE_KEY(iImageComposer,InnerBorder,ib,Bool,1);
IOBJECT_DEFINE_KEY(iImageComposer,NumForegroundWindows,nfg,Int,1);
IOBJECT_DEFINE_KEY(iImageComposer,NumTiles,nt,Int,2);
IOBJECT_DEFINE_KEY(iImageComposer,ScaleBackground,sb,Bool,1);
IOBJECT_DEFINE_KEY(iImageComposer,ImageWidth,w,Int,1);


iImageComposer* iImageComposer::New(iControlModule *cm)
{
	IERROR_ASSERT(cm);
	return new iImageComposer(cm);
}


iImageComposer::iImageComposer(iControlModule *cm) : iObject("ImageComposer"), mControlModule(cm)
{
	mBlockUpdate = mInComposing = false;

	mNumTilesX = mNumTilesY = 1;
	mBorderWidth = 0;
	mBorderColor = iColor(0,0,0);
	mInnerBorder = false;
	mFullWidth = mTileWidth = 640;
	mFullHeight = mTileHeight = 480;

	mBackgroundWindows.Add(iImageComposerBackgroundWindow::New(0,0,this));
}


iImageComposer::~iImageComposer()
{
	int i;

	for(i=0; i<mForegroundWindows.Size(); i++)
	{
		delete mForegroundWindows[i];
	}

	for(i=0; i<mBackgroundWindows.Size(); i++)
	{
		delete mBackgroundWindows[i];
	}
}


void iImageComposer::SetNumTiles(int nx, int ny)
{ 
	int i, j;

	if(nx <= 0) nx = mNumTilesX;
	if(ny <= 0) ny = mNumTilesY;

	if(nx==mNumTilesX && ny==mNumTilesY) return;

	//
	//  CreateBackgroundImage checks for out-of-boundary requests, so we need to have
	//  mNumTiles[X,Y] Set before creating new windows
	//
	int oldNumTilesX = mNumTilesX;
	int oldNumTilesY = mNumTilesY;
	mNumTilesX = nx; 
	mNumTilesY = ny; 

	iArray<iImageComposerBackgroundWindow*> tmp;
	for(j=0; j<mBackgroundWindows.Size(); j++) tmp.Add(mBackgroundWindows[j]);
	mBackgroundWindows.Clear();
	for(j=0; j<nx*ny; j++) mBackgroundWindows.Add(0);
	
	for(j=0; j<ny && j<oldNumTilesY; j++)
	{
		for(i=0; i<nx && i<oldNumTilesX; i++)
		{
			mBackgroundWindows[i+nx*j] = tmp[i+oldNumTilesX*j];
		}
	}

	for(j=ny; j<oldNumTilesY; j++)
	{
		for(i=0; i<oldNumTilesX; i++)
		{
			delete tmp[i+oldNumTilesX*j];
		}
	}

	for(j=0; j<ny; j++)
	{
		for(i=nx; i<oldNumTilesX; i++)
		{
			delete tmp[i+oldNumTilesX*j];
		}
	}

	for(j=oldNumTilesY; j<ny; j++)
	{
		for(i=0; i<nx; i++)
		{
			mBackgroundWindows[i+nx*j] = iImageComposerBackgroundWindow::New(i,j,this);
		}
	}

	for(j=0; j<oldNumTilesY; j++)
	{
		for(i=oldNumTilesX; i<nx; i++)
		{
			mBackgroundWindows[i+nx*j] = iImageComposerBackgroundWindow::New(i,j,this);
		}
	}

	this->UpdateSize();
	this->ClearCache();
}


void iImageComposer::AddForegroundWindow(iViewModule *v)
{
	if(v == 0) return;

	mForegroundWindows.Add(iImageComposerForegroundWindow::New(v,this));
	this->ClearCache();
}


void iImageComposer::RemoveForegroundWindow(int n)
{
	if(n<0 || n>=mForegroundWindows.Size()) return;

	delete mForegroundWindows[n];
	mForegroundWindows.Remove(n);
	this->ClearCache();
}


void iImageComposer::MoveToBack(int n)
{
	if(n<0 || n>=mForegroundWindows.MaxIndex()) return;

	int i;
	iImageComposerForegroundWindow *tmp = mForegroundWindows[n];
	for(i=n; i<mForegroundWindows.MaxIndex(); i++) mForegroundWindows[i] = mForegroundWindows[i+1];
	mForegroundWindows.Last() = tmp;
	this->ClearCache();
}


void iImageComposer::SetScaleBackground(bool s)
{
	if(s != mScaleBackground)
	{
		mScaleBackground = s;
		this->UpdateSize();
		this->ClearCache();
	}
}


void iImageComposer::SetInnerBorder(bool s)
{
	if(s != mInnerBorder)
	{
		mInnerBorder = s;
		this->UpdateSize(); 
		this->ClearCache();
	}
}


void iImageComposer::SetBorderWidth(int s) 
{ 
	if(s >= 0) 
	{ 
		mBorderWidth = s; 
		this->UpdateSize(); 
		this->ClearCache();
	} 
}


void iImageComposer::SetBorderColor(const iColor &c) 
{ 
	if(c != mBorderColor) 
	{ 
		mBorderColor = c; 
		this->ClearCache();
	} 
}


void iImageComposer::Update()
{
	this->UpdateWindowList();
	this->UpdateSize();
}


void iImageComposer::UpdateSize()
{
	if(mBlockUpdate) return;
	
	int i;
	//
	//  Compute real size
	//
	mTileWidth = mTileHeight = 0;
	for(i=0; i<mBackgroundWindows.Size(); i++)
	{
		if(mTileWidth < mBackgroundWindows[i]->GetWindowWidth()) mTileWidth = mBackgroundWindows[i]->GetWindowWidth();
		if(mTileHeight < mBackgroundWindows[i]->GetWindowHeight()) mTileHeight = mBackgroundWindows[i]->GetWindowHeight();
	}
	if(mTileWidth==0 || mTileHeight==0) 
	{
		for(i=0; i<this->GetControlModule()->GetNumberOfViewModules(); i++)
		{
			if(mTileWidth < this->GetControlModule()->GetViewModule(i)->GetThisImageWidth()) mTileWidth = this->GetControlModule()->GetViewModule(i)->GetThisImageWidth(); 
			if(mTileHeight < this->GetControlModule()->GetViewModule(i)->GetThisImageHeight()) mTileHeight = this->GetControlModule()->GetViewModule(i)->GetThisImageHeight();
		}
	}

	mFullWidth = mNumTilesX*mTileWidth + 2*mBorderWidth;
	mFullHeight = mNumTilesY*mTileHeight + 2*mBorderWidth;
	if(mInnerBorder)
	{
		mFullWidth += (mNumTilesX-1)*mBorderWidth;
		mFullHeight += (mNumTilesY-1)*mBorderWidth;
	}

	//
	//  Make sure no foreground window moves out of image area
	//
	for(i=0; i<mForegroundWindows.Size(); i++)
	{
		mForegroundWindows[i]->CorrectPosition();
	}
}


void iImageComposer::UpdateWindowList()
{
	int i;

	for(i=0; i<mBackgroundWindows.Size(); i++) mBackgroundWindows[i]->UpdateWindow();
	for(i=0; i<mForegroundWindows.Size(); i++) 
	{
		mForegroundWindows[i]->UpdateWindow();
		if(mForegroundWindows[i]->GetViewModule() == 0) this->RemoveForegroundWindow(i);
	}
}


iImageComposerBackgroundWindow* iImageComposer::GetBackgroundWindow(int i) const
{
	if(i>=0 && i<mBackgroundWindows.Size()) return mBackgroundWindows[i]; else return 0;
}


iImageComposerForegroundWindow* iImageComposer::GetForegroundWindow(int i) const
{
	if(i>=0 && i<mForegroundWindows.Size()) return mForegroundWindows[i]; else return 0;
}


void iImageComposer::PackStateBody(iString &s) const
{
	int i, iv[2];

	this->PackValue(s,KeyScaleBackground(),mScaleBackground);
	this->PackValue(s,KeyInnerBorder(),mInnerBorder);
	this->PackValue(s,KeyBorderColor(),mBorderColor);
	this->PackValue(s,KeyBorderWidth(),mBorderWidth);
	this->PackValue(s,KeyImageWidth(),mFullWidth);
	this->PackValue(s,KeyImageHeight(),mFullHeight);
	
	iv[0] = mNumTilesX; iv[1] = mNumTilesY;
	this->PackValue(s,KeyNumTiles(),iv,2);
	this->PackValue(s,KeyNumForegroundWindows(),mForegroundWindows.Size());

	int *itmp = new int[mBackgroundWindows.Size()]; IERROR_ASSERT(itmp);
	for(i=0; i<mBackgroundWindows.Size(); i++) itmp[i] = mBackgroundWindows[i]->GetWindowNumber(0);
	this->PackValue(s,KeyBackgroundWindowViewModule(),itmp,mBackgroundWindows.Size());
	for(i=0; i<mBackgroundWindows.Size(); i++) itmp[i] = mBackgroundWindows[i]->GetWindowNumber(1);
	this->PackValue(s,KeyBackgroundWindowViewModule2(),itmp,mBackgroundWindows.Size());
	for(i=0; i<mBackgroundWindows.Size(); i++) itmp[i] = mBackgroundWindows[i]->GetWindowNumber(2);
	this->PackValue(s,KeyBackgroundWindowViewModule3(),itmp,mBackgroundWindows.Size());

	iString *stmp = new iString[mBackgroundWindows.Size()]; IERROR_ASSERT(stmp);
	for(i=0; i<mBackgroundWindows.Size(); i++) stmp[i] = mBackgroundWindows[i]->GetWallpaperFile();
	this->PackValue(s,KeyBackgroundWindowWallpaperFile(),stmp,mBackgroundWindows.Size());
	delete [] stmp;

	//
	//  We need the keys to be present even if there are no windows, so that QueryValue functions
	//  do not fail
	//
	int n = mForegroundWindows.Size();
	if(n == 0) n++;

	if(n > mBackgroundWindows.Size())
	{
		delete [] itmp;
		itmp = new int[n]; IERROR_ASSERT(itmp);
	}
	float *ftmp = new float[n]; IERROR_ASSERT(ftmp);
	bool *btmp = new bool[n]; IERROR_ASSERT(btmp);
	if(mForegroundWindows.Size() == 0)
	{
		itmp[0] = 0;
		ftmp[0] = 1.0;
		btmp[0] = true;
	}

	for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetWindowNumber(0);
	this->PackValue(s,KeyForegroundWindowViewModule(),itmp,n);
	for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetWindowNumber(1);
	this->PackValue(s,KeyForegroundWindowViewModule2(),itmp,n);
	for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetWindowNumber(2);
	this->PackValue(s,KeyForegroundWindowViewModule3(),itmp,n);

	for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetPositionX();
	this->PackValue(s,KeyForegroundWindowPositionX(),itmp,n);
	for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetPositionY();
	this->PackValue(s,KeyForegroundWindowPositionY(),itmp,n);

	for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetBorderWidth();
	this->PackValue(s,KeyForegroundWindowBorderWidth(),itmp,n);

	iColor *ctmp = new iColor[n]; IERROR_ASSERT(ctmp);
	for(i=0; i<mForegroundWindows.Size(); i++) ctmp[i] = mForegroundWindows[i]->GetBorderColor();
	this->PackValue(s,KeyForegroundWindowBorderColor(),ctmp,n);
	delete [] ctmp;

	for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetScale();
	this->PackValue(s,KeyForegroundWindowScale(),ftmp,n);

	for(i=0; i<mForegroundWindows.Size(); i++)
	{
		if(mForegroundWindows[i]->GetZoomSource() == 0)
		{
			itmp[i] = 0;
		}
		else
		{
			itmp[i] = mForegroundWindows[i]->GetZoomSource()->GetId();
		}
	}
	this->PackValue(s,KeyForegroundWindowZoomSource(),itmp,n);

	for(i=0; i<mForegroundWindows.Size(); i++) btmp[i] = (mForegroundWindows[i]->GetZoomFlag() == iImageComposerForegroundWindow::_4Lines);
	this->PackValue(s,KeyForegroundWindowZoom4Line(),btmp,n);

	for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetZoomX();
	this->PackValue(s,KeyForegroundWindowZoomX(),ftmp,n);

	for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetZoomY();
	this->PackValue(s,KeyForegroundWindowZoomY(),ftmp,n);

	for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetZoomFactor();
	this->PackValue(s,KeyForegroundWindowZoomFactor(),ftmp,n);

	delete [] btmp;
	delete [] ftmp;
	delete [] itmp;
}


void iImageComposer::UnPackStateBody(const iString &s)
{
	int i, iv[2]; bool b; iColor c;
	
	mBlockUpdate = true;

	if(this->UnPackValue(s,KeyScaleBackground(),b)) this->SetScaleBackground(b);
	if(this->UnPackValue(s,KeyInnerBorder(),b)) this->SetInnerBorder(b);
	if(this->UnPackValue(s,KeyBorderColor(),c)) this->SetBorderColor(c);
	if(this->UnPackValue(s,KeyBorderWidth(),i)) this->SetBorderWidth(i);
	
	iv[0] = mNumTilesX; iv[1] = mNumTilesY;
	if(this->UnPackValue(s,KeyNumTiles(),iv,2)) this->SetNumTiles(iv[0],iv[1]);

	if(this->UnPackValue(s,KeyNumForegroundWindows(),i))
	{
		while(i < mForegroundWindows.Size()) this->RemoveForegroundWindow(i);
		while(i > mForegroundWindows.Size()) this->AddForegroundWindow(this->GetControlModule()->GetViewModule());
	}

	int *itmp = new int[mBackgroundWindows.Size()]; IERROR_ASSERT(itmp);
	this->UnPackingHelper(mBackgroundWindows.Size(),0,(iImageComposerWindow**)mBackgroundWindows.Data(),itmp,KeyBackgroundWindowViewModule(),s);
	this->UnPackingHelper(mBackgroundWindows.Size(),1,(iImageComposerWindow**)mBackgroundWindows.Data(),itmp,KeyBackgroundWindowViewModule2(),s);
	this->UnPackingHelper(mBackgroundWindows.Size(),2,(iImageComposerWindow**)mBackgroundWindows.Data(),itmp,KeyBackgroundWindowViewModule3(),s);

	iString *stmp = new iString[mBackgroundWindows.Size()]; IERROR_ASSERT(stmp);
	for(i=0; i<mBackgroundWindows.Size(); i++) stmp[i] = mBackgroundWindows[i]->GetWallpaperFile();
	if(this->UnPackValue(s,KeyBackgroundWindowWallpaperFile(),stmp,mBackgroundWindows.Size()))
	{
		for(i=0; i<mBackgroundWindows.Size(); i++) mBackgroundWindows[i]->LoadWallpaperImage(stmp[i]);
	}
	delete [] stmp;

	//
	//  All background windows are loaded. We can update  now so that
	//  position setting works.
	//
	mBlockUpdate = false;
	this->Update();

	if(mForegroundWindows.Size() > 0)
	{
		if(mForegroundWindows.Size() > mBackgroundWindows.Size())
		{
			delete [] itmp;
			itmp = new int[mForegroundWindows.Size()]; IERROR_ASSERT(itmp);
		}
		float *ftmp = new float[mForegroundWindows.Size()]; IERROR_ASSERT(ftmp);
		bool *btmp = new bool[mForegroundWindows.Size()]; IERROR_ASSERT(btmp);
		
		this->UnPackingHelper(mForegroundWindows.Size(),0,(iImageComposerWindow**)mForegroundWindows.Data(),itmp,KeyForegroundWindowViewModule(),s);
		this->UnPackingHelper(mForegroundWindows.Size(),1,(iImageComposerWindow**)mForegroundWindows.Data(),itmp,KeyForegroundWindowViewModule2(),s);
		this->UnPackingHelper(mForegroundWindows.Size(),2,(iImageComposerWindow**)mForegroundWindows.Data(),itmp,KeyForegroundWindowViewModule3(),s);

		for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetBorderWidth();
		if(this->UnPackValue(s,KeyForegroundWindowBorderWidth(),itmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetBorderWidth(itmp[i]);
		}

		iColor *ctmp = new iColor[mForegroundWindows.Size()]; IERROR_ASSERT(ctmp);
		for(i=0; i<mForegroundWindows.Size(); i++) ctmp[i] = mForegroundWindows[i]->GetBorderColor();
		if(this->UnPackValue(s,KeyForegroundWindowBorderColor(),ctmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetBorderColor(ctmp[i]);
		}
		delete [] ctmp;

		for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetScale();
		if(this->UnPackValue(s,KeyForegroundWindowScale(),ftmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetScale(ftmp[i]);
		}

		for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetPositionX();
		if(this->UnPackValue(s,KeyForegroundWindowPositionX(),itmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetPositionX(itmp[i]);
		}
		
		for(i=0; i<mForegroundWindows.Size(); i++) itmp[i] = mForegroundWindows[i]->GetPositionY();
		if(this->UnPackValue(s,KeyForegroundWindowPositionY(),itmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetPositionY(itmp[i]);
		}

		//
		//  As an optimization, load positions first, then the source, and then the height - loading the height
		//  will adjust the position, but only once.
		//
		for(i=0; i<mForegroundWindows.Size(); i++) btmp[i] = (mForegroundWindows[i]->GetZoomFlag() == iImageComposerForegroundWindow::_4Lines);
		if(this->UnPackValue(s,KeyForegroundWindowZoom4Line(),btmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetZoomFlag(btmp[i]?iImageComposerForegroundWindow::_4Lines:1);
		}

		for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetZoomX();
		if(this->UnPackValue(s,KeyForegroundWindowZoomX(),ftmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetZoomPosition(ftmp[i],mForegroundWindows[i]->GetZoomY());
		}

		for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetZoomY();
		if(this->UnPackValue(s,KeyForegroundWindowZoomY(),ftmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetZoomPosition(mForegroundWindows[i]->GetZoomX(),ftmp[i]);
		}

		for(i=0; i<mForegroundWindows.Size(); i++)
		{
			if(mForegroundWindows[i]->GetZoomSource() == 0)
			{
				itmp[i] = 0;
			}
			else
			{
				itmp[i] = mForegroundWindows[i]->GetZoomSource()->GetId();
			}
		}
		if(this->UnPackValue(s,KeyForegroundWindowZoomSource(),itmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++)
			{
				if(itmp[i] < 0)
				{
					mForegroundWindows[i]->SetZoomSource(this->GetBackgroundWindow(-itmp[i]-1));
				}
				else if(itmp[i] > 0)
				{
					mForegroundWindows[i]->SetZoomSource(this->GetForegroundWindow(itmp[i]-1));
				}
				else
				{
					mForegroundWindows[i]->SetZoomSource(0);
				}
			}
		}

		for(i=0; i<mForegroundWindows.Size(); i++) ftmp[i] = mForegroundWindows[i]->GetZoomFactor();
		if(this->UnPackValue(s,KeyForegroundWindowZoomFactor(),ftmp,mForegroundWindows.Size()))
		{
			for(i=0; i<mForegroundWindows.Size(); i++) mForegroundWindows[i]->SetZoomFactor(ftmp[i]);
		}

		delete [] ftmp;
		delete [] btmp;
	}

	delete [] itmp;
}


void iImageComposer::UnPackingHelper(int n, int j, iImageComposerWindow **win, int *itmp, const iObjectKey &key, const iString &s)
{
	int i, k;

	for(i=0; i<n; i++) itmp[i] = win[i]->GetWindowNumber(j);
	if(this->UnPackValue(s,key,itmp,n)) 
	{
		for(i=0; i<n; i++) if(itmp[i]>=0 && itmp[i]!=win[i]->GetWindowNumber(j))
		{
			for(k=0; k<this->GetControlModule()->GetNumberOfViewModules() && itmp[i]!=this->GetControlModule()->GetViewModule(k)->GetWindowNumber(); k++);
			if(k < this->GetControlModule()->GetNumberOfViewModules()) win[i]->SetViewModule(this->GetControlModule()->GetViewModule(k),j);
		}
	}
}


bool iImageComposer::IsActive() 
{
	int i;
	
	//
	//  Is there an active background window?
	//
	for(i=0; i<mBackgroundWindows.Size(); i++)
	{
		if(!mBackgroundWindows[i]->IsEmpty()) return true;
	}

	//
	//  Is there an active foreground window?
	//
	if(mForegroundWindows.Size() > 0) return true;

	//
	// At the very least, is there a border?
	//
	return mBorderWidth > 0;
}

										  
void iImageComposer::Compose(iViewModule *vm, iStereoImageArray &images) 
{ 
	this->GetErrorStatus()->Clear();

	if(vm == 0)
	{
		this->GetErrorStatus()->Set("No ViewModule is specified");
		return;
	}

	mInComposing = true;

	//
	//  We received a request to compose the images and put the data into images.
	//  First, make sure that Composer is up to date
	//
	this->UpdateWindowList();

	iStereoImage outIm;
	char rgb[3];
	int i, j, k, xoff;
	unsigned char *dPtr, *dPtr1;
	
	outIm.Scale(mFullWidth,mFullHeight);
	dPtr = outIm.LeftEye().DataPointer();
	int d = outIm.Depth();
	bool work = false;

	if(mBorderWidth > 0)
	{
		work = true;
		//
		//  Create the border
		//
		mBorderColor.GetRGB(rgb);
		//
		//  Vertical lines
		//
		for(j=0; j<mFullHeight; j++)  // sort of a waste, but it is much simpler this way
		{
			for(i=0; i<mBorderWidth; i++)
			{
				dPtr1 = dPtr + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
			for(i=mFullWidth-mBorderWidth; i<mFullWidth; i++)
			{
				dPtr1 = dPtr + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}

			if(mInnerBorder)
			{
				for(k=1; k<mNumTilesX; k++)
				{
					xoff = k*(mBorderWidth+mTileWidth);
					for(i=xoff; i<xoff+mBorderWidth; i++)
					{
						dPtr1 = dPtr + d*(i+mFullWidth*j);
						for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
		}
		//
		//  Horizontal lines
		//
		for(j=0; j<mBorderWidth; j++)  // sort of a waste, but it is much simpler this way
		{
			for(i=0; i<mFullWidth; i++)
			{
				dPtr1 = dPtr + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
		}

		for(j=mFullHeight-mBorderWidth; j<mFullHeight; j++)
		{
			for(i=0; i<mFullWidth; i++)
			{
				dPtr1 = dPtr + d*(i+mFullWidth*j);
				for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
			}
		}

		if(mInnerBorder)
		{
			for(k=1; k<mNumTilesY; k++)
			{
				xoff = k*(mBorderWidth+mTileHeight);
				for(j=xoff; j<xoff+mBorderWidth; j++)
				{
					for(i=0; i<mFullWidth; i++)
					{
						dPtr1 = dPtr + d*(i+mFullWidth*j);
						for(k=0; k<d; k++) dPtr1[k] = rgb[k];  // fastest assignment!!
					}
				}
			}
		}
	}

	//
	//  Background windows 
	//
	for(i=0; i<mBackgroundWindows.Size(); i++)
	{
		if(!mBackgroundWindows[i]->IsEmpty()) work = true;
		mBackgroundWindows[i]->Draw(outIm);
	}

	//
	//  Foreground windows 
	//
	for(i=0; i<mForegroundWindows.Size(); i++)
	{
		if(!mForegroundWindows[i]->IsEmpty()) work = true;
		mForegroundWindows[i]->Draw(outIm);
	}

	if(work)
	{
		images.Clear();
		images.Add(outIm);
	}
	else
	{
		this->GetErrorStatus()->Monitor(vm->GetErrorStatus());
		vm->RenderImages(images);
	}

	mInComposing = false;
}

