/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iinteractoreventrecorder.h"


#include "ierror.h"

#include <vtkCallbackCommand.h>
#include <vtkRenderWindowInteractor.h>


iInteractorEventRecorder::iInteractorEventRecorder()
{
	mStopped = false;
}


iInteractorEventRecorder::~iInteractorEventRecorder()
{
}


void iInteractorEventRecorder::SetFileName(const char *s)
{
	if(this->InputStream != 0)
    {
		delete this->InputStream;
		this->InputStream = 0;
    }
	
	if(this->OutputStream != 0)
    {
		delete this->OutputStream;
		this->OutputStream = 0;
    }

	vtkInteractorEventRecorder::SetFileName(s);

	this->OutputStream = new fstream(this->FileName,ios::in|ios::out); IERROR_ASSERT(this->OutputStream);
	if(this->OutputStream->fail())
	{
		vtkErrorMacro(<< "Unable to open file: "<< this->FileName);
		delete this->OutputStream;
		return;
	}
}


void iInteractorEventRecorder::Record()
{
	if(this->State == vtkInteractorEventRecorder::Start)
    {
		if(!this->OutputStream ) //need to open file
		{
			this->OutputStream = new fstream(this->FileName, ios::in|ios::out); IERROR_ASSERT(this->OutputStream);
			if(this->OutputStream->fail())
			{
				vtkErrorMacro(<< "Unable to open file: "<< this->FileName);
				delete this->OutputStream;
				return;
			}
		}

		this->OutputStream->clear();

		if((int)this->OutputStream->tellp() == 0)
		{
			*this->OutputStream << "# StreamVersion " << vtkInteractorEventRecorder::StreamVersion << "\n";
		}
		
		vtkDebugMacro(<<"Recording");
		this->State = vtkInteractorEventRecorder::Recording;
    }
}


void iInteractorEventRecorder::Play()
{
	//
	//  Copied from vtkInteractorEventRecorder::Play() with minor modifications
	//
	if(this->State == vtkInteractorEventRecorder::Start)
    {
		if(this->ReadFromInputString)
		{
			vtkDebugMacro(<< "Reading from InputString");
			int len;
			if ( this->InputString == 0 || (len = strlen(this->InputString) <= 0) )
			{
				vtkErrorMacro(<< "No input string specified");
				return;
			}
			this->InputStream = new istrstream(this->InputString, len); IERROR_ASSERT(this->InputStream);
			if (this->InputStream->fail())
			{
				vtkErrorMacro(<< "Unable to read from string");
				delete this->InputStream;
				return;
			}
		}
		else
		{
			if(!this->OutputStream) //need to open file
			{
				this->OutputStream = new fstream(this->FileName,ios::in|ios::out); IERROR_ASSERT(this->OutputStream);
				if (this->OutputStream->fail())
				{
					vtkErrorMacro(<< "Unable to open file: "<< this->FileName);
					delete this->OutputStream;
					return;
				}
//				if(this->OutputStream && this->OutputStream->tellp()>0)
//				{
//					this->OutputStream->Seekg(this->OutputStream->tellp());
//				}
			}
		}
		
		vtkDebugMacro(<<"Playing");
		this->State = vtkInteractorEventRecorder::Playing;
		
		// Read events and invoke them on the object in question
		char event[128], keySym[64], buffer[512];
		int pos[2], ctrlKey, shiftKey, keyCode, repeatCount;
		float stream_version = 0.0, tempf;
		
		while(!mStopped && !this->OutputStream->eof())
		{
			this->OutputStream->width(256);
			*static_cast<fstream *>(this->OutputStream) >> event;
			
			// Quick skip comment
			if (*event == '#')
			{
				static_cast<fstream *>(this->OutputStream)->getline(buffer, 512);
				
				// Parse the StreamVersion (not using >> since comment could be empty)
				// Expecting: # StreamVersion x.y
				
				if (strlen(buffer) > 15 && !strncmp(buffer, " StreamVersion ", 15))
				{
					int res = sscanf(buffer + 15, "%f", &tempf);
					if (res && res != EOF)
					{
						stream_version = tempf;
					}
				}
			}
			else
			{
				unsigned long ievent = vtkCommand::GetEventIdFromString(event);
				if (ievent == vtkCommand::NoEvent)
				{
					static_cast<fstream *>(this->OutputStream)->ignore(512, '\n');
				}
				else
				{
					if (stream_version >= 1.1)
					{
						// We could grab the time info here
					}
					*static_cast<fstream *>(this->OutputStream) >> pos[0];
					*static_cast<fstream *>(this->OutputStream) >> pos[1];
					*static_cast<fstream *>(this->OutputStream) >> ctrlKey;
					*static_cast<fstream *>(this->OutputStream) >> shiftKey;
					*static_cast<fstream *>(this->OutputStream) >> keyCode;
					*static_cast<fstream *>(this->OutputStream) >> repeatCount;
					*static_cast<fstream *>(this->OutputStream) >> keySym;
					
					this->Interactor->SetEventPosition(pos);
					this->Interactor->SetControlKey(ctrlKey);
					this->Interactor->SetShiftKey(shiftKey);
					this->Interactor->SetKeyCode(keyCode);
					this->Interactor->SetRepeatCount(repeatCount);
					this->Interactor->SetKeySym(keySym);
					
					this->Interactor->InvokeEvent(ievent, 0);
				}
			}
		}
    }
	
	mStopped = false;
	this->State = vtkInteractorEventRecorder::Start;
	//
	//  Clear current button press events
	//
	this->Interactor->SetControlKey(0);
	this->Interactor->SetShiftKey(0);
	this->Interactor->SetKeyCode(0);
	this->Interactor->SetRepeatCount(0);
	this->Interactor->SetKeySym(0);

	this->Interactor->InvokeEvent(vtkCommand::GetEventIdFromString("LeftButtonReleaseEvent"),0);
	this->Interactor->InvokeEvent(vtkCommand::GetEventIdFromString("MiddleButtonReleaseEvent"),0);
	this->Interactor->InvokeEvent(vtkCommand::GetEventIdFromString("RightButtonReleaseEvent"),0);
}


void iInteractorEventRecorder::Stop()
{
	if(this->State == vtkInteractorEventRecorder::Playing) 
	{
		mStopped = true;
	}
	this->State = vtkInteractorEventRecorder::Start;
	this->Modified();
}


void iInteractorEventRecorder::Rewind()
{
	if(this->OutputStream != 0) //need to already have an open file
	{
		this->OutputStream->clear();
		static_cast<fstream *>(this->OutputStream)->seekp(0);
		static_cast<fstream *>(this->OutputStream)->seekg(0);
	}
}


bool iInteractorEventRecorder::IsPlaying() const
{
	return this->State == vtkInteractorEventRecorder::Playing;
}


bool iInteractorEventRecorder::IsRecording() const
{
	return this->State == vtkInteractorEventRecorder::Recording;
}


bool iInteractorEventRecorder::IsStopped() const
{
	return this->State == vtkInteractorEventRecorder::Start;
}


bool iInteractorEventRecorder::IsAtEnd() const
{
	if(this->OutputStream!=0 && !this->OutputStream->eof()) return false; else return true; 
}

