/*
 * Copyright (c) 2008, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.cwt.font.truetype;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.intarsys.cwt.font.FontStyle;
import de.intarsys.tools.locator.ILocator;
import de.intarsys.tools.randomaccess.IRandomAccess;
import de.intarsys.tools.randomaccess.RandomAccessByteArray;
import de.intarsys.tools.stream.StreamTools;
import de.intarsys.tools.string.StringTools;

/**
 * This class represents a true type font. Currently only single font files are
 * supported. The bytes defining the font are read completely and stored for
 * later use. This class is under construction, it's use is for reading true
 * types and make some operations on it rather than creating from the scratch.
 */
public class TTFont {
	public static final byte[] TABLE_POST = "post".getBytes();//$NON-NLS-1$

	public static final byte[] TABLE_PREP = "prep".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_LOCA = "loca".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_GLYF = "glyf".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_FGPM = "fpgm".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_CVT = "cvt ".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_OS2 = "OS/2".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_NAME = "name".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_MAXP = "maxp".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_HMTX = "hmtx".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_HHEA = "hhea".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_HEAD = "head".getBytes(); //$NON-NLS-1$

	public static final byte[] TABLE_CMAP = "cmap".getBytes(); //$NON-NLS-1$

	public static final int ARG_1_AND_2_ARE_WORDS = 1;

	public static final int WE_HAVE_A_SCALE = 8;

	public static final int MORE_COMPONENTS = 32;

	public static final int WE_HAVE_AN_X_AND_Y_SCALE = 64;

	public static final int WE_HAVE_A_TWO_BY_TWO = 128;

	public static byte[][] SubsetTables = { TABLE_CMAP, TABLE_HEAD, TABLE_HHEA,
			TABLE_HMTX, TABLE_MAXP, TABLE_NAME, TABLE_OS2, TABLE_CVT,
			TABLE_FGPM, TABLE_GLYF, TABLE_LOCA, TABLE_PREP };

	public static TTFont createFromLocator(ILocator locator) throws IOException {
		TTFont result = new TTFont();
		result.setLocator(locator);
		result.initializeFromLocator();
		return result;
	}

	/** objectified versions of the "cmap" tables and subtables */
	private Map cmaps;

	private ILocator locator;

	// Font names in Postscript notation
	private String fontFamilyName = null;

	private String psName = null;

	/** "objectified" version of the "head" table */
	private TTFontHeader fontHeader;

	/** objectified version of the "hhea" table */
	private TTHorizontalHeader horizontalHeader;

	/** objectified version of the "os/2" table */
	private TTMetrics metrics;

	/** objectified version of the "naming" table */
	private TTNaming naming;

	/** objectified versions of the "post" table */
	private TTPostScriptInformation postScriptInformation;

	/** objectified version of the "hmtx" table */
	private int[] glyphWidths;

	/** the parsed table directory information */
	private TTTable[] tables;

	/** The style of this font */
	private FontStyle fontStyle = FontStyle.REGULAR;

	/**
	 * Create an empty true type font.
	 */
	protected TTFont() {
		super();
	}

	protected Set addCompositeGlyphs(IRandomAccess glyfRandom, int[] locations,
			Set glyphs) throws IOException, TrueTypeException {
		glyphs.add(new Integer(0));
		Set allGlyphs = new HashSet();
		allGlyphs.addAll(glyphs);
		for (Iterator i = glyphs.iterator(); i.hasNext();) {
			int codePoint = ((Integer) i.next()).intValue();
			addCompositeGlyphs(glyfRandom, locations, allGlyphs, codePoint);
		}
		return allGlyphs;
	}

	protected void addCompositeGlyphs(IRandomAccess random, int[] locations,
			Set glyphs, int codePoint) throws IOException, TrueTypeException {
		if (locations[codePoint] == locations[codePoint + 1]) {
			return;
		}
		random.seek(locations[codePoint]);
		TTFontParser parser = new TTFontParser();
		int numContours = parser.readShort(random);
		if (numContours >= 0) {
			return;
		}
		random.seekBy(8);

		for (;;) {
			int flags = parser.readUShort(random);
			int codePointRef = parser.readUShort(random);
			glyphs.add(new Integer(codePointRef));

			if ((flags & MORE_COMPONENTS) == 0) {
				return;
			}

			int skip;

			if ((flags & ARG_1_AND_2_ARE_WORDS) != 0) {
				skip = 4;
			} else {
				skip = 2;
			}

			if ((flags & WE_HAVE_A_SCALE) != 0) {
				skip += 2;
			} else if ((flags & WE_HAVE_AN_X_AND_Y_SCALE) != 0) {
				skip += 4;
			}

			if ((flags & WE_HAVE_A_TWO_BY_TWO) != 0) {
				skip += 8;
			}
			random.seekBy(skip);
		}
	}

	protected TTFont copySubset() {
		TTFont resultFont = new TTFont();
		List newTables = new ArrayList();

		for (int i = 0; i < SubsetTables.length; i++) {
			TTTable table = getTable(SubsetTables[i]);

			if (table != null) {
				newTables.add(table);
			}
		}

		resultFont.setTables((TTTable[]) newTables.toArray(new TTTable[0]));

		return resultFont;
	}

	protected void createGlyphTable(TTTable loca, TTTable glyf,
			IRandomAccess glyfRandom, int[] oldLocations, Set glyphs)
			throws IOException, TrueTypeException {
		int newLength = 0;

		for (Iterator i = glyphs.iterator(); i.hasNext();) {
			int codePoint = ((Integer) i.next()).intValue();

			if ((codePoint + 1) >= oldLocations.length) {
				continue;
			}

			newLength += (oldLocations[codePoint + 1] - oldLocations[codePoint]);
		}

		newLength = (newLength + 3) & (~3);

		int[] newLocations = new int[oldLocations.length];
		byte[] newGlyfData = new byte[newLength];
		int ptr = 0;

		for (int i = 0; i < oldLocations.length; i++) {
			newLocations[i] = ptr;

			if (glyphs.contains(new Integer(i))) {
				int glyfstart = oldLocations[i];
				int glyflength = oldLocations[i + 1] - glyfstart;

				if (glyflength > 0) {
					glyfRandom.seek(glyfstart);
					glyfRandom.read(newGlyfData, ptr, glyflength);
					ptr += glyflength;
				}
			}
		}

		RandomAccessByteArray random = new RandomAccessByteArray(null);
		TTFontSerializer serializer = new TTFontSerializer();
		serializer.write_loca(random, newLocations, getFontHeader()
				.isShortLocationFormat());
		loca.setBytes(random.toByteArray());
		glyf.setBytes(newGlyfData);
	}

	public TTFont createSubset(Set glyphs) throws IOException,
			TrueTypeException {
		TTTable loca = getTable(TABLE_LOCA);
		int[] locations = new TTFontParser().parseTable_loca(loca,
				getFontHeader().isShortLocationFormat());
		TTTable glyf = getTable(TABLE_GLYF);
		IRandomAccess glyfRandom = glyf.getRandomAccess();
		try {
			//
			TTFont result = copySubset();
			Set compositeGlyphs = result.addCompositeGlyphs(glyfRandom,
					locations, glyphs);
			result.createGlyphTable(loca, glyf, glyfRandom, locations,
					compositeGlyphs);
			return result;
		} finally {
			StreamTools.close(glyfRandom);
		}
	}

	public Map getCMaps() throws TrueTypeException {
		if (cmaps == null) {
			TTFontParser parser = new TTFontParser();

			try {
				cmaps = parser.parseTable_cmap(getTable(TABLE_CMAP));
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}

		return cmaps;
	}

	public Map getCMapsAt(int platformID, int platformSpecificID)
			throws TrueTypeException {
		String key = StringTools.EMPTY + platformID + ":" + platformSpecificID; //$NON-NLS-1$
		Object result = getCMaps().get(key);

		if (result instanceof TTTable) {
			// not yet parsed
			TTFontParser parser = new TTFontParser();

			try {
				Map submap = parser.parseTable_cmap_subtable((TTTable) result);
				getCMaps().put(key, submap);
				result = submap;
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}

		return (Map) result;
	}

	public String getFontFamilyName() {
		return fontFamilyName;
	}

	public TTFontHeader getFontHeader() throws TrueTypeException {
		if (fontHeader == null) {
			TTFontParser parser = new TTFontParser();

			try {
				fontHeader = parser.parseTable_head(getTable(TABLE_HEAD));
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}

		return fontHeader;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.intarsys.font.IFont#getFontName()
	 */
	public String getFontName() {
		return getPsName();
	}

	public FontStyle getFontStyle() {
		return fontStyle;
	}

	public int getGlyphWidth(int codePoint) throws TrueTypeException {
		if (codePoint < getGlyphWidths().length) {
			return getGlyphWidths()[codePoint];
		}
		return getGlyphWidths()[getGlyphWidths().length - 1];
	}

	protected int[] getGlyphWidths() throws TrueTypeException {
		if (glyphWidths == null) {
			TTFontParser parser = new TTFontParser();

			try {
				glyphWidths = parser.parseTable_hmtx(getTable(TABLE_HMTX),
						getHorizontalHeader().getNumberOfHMetrics());
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}

		return glyphWidths;
	}

	public TTHorizontalHeader getHorizontalHeader() throws TrueTypeException {
		if (horizontalHeader == null) {
			TTFontParser parser = new TTFontParser();

			try {
				horizontalHeader = parser.parseTable_hhea(getTable(TABLE_HHEA));
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}

		return horizontalHeader;
	}

	public ILocator getLocator() {
		return locator;
	}

	public TTMetrics getMetrics() throws TrueTypeException {
		if (metrics == null) {
			TTFontParser parser = new TTFontParser();

			try {
				metrics = parser.parseTable_os2(getTable(TABLE_OS2));
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}

		return metrics;
	}

	public TTNaming getNaming() throws TrueTypeException {
		if (naming == null) {
			TTFontParser parser = new TTFontParser();

			try {
				TTTable table = getTable(TABLE_NAME);
				if (table != null) {
					naming = parser.parseTable_name(table);
				}
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}
		return naming;
	}

	public TTPostScriptInformation getPostScriptInformation()
			throws TrueTypeException {
		if (postScriptInformation == null) {
			TTFontParser parser = new TTFontParser();

			try {
				postScriptInformation = parser
						.parseTable_post(getTable(TABLE_POST));
			} catch (IOException e) {
				throw new TrueTypeException(e.getMessage());
			}
		}

		return postScriptInformation;
	}

	public String getPsName() {
		return psName;
	}

	public TTTable getTable(byte[] name) {
		for (int i = 0; i < getTables().length; i++) {
			TTTable current = tables[i];

			if (Arrays.equals(current.getName(), name)) {
				return current;
			}
		}

		return null;
	}

	public TTTable[] getTables() {
		return tables;
	}

	protected void initializeFromLocator() throws IOException {
		IRandomAccess random = null;
		try {
			random = getLocator().getRandomAccess();
			TTFontParser parser = new TTFontParser();
			parser.parseTables(this);
			try {
				setFontName(this);
			} catch (TrueTypeException e) {
				throw new IOException(e.getMessage());
			}

		} finally {
			StreamTools.close(random);
		}
	}

	protected void setFontFamilyName(String string) {
		fontFamilyName = string;
	}

	protected void setFontName(TTFont font) throws TrueTypeException {
		TTNaming tempNaming = font.getNaming();
		if (tempNaming != null) {
			font.setFontFamilyName(tempNaming
					.getValue(ITTNamingIDs.FontFamilyName));
			String styleName = tempNaming
					.getValue(ITTNamingIDs.FontSubfamilyName);
			font.setFontStyle(FontStyle.getFontStyle(styleName));
			font.setPsName(tempNaming.getValue(ITTNamingIDs.PSName));
			if (font.getPsName() == null) {
				// Here we should use operating system specific information
				// which can't be done by Java. So we use our own
				// implementation and hope we it works.
				// todo postscript name
				font.setPsName(font.getFontFamilyName() + "-" //$NON-NLS-1$
						+ font.getFontStyle().getId());
			}
		}
	}

	protected void setFontStyle(FontStyle fontStyle) {
		this.fontStyle = fontStyle;
	}

	protected void setLocator(ILocator locator) {
		this.locator = locator;
	}

	public void setPsName(String string) {
		psName = string;
	}

	protected void setTables(TTTable[] tables) {
		this.tables = tables;
	}

}
