/****************************************************************************
**
** This file is part of the LibreCAD project, a 2D CAD program
**
** Copyright (C) 2010 R. van Twisk (librecad@rvt.dds.nl)
** Copyright (C) 2001-2003 RibbonSoft. All rights reserved.
**
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file gpl-2.0.txt included in the
** packaging of this file.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
** This copyright notice MUST APPEAR in all copies of the script!
**
**********************************************************************/


#include "rs_constructionline.h"

#include "rs_debug.h"



/**
 * Constructor.
 */
RS_ConstructionLine::RS_ConstructionLine(RS_EntityContainer* parent,
        const RS_ConstructionLineData& d)
        :RS_AtomicEntity(parent), data(d) {

    calculateBorders();
}



/**
 * Destructor.
 */
RS_ConstructionLine::~RS_ConstructionLine() {}



RS_Entity* RS_ConstructionLine::clone() {
    RS_ConstructionLine* c = new RS_ConstructionLine(*this);
    c->initId();
    return c;
}



void RS_ConstructionLine::calculateBorders() {
    minV = RS_Vector::minimum(data.point1, data.point2);
    maxV = RS_Vector::maximum(data.point1, data.point2);
}



RS_Vector RS_ConstructionLine::getNearestEndpoint(const RS_Vector& coord,
        double* dist) const{
    double dist1, dist2;

    dist1 = (data.point1-coord).squared();
    dist2 = (data.point2-coord).squared();

    if (dist2<dist1) {
        if (dist!=NULL) {
            *dist = sqrt(dist2);
        }
        return data.point2;
    } else {
        if (dist!=NULL) {
            *dist = sqrt(dist1);
        }
        return data.point1;
    }
}



RS_Vector RS_ConstructionLine::getNearestPointOnEntity(const RS_Vector& coord,
        bool /*onEntity*/, double* /*dist*/, RS_Entity** entity) const{

    if (entity!=NULL) {
        *entity = const_cast<RS_ConstructionLine*>(this);
    }

    RS_Vector ae = data.point2-data.point1;
    RS_Vector ea = data.point1-data.point2;
    RS_Vector ap = coord-data.point1;
    RS_Vector ep = coord-data.point2;

        if (ae.magnitude()<1.0e-6 || ea.magnitude()<1.0e-6) {
                return RS_Vector(false);
        }

    // Orthogonal projection from both sides:
    RS_Vector ba = ae * RS_Vector::dotP(ae, ap)
                   / (ae.magnitude()*ae.magnitude());
    RS_Vector be = ea * RS_Vector::dotP(ea, ep)
                   / (ea.magnitude()*ea.magnitude());

    return data.point1+ba;
}



RS_Vector RS_ConstructionLine::getNearestCenter(const RS_Vector& /*coord*/,
        double* dist) {

    if (dist!=NULL) {
        *dist = RS_MAXDOUBLE;
    }

    return RS_Vector(false);
}



RS_Vector RS_ConstructionLine::getMiddlePoint(){
    return RS_Vector(false);
}
RS_Vector RS_ConstructionLine::getNearestMiddle(const RS_Vector& /*coord*/,
        double* dist, const int /*middlePoints*/)const {
    if (dist!=NULL) {
        *dist = RS_MAXDOUBLE;
    }
    return RS_Vector(false);
}



RS_Vector RS_ConstructionLine::getNearestDist(double /*distance*/,
        const RS_Vector& /*coord*/,
        double* dist) {
    if (dist!=NULL) {
        *dist = RS_MAXDOUBLE;
    }
    return RS_Vector(false);
}


double RS_ConstructionLine::getDistanceToPoint(const RS_Vector& coord,
        RS_Entity** entity,
        RS2::ResolveLevel /*level*/, double /*solidDist*/) const {

    RS_DEBUG->print("RS_ConstructionLine::getDistanceToPoint");

    if (entity!=NULL) {
        *entity = const_cast<RS_ConstructionLine*>(this);
    }
    //double dist = RS_MAXDOUBLE;
    RS_Vector se = data.point2-data.point1;
    double d(se.magnitude());
    if(d<RS_TOLERANCE) {
        //line too short
        return RS_MAXDOUBLE;
    }
    se.set( se.x/d,-se.y/d); //normalized
    RS_Vector vpc= coord - data.point1;
    vpc.rotate(se); // rotate to use the line as x-axis, and the distance is fabs(y)
    return ( fabs(vpc.y) );
}



void RS_ConstructionLine::move(const RS_Vector& offset) {
    data.point1.move(offset);
    data.point2.move(offset);
    //calculateBorders();
}



void RS_ConstructionLine::rotate(const RS_Vector& center, const double& angle) {
    RS_Vector angleVector(angle);
    data.point1.rotate(center, angleVector);
    data.point2.rotate(center, angleVector);
    //calculateBorders();
}

void RS_ConstructionLine::rotate(const RS_Vector& center, const RS_Vector& angleVector) {
    data.point1.rotate(center, angleVector);
    data.point2.rotate(center, angleVector);
    //calculateBorders();
}

void RS_ConstructionLine::scale(const RS_Vector& center, const RS_Vector& factor) {
    data.point1.scale(center, factor);
    data.point2.scale(center, factor);
    //calculateBorders();
}



void RS_ConstructionLine::mirror(const RS_Vector& axisPoint1, const RS_Vector& axisPoint2) {
        data.point1.mirror(axisPoint1, axisPoint2);
        data.point2.mirror(axisPoint1, axisPoint2);
}



/**
 * Dumps the point's data to stdout.
 */
std::ostream& operator << (std::ostream& os, const RS_ConstructionLine& l) {
    os << " ConstructionLine: " << l.getData() << "\n";
    return os;
}


