/*
 * Copyright 2010-2011 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cache.ehcache;

import java.util.Collection;
import java.util.LinkedHashSet;

import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Status;

import org.springframework.cache.Cache;
import org.springframework.cache.support.AbstractCacheManager;
import org.springframework.util.Assert;

/**
 * CacheManager backed by an Ehcache {@link net.sf.ehcache.CacheManager}.
 * 
 * @author Costin Leau
 */
public class EhCacheCacheManager extends AbstractCacheManager {

	private net.sf.ehcache.CacheManager cacheManager;

	@Override
	protected Collection<Cache> loadCaches() {
		Assert.notNull(cacheManager, "a backing Ehcache cache manager is required");
		Status status = cacheManager.getStatus();

		Assert.isTrue(Status.STATUS_ALIVE.equals(status),
				"an 'alive' Ehcache cache manager is required - current cache is " + status.toString());

		String[] names = cacheManager.getCacheNames();
		Collection<Cache> caches = new LinkedHashSet<Cache>(names.length);
		
		for (String name : names) {
			caches.add(new EhCacheCache(cacheManager.getEhcache(name)));
		}
		
		return caches;
	}

	public Cache getCache(String name) {
		Cache cache = super.getCache(name);
		if (cache == null) {
			// check the Ehcache cache again
			// in case the cache was added at runtime

			Ehcache ehcache = cacheManager.getEhcache(name);
			if (ehcache != null) {
				// reinitialize cache map
				afterPropertiesSet();
				cache = super.getCache(name);
			}
		}

		return cache;
	}

	/**
	 * Sets the backing Ehcache {@link net.sf.ehcache.CacheManager}.
	 * 
	 * @param cacheManager backing Ehcache {@link net.sf.ehcache.CacheManager}
	 */
	public void setCacheManager(net.sf.ehcache.CacheManager cacheManager) {
		this.cacheManager = cacheManager;
	}
}