# -*- coding: utf-8 -*-

import unittest

from mcomix.bookmarks.bookmark import *

class BookmarkTest(unittest.TestCase):
    """ Tests behaviour of the Bookmark class. """

    def setUp(self):
        self.bookmark = Bookmark('Page 1', 'file.zip', 0, 0)

    def test_delete_bookmark(self):
        """ Removing a detached bookmark shouldn't raise an exception. """
        self.bookmark.delete()

    def test_delete_bookmark_from_folder(self):
        """ Removing item from folder should work as if the item was deleted
        directly at folder level. """
        folder = BookmarkFolder('Root folder')
        folder.add_bookmark(self.bookmark)

        self.bookmark.delete()

        self.assertEqual(len(folder), 0)
        self.assertIsNot(folder, self.bookmark.parent)

    def test_equality(self):
        """ Two bookmarks are identical as long as C{path} and
        C{page} remain identical. """
        other = Bookmark('Page 2', 'file.zip')
        self.assertEqual(self.bookmark, other)

        other = Bookmark('Page 1', 'file.zip', page=1)
        self.assertNotEqual(self.bookmark, other)

        other = Bookmark('Page 1', 'file.zip', page=0, num_pages=1)
        self.assertEqual(self.bookmark, other)

        other = Bookmark('Page 1', 'otherfile.zip')
        self.assertNotEqual(self.bookmark, other)

class BookmarkFolderTest(unittest.TestCase):
    """ Tests behaviour of the BookmarkFolder class. """

    def setUp(self):
        self.folder = BookmarkFolder('Root Folder')
        self.assertEqual(len(self.folder), 0)

    def test_add_bookmark(self):
        """ Adding a bookmark. """
        bookmark = Bookmark('Page 1', 'file.zip')
        self.folder.add_bookmark(bookmark)

        self.assertEqual(len(self.folder), 1)
        self.assertIn(bookmark, self.folder)
        self.assertEqual(bookmark, self.folder[0])
        self.assertIs(bookmark.parent, self.folder)

    def test_add_identical(self):
        """ A single bookmark should not be contained more than once
        in a folder. """
        bookmark = Bookmark('Page 1', 'file.zip')
        self.folder.add_bookmark(bookmark)
        self.folder.add_bookmark(bookmark)

        self.assertEqual(len(self.folder), 1)

    def test_delete_bookmark_from_folder(self):
        """ Removing a bookmark from a folder. """
        bookmark = Bookmark('Page 1', 'file.zip')
        self.folder.add_bookmark(bookmark)

        self.folder.delete_bookmark(bookmark)

        self.assertEqual(len(self.folder), 0)
        self.assertIsNot(self.folder, bookmark.parent)

    def test_delete(self):
        """ Deleting a folder recursively deletes all content. """
        newfolder = BookmarkFolder('Folder A')
        bookmark = Bookmark('Page 1', 'file.zip')
        newfolder.add_bookmark(bookmark)
        self.folder.add_bookmark(newfolder)

        self.folder.delete()

        self.assertEqual(len(self.folder), 0)
        self.assertEqual(len(newfolder), 0)
        self.assertIsNot(newfolder.parent, self.folder)
        self.assertIsNot(bookmark.parent, newfolder)

    def test_delete_from_subfolder(self):
        """ Deleting an item from a subfolder keeps the folder intact. """
        newfolder = BookmarkFolder('Folder A')
        bookmark = Bookmark('Page 1', 'file.zip', 0)
        bookmark2 = Bookmark('Page 2', 'file.zip', 1)
        newfolder.add_bookmark(bookmark)
        newfolder.add_bookmark(bookmark2)
        self.folder.add_bookmark(newfolder)

        bookmark.delete()

        self.assertEqual(len(newfolder), 1)
        self.assertEqual(len(self.folder), 1)
        self.assertIs(newfolder.parent, self.folder)
        self.assertIsNot(bookmark.parent, newfolder)

    def test_remove_bookmark(self):
        """ Removing a folder keeps everything it contains. """
        newfolder = BookmarkFolder('Folder A')
        bookmark = Bookmark('Page 1', 'file.zip', 0)
        newfolder.add_bookmark(bookmark)
        self.folder.add_bookmark(newfolder)

        self.folder.remove_bookmark(newfolder)

        self.assertEqual(len(self.folder), 0)
        self.assertEqual(len(newfolder), 1)
        self.assertIn(bookmark, newfolder)

# vim: expandtab:sw=4:ts=4
