/* tcl_barray.c
  
   The "barray" Tcl command and the "barray" Tcl object type
   are implemented here.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "util-tcl.h"
#include "dumbhex.h"

/*
 * If 0, then "ABC" is represented as "\x41\x42\x43".
 * If 1, then "ABC" is represented as "ABC".
 */
static int escape_printable;

enum formats {
    FORMAT_STRING,
    FORMAT_OCTAL,
    FORMAT_HEX,
    FORMAT_DECIMAL,
    FORMAT_UNSIGNED
};

/*
 * Specifies all the things that can be changed with the /FMT specification
 * in an examine command.
 */
struct formatspec {
    unsigned count;
    enum formats format;
    size_t size;
    int littleendian;
};

/*
 * s can be:
 *
 * "/1234"   -> count is 1234, leave format and size unchanged
 * "/1234x"  -> count is 1234, format is hex, leave size unchanged
 * "/1234b"  -> count is 1234, size is byte, leave format unchanged
 * "/1234xb" -> count is 1234, format is hex, size is byte
 * "/b"      -> count is 1, size is byte, leave format unchanged
 * "/x"      -> count is 1, format is hex, leave size unchanged
 *
 * This mimics the "examine" gdb command. Note that count is always
 * filled in.
 */
static int
parse_format(const char *s, struct formatspec *f)
{
    char *endptr;
    size_t c;

    /* Skip the slash and perform data validation at the same time */
    if (*s++ != '/')
	return -1;

    c = strtoul(s, &endptr, 10);
    f->count = c ? c : 1;

    for (s = endptr; *s; s++) {
	switch (*s) {
	/* Formats */
	case 's':
	    f->format = FORMAT_STRING;
	    break;
	case 'o':
	    f->format = FORMAT_OCTAL;
	    break;
	case 'x':
	    f->format = FORMAT_HEX;
	    break;
	case 'd':
	    f->format = FORMAT_DECIMAL;
	    break;
	case 'u':
	    f->format = FORMAT_UNSIGNED;
	    break;

	/* Sizes */
	case 'b':
	    f->size = sizeof(uint8_t);
	    break;
	case 'h':
	    f->size = sizeof(uint16_t);
	    break;
	case 'w':
	    f->size = sizeof(uint32_t);
	    break;

	/* Endianess (not present in gdb's examine command) */
	case '<':
	    f->littleendian = 1;
	    break;
	case '>':
	    f->littleendian = 0;
	    break;

	default:
	    return -1;
	}
    }

    return 0;
}

static int
dumpx(const struct payload *p, const struct formatspec *f, unsigned *offset)
{
    uint32_t value;
    const char *format;
    unsigned i, j, objects_per_line;
    void *ptr;

    if (*offset + f->count*f->size > p->len)
	return -1;

    /*
     * format   size   Number per line
     *   x        1            8
     *   x	  2            8
     *   x        4            4
     *   d        1            8
     *   d	  2            8
     *   d        4            4
     *   u        1            8
     *   u	  2            8
     *   u        4            4
     *   o        1            8
     *   o	  2            8
     *   o        4            4
     */
    if (f->format == FORMAT_STRING)
	objects_per_line = 1;
    else
	objects_per_line = f->size == sizeof(uint32_t) ? 4 : 8;

    for (ptr = p->data + *offset, i = 0; i <= f->count/objects_per_line; i++) {
	printf("0x%04x:  ", *offset);

	for (j = 0; j < objects_per_line; j++) {
	    if (i*objects_per_line + j >= f->count)
		break;

	    switch (f->format) {
	    case FORMAT_STRING:
		format = "\"%s\"";
		break;
	    case FORMAT_OCTAL:
		if (f->size == sizeof(uint8_t) ) {
		    format = "%03o ";
		} else if (f->size == sizeof(uint16_t) ) {
		    format = "%06o ";
		} else {
		    format = "%011o ";
		}
		break;
	    case FORMAT_HEX:
		if (f->size == sizeof(uint8_t) ) {
		    format = "0x%02x ";
		} else if (f->size == sizeof(uint16_t) ) {
		    format = "0x%04x ";
		} else {
		    format = "0x%08x ";
		}
		break;
	    case FORMAT_DECIMAL:
		if (f->size == sizeof(uint8_t) ) {
		    format = "0x%d ";
		} else if (f->size == sizeof(uint16_t) ) {
		    format = "0x%d ";
		} else {
		    format = "0x%d ";
		}
		break;
	    case FORMAT_UNSIGNED:
		if (f->size == sizeof(uint8_t) ) {
		    format = "0x%u ";
		} else if (f->size == sizeof(uint16_t) ) {
		    format = "0x%u ";
		} else {
		    format = "0x%u ";
		}
		break;
	    default:
		/* Just to shut the compiler up. We should never end up here */
		format = "";
	    }

	    if (f->format == FORMAT_STRING) {
		value = (uint32_t) ptr;
	    } else if (f->size == sizeof(uint8_t) ) {
		value = ( (uint8_t *) ptr)[i*objects_per_line + j];
	    } else if (f->size == sizeof(uint16_t) ) {
		value = SVAL(ptr, (i*objects_per_line + j)*f->size);
		if (!f->littleendian)
		    value = htons(value);
	    } else {
		value = IVAL(ptr, (i*objects_per_line + j)*f->size);
		if (!f->littleendian)
		    value = htonl(value);
	    }

	    printf(format, value);

	    *offset += f->format == FORMAT_STRING ? strlen(ptr) + 1 : f->size;
	}

	printf("\n");
    }

    return 0;
}

/********************************************************************
 *		     The "barray" Tcl object type                   *
 ********************************************************************/

/* Forward declarations */
static void update_string(Tcl_Obj *);
static void free_int_rep(Tcl_Obj *);
static int set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr);
static void dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclBArrayType = {
    .name = "barray",
    .freeIntRepProc = &free_int_rep,
    .dupIntRepProc = &dup_int_rep,
    .updateStringProc = &update_string,
    .setFromAnyProc = &set_from_any
};

static int
set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr _U_)
{
    if (interp)
	Tcl_SetResult(interp,
		      "barray object internal representation can't be created "
		      "from string", TCL_VOLATILE);

    return TCL_ERROR;
}

static void
dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    struct payload *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = (struct payload *) ckalloc(sizeof(struct payload) );
    new->len = old->len;
    new->data = (uint8_t *) ckalloc(new->len);
    memcpy(new->data, old->data, new->len);

    dupPtr->typePtr = &tclBArrayType;
    dupPtr->internalRep.otherValuePtr = new;
}

static void
update_string(Tcl_Obj *barray)
{
    unsigned i;
    struct payload *p;
    char *s;
    size_t len;
    int c;
    /*
     * Characters in this set are never displayed as they are, even if they
     * are printable. Instead, they are displayed as a hexadecimal escape
     * sequence.
     */
    static const char *banned_chars = ":\"' !";
    /*
     * Characters that will be escaped using the corresponding C escape
     * sequence.
     */
    static const char *escape_chars = "\\\n\t\r";

    p = barray->internalRep.otherValuePtr;

    /*
     * Compute length of the string for the worst case scenario, i.e. all bytes
     * in the byte array are non-printable.
     */
    len = p->len*(sizeof("\\xAB") - 1) + 1 /* for '\0' */;

    barray->bytes = ckalloc(len);
    if (!barray->bytes)
	return;

    for (s = barray->bytes, *s = '\0', i = 0; i < p->len; i++) {
	c = p->data[i];

	if (escape_printable) {
	    if (c && strchr(escape_chars, c) ) {
		switch (c) {
		case '\\':
		    strlcpy(s, "\\\\", len);
		    break;
		case '\n':
		    strlcpy(s, "\\n", len);
		    break;
		case '\r':
		    strlcpy(s, "\\r", len);
		    break;
		case '\t':
		    strlcpy(s, "\\t", len);
		    break;
		}

		s += 2;
	    } else if (isprint(c) && !strchr(banned_chars, c) ) {
		*s++ = c;
	    } else {
		*s++ = '\\';
		*s++ = 'x';
		snprintf(s, len, "%02x", c);
		s += 2;
	    }
	} else {
	    *s++ = '\\';
	    *s++ = 'x';
	    snprintf(s, len, "%02x", c);
	    s += 2;
	}
    }

    *s = '\0';

    barray->length = strlen(barray->bytes);
}

static void
free_int_rep(Tcl_Obj *obj)
{
    struct payload *p;

    p = obj->internalRep.otherValuePtr;
    ckfree( (char *) p->data);
    ckfree( (char *) p);
}

Tcl_Obj *
Tcl_NewBArrayObj(const uint8_t *data, size_t len)
{
    Tcl_Obj *obj;
    struct payload *p;

    obj = Tcl_NewObj();

    p = (struct payload *) ckalloc(sizeof(struct payload) );
    p->len = len;
    p->data = (uint8_t *) ckalloc(len);
    memcpy(p->data, data, len);

    obj->bytes = NULL;
    obj->typePtr = &tclBArrayType;
    obj->internalRep.otherValuePtr = p;

    return obj;
}

uint8_t *
Tcl_GetBArrayFromObj(Tcl_Obj *obj, size_t *size)
{
    struct payload *p;

    if (obj->typePtr != &tclBArrayType)
	return NULL;

    p = obj->internalRep.otherValuePtr;

    *size = p->len;

    return p->data;
}

/********************************************************************
 *                            barray new                            *
 ********************************************************************/

static int
tcl_barray_new(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    struct payload p;
    char errbuf[PAYLOAD_ERRBUF_SIZE];

    if (objc != 2) {
	nexp_error(interp, "usage: barray new <byte array spec>");
	return TCL_ERROR;
    }

    if (create_payload(Tcl_GetString(objv[1]), &p, errbuf) == -1) {
	nexp_error(interp, errbuf);
	return TCL_ERROR;
    }

    obj = Tcl_NewBArrayObj(p.data, p.len);
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *                         barray heximport                         *
 ********************************************************************/

static int
tcl_barray_heximport(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    GByteArray *array;
    int i, index;
    int prefixes_to_ignore = 0, ignore_after_col = 0;
    static const char *options[] = {
	"-ignore-prefixes", "-ignore-after-column", NULL
    };
    enum options {
	OPT_IGNPREFIXES, OPT_IGNCOLUMNS
    };

    for (i = 1; i < objc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "flag", 0, &index)
	    != TCL_OK)
	    return TCL_ERROR;

	switch ( (enum options) index) {
	case OPT_IGNPREFIXES:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, NULL);
		return TCL_ERROR;
	    }

	    Tcl_GetIntFromObj(interp, objv[i], &prefixes_to_ignore);
	    break;
	case OPT_IGNCOLUMNS:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, NULL);
		return TCL_ERROR;
	    }

	    Tcl_GetIntFromObj(interp, objv[i], &ignore_after_col);
	    break;
	}
    }

    if (i != objc - 1) {
	nexp_error(interp,
		   "usage: barray hex-import [-ignore-prefixes <prefixes>] "
		   "[-ignore-after-column <col>] <hex. string>");
	return TCL_ERROR;
    }

    array = dh_parsehex(Tcl_GetString(objv[i]), prefixes_to_ignore,
			ignore_after_col);

    if (array->len == 0) {
	nexp_error(interp, "Hexadecimal string produced no data");
	g_byte_array_free(array, TRUE);
	return TCL_ERROR;
    }

    obj = Tcl_NewBArrayObj(array->data, array->len);
    Tcl_SetObjResult(interp, obj);

    g_byte_array_free(array, TRUE);

    return TCL_OK;
}

/********************************************************************
 *                            barray len                            *
 ********************************************************************/

static int
tcl_barray_len(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    struct payload *p;

    if (objc != 2) {
	nexp_error(interp, "usage: barray length <barray variable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    p = (struct payload *) obj->internalRep.otherValuePtr;

    obj = Tcl_NewIntObj(p->len);
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *                          barray delete                           *
 ********************************************************************/

static int
tcl_barray_delete(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;

    if (objc != 2) {
	nexp_error(interp, "usage: barray delete <barray variable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    return Tcl_UnsetVar(interp, Tcl_GetString(objv[1]), TCL_LEAVE_ERR_MSG);
}

/********************************************************************
 *                          barray examine                          *
 ********************************************************************/

static int
tcl_barray_examine(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    static unsigned offset = 0;
    static struct formatspec f = {
	.count = 1,
	.format = FORMAT_HEX,
	.size = 1,
	.littleendian = 0
    };
    struct payload *p;

    if (objc < 2 || objc > 4)
	goto usage;

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   objv[1]);
	return TCL_ERROR;
    }

    p = obj->internalRep.otherValuePtr;

    if (objc == 3) {
	if (Tcl_GetString(objv[2])[0] == '/') {
	    if (parse_format(Tcl_GetString(objv[2]), &f) == -1) {
		nexp_error(interp, "invalid /FMT");
		return TCL_ERROR;
	    }
	} else {
	    offset = strtoul(Tcl_GetString(objv[2]), NULL, 0);
	}
    } if (objc == 4) {
	if (Tcl_GetString(objv[2])[0] != '/')
	    goto usage;

	if (parse_format(Tcl_GetString(objv[2]), &f) == -1) {
	    nexp_error(interp, "invalid /FMT");
	    return TCL_ERROR;
	}

	offset = strtoul(Tcl_GetString(objv[3]), NULL, 0);
    }

    if (dumpx(p, &f, &offset) == -1) {
	nexp_error(interp, "offset/count/size combination would access data "
			   "outside packet boundary");
	return TCL_ERROR;
    }

    return TCL_OK;

usage:
    nexp_error(interp,
	       "usage: barray examine <barray variable> [/FMT] [offset]");
    return TCL_ERROR;
}

/********************************************************************
 *                            barray dump                           *
 ********************************************************************/

/*
 * FIXME: we should be using Tcl_GetBArrayFromObj() here instead of
 * accessing the internal fields directly; see tcl_barray_slice().
 */
static int
tcl_barray_dump(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    struct payload *p;

    if (objc != 2) {
	nexp_error(interp, "usage: barray dump <barray variable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    p = obj->internalRep.otherValuePtr;

    dump(p->data, p->len);

    return TCL_OK;
}

/********************************************************************
 *                            barray slice                          *
 ********************************************************************/

static int
tcl_barray_slice(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    char *s;
    size_t start, end;
    struct payload p, slice;

    if (objc != 3) {
	nexp_error(interp,
		   "usage: barray slice <barray variable> <slice spec>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    p.data = Tcl_GetBArrayFromObj(obj, &p.len);

    start = strtoul(Tcl_GetString(objv[2]), NULL, 0);

    s = strchr(Tcl_GetString(objv[2]), ':');
    if (!s) {
	s = strchr(Tcl_GetString(objv[2]), 'l');
	if (!s) {
	    nexp_error(interp, "invalid slice specification");
	    return TCL_ERROR;
	}

	slice.len = strtoul(s + 1, NULL, 0);

	if (start + slice.len > p.len) {
	    nexp_error(interp, "invalid start or end of byte array");
	    return TCL_ERROR;
	}
    } else {
	end = strtoul(s + 1, NULL, 0);
	if (end == 0)
	    end = p.len - 1;

	if (start > end || end > p.len - 1) {
	    nexp_error(interp, "invalid start or end of byte array");
	    return TCL_ERROR;
	}

	slice.len = end - start + 1;
    }

    slice.data = xmalloc(slice.len);
    memcpy(slice.data, p.data + start, slice.len);

    obj = Tcl_NewBArrayObj(slice.data, slice.len);
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *			      barray cmp			    *
 ********************************************************************/

static int
tcl_barray_cmp(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj1, *obj2, *result;
    struct payload p1, p2;
    size_t n;

    if (objc != 3) {
	nexp_error(interp,
		   "usage: barray cmp <barray variable 1> <barray variable 2>");
	return TCL_ERROR;
    }

    obj1 = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj1)
	return TCL_ERROR;

    if (obj1->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    obj2 = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj2)
	return TCL_ERROR;

    if (obj2->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[2]) );
	return TCL_ERROR;
    }

    p1.data = Tcl_GetBArrayFromObj(obj1, &p1.len);
    p2.data = Tcl_GetBArrayFromObj(obj2, &p2.len);

    n = p1.len < p2.len ? p1.len : p2.len;

    result = Tcl_NewIntObj(memcmp(p1.data, p2.data, n) );
    Tcl_SetObjResult(interp, result);

    return TCL_OK;
}

/********************************************************************
 *			    barray string			    *
 ********************************************************************/

static int
tcl_barray_string(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    struct payload p;

    if (objc != 2) {
	nexp_error(interp, "usage: barray string <barray variable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    p.data = Tcl_GetBArrayFromObj(obj, &p.len);

    /*
     * Important: can't use Tcl_NewStringObj() here since then
     * Tcl will perform Unicode conversion in some situations. We
     * want to return raw, binary data, so we must use
     * Tcl_NewByteArrayObj().
     */
    obj = Tcl_NewByteArrayObj(p.data, p.len);
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *			    barray cksum 			    *
 ********************************************************************/

static int
tcl_barray_cksum(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    Tcl_Obj *obj;
    struct payload p;

    if (objc != 2) {
	nexp_error(interp, "usage: barray cksum <barray variable>");
	return TCL_ERROR;
    }

    obj = Tcl_ObjGetVar2(interp, objv[1], NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object",
		   Tcl_GetString(objv[1]) );
	return TCL_ERROR;
    }

    p.data = Tcl_GetBArrayFromObj(obj, &p.len);

    obj = Tcl_NewIntObj(in_checksum(p.data, p.len) );
    Tcl_SetObjResult(interp, obj);

    return TCL_OK;
}

/********************************************************************
 *				barray                              *
 ********************************************************************/

static int
NExp_BArrayObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
		  Tcl_Obj * const *objv)
{
    int index, retval;
    static const char *subcmds[] = {
	"exists", "new", "length", "delete", "examine", "dump", "slice",
	"cmp", "string", "cksum", "hex-import", NULL
    };
    enum subcmds {
	SUBCMD_EXISTS, SUBCMD_NEW, SUBCMD_LENGTH, SUBCMD_DELETE,
	SUBCMD_EXAMINE, SUBCMD_DUMP, SUBCMD_SLICE, SUBCMD_CMP, SUBCMD_STRING,
	SUBCMD_CKSUM, SUBCMD_HEXIMPORT
    };

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "subcmd barrayName ?arg ...?");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_EXISTS: {
	Tcl_Obj *obj;

	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "barrayName");
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj || obj->typePtr != &tclBArrayType) {
	    /* Variable does not exist or it's not of our type */
	    obj = Tcl_NewBooleanObj(0);
	} else {
	    obj = Tcl_NewBooleanObj(1);
	}

	Tcl_SetObjResult(interp, obj);
	return TCL_OK;
    }
    case SUBCMD_NEW:
	retval = tcl_barray_new(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_LENGTH:
	retval = tcl_barray_len(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_DELETE:
	retval = tcl_barray_delete(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_EXAMINE:
	retval = tcl_barray_examine(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_DUMP:
	retval = tcl_barray_dump(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_SLICE:
	retval = tcl_barray_slice(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_CMP:
	retval = tcl_barray_cmp(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_STRING:
	retval = tcl_barray_string(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_CKSUM:
	retval = tcl_barray_cksum(interp, objc - 1, &objv[1]);
	break;
    case SUBCMD_HEXIMPORT:
	retval = tcl_barray_heximport(interp, objc - 1, &objv[1]);
	break;
    }

    return retval;
}

/********************************************************************
 *				  x                                 *
 ********************************************************************/

static int
NExp_XCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
	  const char **argv)
{
    Tcl_Obj *obj;
    static unsigned offset = 0;
    static struct formatspec f = {
	.count = 1,
	.format = FORMAT_HEX,
	.size = 1,
	.littleendian = 0
    };
    struct payload *p;

    obj = Tcl_GetVar2Ex(interp, PACKETDATA_VARNAME, NULL, TCL_LEAVE_ERR_MSG);
    if (!obj)
	return TCL_ERROR;

    if (obj->typePtr != &tclBArrayType) {
	nexp_error(interp, "\"%s\" is not a barray object", PACKETDATA_VARNAME);
	return TCL_ERROR;
    }

    p = obj->internalRep.otherValuePtr;

    if (argc == 1)
	/*
	 * Use previously used size, format, count, and offset.
	 */
	;
    else if (argc == 2) {
	if (argv[1][0] == '/') {
	    if (parse_format(argv[1], &f) == -1) {
		nexp_error(interp, "invalid /FMT");
		return TCL_ERROR;
	    }
	} else
	    offset = strtoul(argv[1], NULL, 0);
    } else if (argc == 3) {
	if (argv[1][0] != '/')
	    goto usage;

	if (parse_format(argv[1], &f) == -1) {
	    nexp_error(interp, "invalid /FMT");
	    return TCL_ERROR;
	}
	offset = strtoul(argv[2], NULL, 0);
    } else
	goto usage;

    if (dumpx(p, &f, &offset) == -1) {
	nexp_error(interp, "offset/count/size combination would access data "
			   "outside packet boundary");
	return TCL_ERROR;
    }

    return TCL_OK;

usage:
    nexp_error(interp, "usage: x [/FMT] [offset]");
    return TCL_ERROR;
}

static struct nexp_cmd_data cmd_data[] = {
    {"barray", NExp_BArrayObjCmd, NULL, 0, 0},
    {"x", NULL, NExp_XCmd, 0, 0},
#if 0
    {"e", NULL, NExp_ECmd, 0, 0},
#endif

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_barray_cmd(Tcl_Interp *interp)
{
    Tcl_RegisterObjType(&tclBArrayType);

    setup_tcl_vartrace(interp, TCLVAR_INT, "escape_printable",
		       &escape_printable);

    nexp_create_commands(interp, cmd_data);
}
