/* tgn.c

   Network Expect's TGN: a simple CLI-based traffic generator.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <pcap.h>
#include <dnet.h>

#include "tgn.h"
#include "nexp_pbuild.h"
#include "nexp_speakers.h"
#include "util.h"
#include "xmalloc.h"

static int rseed;

int vflag, tflag;
char *program_name;

static void
version(void)
{
    printf("\
TGN Version %s\n\
Copyright (c) 2007, 2008, 2009, 2010 Eloy Paris <peloy@netexpect.org>\n\
\n", PACKAGE_VERSION);
}

static void
usage(void)
{
    printf("\
Usage: %s [OPTION...] <PDU definition>\n\
  -V                   display version information\n\
  -s                   set a specific random seed\n\
  -v                   increase verbosity level\n\
  -l                   list speakers\n\
  -w <PCAP filename>   send generated traffic to this PCAP file\n\
  -o <output speaker>  use specific speaker for output\n\
  -c <count>           number of times to send packet\n\
  -r <PPS rate>        specify sending rate in packets-per-second\n\
  -d <delay>           specify sending sending delay in seconds\n\
  -n                   dry run: just dump the PDU definition\n\
\n", program_name);
}

int
main(int argc, char **argv)
{
    int c;
    int list_speakers = 0, pcap_writing = 0;
    char *speaker_name = NULL;
    char *pdudef;
    pcb_t pcb;
    char errbuf[PDU_ERRBUF_SIZE];
    int i, count, packet_count, packets_sent;
    int dry_run;
    size_t packet_size;
    uint8_t *packet;
    double delay;
    struct nexp_speaker speaker_parms;

    program_name = strrchr(argv[0], '/');
    if (program_name)
	program_name++;
    else
	program_name = argv[0];

    memset(&pcb, 0, sizeof pcb);
    count = 0;
    dry_run = 0;

    while ( (c = getopt(argc, argv, "hvVs:lc:no:d:r:w:") ) != -1) {
	switch (c) {
	case 'h':
	    usage();
	    return EXIT_SUCCESS;
	case 'v':
	    vflag++;
	    break;
	case 'V':
	    version();
	    return EXIT_SUCCESS;
	case 's':
	    rseed = atoi(optarg);
	    break;
	case 'l':
	    list_speakers++;
	    break;
	case 'c':
	    count = atoi(optarg);
	    break;
	case 'n':
	    dry_run++;
	    break;
	case 'o':
	    if (pcap_writing) {
		fprintf(stderr, "-o and -w options are mutually exclusive.\n");
		return EXIT_FAILURE;
	    }

	    speaker_name = optarg;
	    break;
	case 'd':
	    delay = strtod(optarg, NULL);
	    pcb.delay.tv_sec = delay;
	    pcb.delay.tv_usec = (delay - pcb.delay.tv_sec)*1000000UL;
	    break;
	case 'r':
	    delay = strtod(optarg, NULL);
	    if (delay == 0.0) {
		fprintf(stderr, "Rate can't be 0 packets per second.");
		return EXIT_FAILURE;
	    }

	    delay = 1/delay;

	    pcb.delay.tv_sec = delay;
	    pcb.delay.tv_usec = (delay - pcb.delay.tv_sec)*1000000UL;
	    break;
	case 'w':
	    if (speaker_name) {
		fprintf(stderr, "-o and -w options are mutually exclusive.\n");
		return EXIT_FAILURE;
	    }

	    speaker_parms.type = SPEAKER_PCAP;
	    speaker_parms._pcap.fname = optarg;

	    pcb.speaker = nexp_newspeaker(&speaker_parms, errbuf);
	    if (!pcb.speaker) {
		fprintf(stderr, "%s\n", errbuf);
		return EXIT_FAILURE;
	    }

	    pcap_writing++;
	    break;
	default:
	    usage();
	    return EXIT_FAILURE;
	}
    }

    if (!list_speakers) {
	/*
	 * A PDU definition is needed if we're not listing speakers
	 */
	pdudef = copy_argv(&argv[optind]);
	if (!pdudef) {
	    fprintf(stderr, "%s: a PDU definition is needed.\n",
		    program_name);
	    return EXIT_FAILURE;
	}
    }

    srand(rseed ? rseed : time(NULL) );

    create_default_speakers();

    if (atexit(close_speakers) != 0) {
	if (!list_speakers)
	    free(pdudef);
	fprintf(stderr, "cannot set exit function close_speakers()\n");
	exit(EXIT_FAILURE);
    }

    if (list_speakers) {
	speakers_info();
	return EXIT_SUCCESS;
    }

    if (speaker_name) {
	/* User specified a speaker via the -o option */
	pcb.speaker = lookup_speaker(speaker_name);
	if (!pcb.speaker) {
	    fprintf(stderr, "No speaker named \"%s\".\n", speaker_name);
	    return EXIT_FAILURE;
	}
    }

    pb_init(); /* Initialize the libpbuild subsystem */

    if ( (pcb.pdu = pb_parsedef(pdudef, errbuf) ) == NULL) {
	fprintf(stderr, "%s\n", errbuf);
	return EXIT_FAILURE;
    }

    pcb.def = pdudef;

    if (dry_run) {
	pcb_dump(&pcb, vflag);
	printf("\n");
	pb_dumppdu(pcb.pdu);
	pcb_destroy(&pcb);
	return EXIT_SUCCESS;
    }

    if (!pcb.speaker) {
	/* User didn't specify a speaker; look for a default speaker. */
	pcb.speaker = lookup_speaker("ip"); /* Exists if running as root */
	if (!pcb.speaker)
	    pcb.speaker = lookup_speaker("hex"); /* Guaranteed to exist */
    }

    packet_count = count ? count : pb_permutation_count(pcb.pdu);
    packet = xmalloc(pb_len(pcb.pdu) );

    /* Sending loop */
    for (packets_sent = i = 0; i < packet_count; i++) {
	packet_size = pb_build(pcb.pdu, packet, NULL);

	/*
	 * Handle inter-packet sending delay.
	 */
	if (timerisset(&pcb.delay)
	    && timerisset(&pcb.speaker->ts) ) {
	    struct timeval now, later;

	    timeradd(&pcb.speaker->ts, &pcb.delay, &later);

	    gettimeofday(&now, NULL);
	    /*
	     * Some versions of Solaris (seen in Solaris 8) don't
	     * like using timercmp() with "<=" or ">=". Rearranging
	     * the operands and just using '>' or '<' works around
	     * this limitation.
	     */
	    while (timercmp(&later, &now, >) )
		gettimeofday(&now, NULL);
	}

	nexp_pdu_output(&pcb, packet, packet_size);

	packets_sent++;
    }

    printf("Sent %d packets.\n", packets_sent);

    free(packet);

    pcb_destroy(&pcb);

    return EXIT_SUCCESS;
}
