/* OpenVAS Client
 * Copyright (C) 1998 - 2001 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <includes.h>

#include <openvas/network.h>

#include "openvas_plugin.h"
#include "comm.h"
#include "auth.h"
#include "parser.h"
#include "attack.h"
#include "context.h"
#include "preferences.h"
#include "error_dlg.h"
#include "openvas_i18n.h"
#include "hash_table_file.h"

#include <openvas/omp/omp.h> /* for omp_authenticate */
#include <openvas_server.h>

static void setup_plug_list(struct openvas_plugin *,struct openvas_plugin *, char *);

/**
 * @brief Restores a session.
 */
#ifdef ENABLE_SAVE_TESTS
void
restore_attack (char * session_name, struct context *context)
{
  struct arglist * preferences = context->prefs;
  char * plug_list, * old_plug_list;
  struct arglist * serv_prefs;

  /* Count how many plugins we have */
  int num_plug     = openvas_plugin_list_length (context->plugins);
  int num_scanners = openvas_plugin_list_length (context->scanners);

  /* Set up the plugin list, according to the
   * OpenVAS Transfer Protocol version 1.1 / OTP */
  plug_list = emalloc(num_plug*100+1+num_scanners*100+1);
  setup_plug_list(context->plugins, context->scanners, plug_list);
  if( plug_list[0] == '\0' )
    sprintf(plug_list, "0");

  /* Add plugin_set to server preferences so it gets sent to the server */
  serv_prefs = arg_get_value(preferences, "SERVER_PREFS");
  if((old_plug_list = arg_get_value(serv_prefs, "plugin_set")))
   {
   efree(&old_plug_list);
   arg_set_type(serv_prefs, "plugin_set", ARG_STRING);
   arg_set_value(serv_prefs, "plugin_set", strlen(plug_list)+1, plug_list);
   }
  else
   arg_add_value(serv_prefs, "plugin_set", ARG_STRING, strlen(plug_list), plug_list);
  comm_send_preferences(context);
  preferences_save(context);
  comm_send_rules(context);
  comm_restore_session(context, session_name);
  /* efree(&plug_list); it is stored as SERVER_PREFS[plugin_set] */
}
#endif

#ifdef USE_OMP
GSList* send_ssh_credential_files (GSList*);
#endif

/**
 * @brief Sends the order to start a new attack to the scanner (openvassd).
 *
 * @param hostname Name of the host to test first.
 * @param context Context to use.
 */
int
attack_host (char* hostname, struct context* context)
{
  struct arglist * preferences = context->prefs;
  char * plug_list, * old_plug_list;
  struct arglist * serv_prefs;
  int num_plug = 0;
  int num_scanners = 0;

#ifdef USE_OMP
  if (context->protocol == PROTOCOL_OMP)
    {
      int socket, port;
      gnutls_session_t session;
      const char *hostname, *login, *password;
      struct context* manager;

      /* Ensure the RC file is up to date. */

      context_sync_plugin_prefs (context);
      preferences_save (context);

      /* Connect. */

      manager = context_by_type (context, CONTEXT_SERVER);
      hostname = prefs_get_string (manager, "nessusd_host");
      port = prefs_get_int (manager, "nessusd_port");
      login = prefs_get_string (manager, "nessusd_user");
      /* It's up to the caller (prefs_dialog_execute via
       * monitor_dialog_setup) to query the user as necessary to ensure
       * that a password is available. */
      password = manager->passwd;

      socket = openvas_server_open (&session, hostname, port);
      if (socket == -1)
        {
          show_error (_("Failed to connect to manager"));
          return 0;
        }

      if (omp_authenticate (&session, login, password))
        {
          openvas_server_close (socket, session);
          show_error (_("Failed to authenticate as %s with manager"), login);
          return 0;
        }

      /* Send rcfile, in case it is newer. */
      {
        gchar *rcfile, *rc;
        GError* error;
        gsize rc_len;

        rcfile = preferences_get_altname (context, NULL);

        error = NULL;
        g_file_get_contents (rcfile, &rc, &rc_len, &error);
        if (error)
          {
            g_error_free (error);
            openvas_server_close (socket, session);
            show_error (_("Failed to authenticate read in RC file %s"),
                        rcfile);
            return 0;
          }

        if (omp_modify_task (&session, prefs_get_string (context, "id"),
                             rc, NULL, NULL))
          {
            openvas_server_close (socket, session);
            show_error (_("Failed to set RC file on manager task"));
            return 0;
          }
      }

      /* Send files, in case they are newer. */

      if (comm_omp_send_files (context, socket, session) == FALSE)
        return 0;

      /* Start the task. */

      if (omp_start_task (&session, prefs_get_string (context, "id")))
        {
          show_error (_("Failed to start task"));
          openvas_server_close (socket, session);
          return 0;
        }

      openvas_server_close (socket, session);
      return 1;
    }
#endif /* USE_OMP */

  // Save the .host_sshlogin file so that the current selection will be sent to
  // the server (in comm.c:send_ssh_credential_files)
  // FIXME: Should be bundled somewhere when signal handling is fixed. (e.g. context_save_recurse)
  if (Context->map_target_sshlogin != NULL)
    {
      char* fileloc = g_build_filename (Context->dir, ".host_sshlogins", NULL);
      gboolean success = hash_table_file_write (Context->map_target_sshlogin, fileloc);
      if (success == FALSE)
        show_warning (_("Could not update Login selection per target - file.\n"));
      efree (&fileloc);
    }

  /* Count how many plugins we have */
  num_plug     = openvas_plugin_list_length (context->plugins);
  num_scanners = openvas_plugin_list_length (context->scanners);

  /* Set up the plugin list, according to the
   * OpenVAS Transfer Protocol version 1.1 / OTP */
  plug_list = emalloc(num_plug*100+1+num_scanners*100+1);
  setup_plug_list(context->plugins, context->scanners, plug_list);
  if( plug_list[0] == '\0' )
    sprintf(plug_list, "0");

  /* Add plugin_set to server preferences so it gets sent to the server */
  serv_prefs = arg_get_value(preferences, "SERVER_PREFS");
  if((old_plug_list = arg_get_value(serv_prefs, "plugin_set")))
   {
   efree(&old_plug_list);
   arg_set_type(serv_prefs, "plugin_set", ARG_STRING);
   arg_set_value(serv_prefs, "plugin_set", strlen(plug_list)+1, plug_list);
   }
  else
   arg_add_value(serv_prefs, "plugin_set", ARG_STRING, strlen(plug_list), plug_list);
  comm_send_preferences(context);
  preferences_save(context);
  comm_send_rules(context);

  network_printf(context->socket, "CLIENT <|> LONG_ATTACK <|>\n");
  network_printf(context->socket, "%d\n", strlen(hostname));
  if(context->socket > 0)
  {
   int len = strlen(hostname);
   int n = 0;
   /* send by packets of 1024 bytes */
   while(n < len)
   {
    int size = 1024;
    int m = 0;
    while(m < size)
    {
     int e;
     if((len - m - n) < size)size = len-m-n;
     e = nsend(context->socket, &(hostname[n+m]), size, 0);
     if(e < 0)
     {
      perror("send ");
      return 0;
     }
     m+=e;
    }
    n+=m;
   }
   return 1;
  }
  else
    return 0;
}

/**
 * @brief Converts OIDs of the NVTs wich are enabled to a string.
 *
 * OIDs are separated by semicolons
 * (ie : '1.3.6.1.4.1.25623.1.0.1;1.3.6.1.4.1.25623.1.0.3;1.3.6.1.4.1.25623.1.0.4')
 *
 * @param[out] plug_list Will contain the list when done.
 */
static void
setup_plug_list (struct openvas_plugin * plugs, struct openvas_plugin * scanners,
                 char * plug_list)
{
  struct openvas_plugin * w = NULL;
  int i = 0;
  char sp[100];

  for (i=0; i<2; i++)
    {
      if (i == 0) w = plugs;
      else w = scanners;

      while (w != NULL)
        {
          if (w->enabled)
            {
              snprintf (sp, sizeof(sp), "%s;", w->oid);
              memcpy (plug_list, sp, strlen(sp) + 1);
              plug_list += strlen (sp);
            }
          w = w->next;
        }
      w = plugs;
    }
}
