/* $Id: context.h,v 1.16 2006-07-03 14:43:21 bh Exp $
 *
 * Copyright (C) 2004 by Greenbone Networks GmbH
 * Author(s):
 * Thomas Arendsen Hein <thomas@intevation.de>
 *
 * This program is free software under the GNU GPL (>=v2)
 * Read the file COPYING coming with the software for details.
 */

#ifndef CONTEXT_H
#define CONTEXT_H

#ifdef USE_GTK
#include <gtk/gtk.h>
#endif

#ifndef USE_GTK
#include <glib.h>
#endif

#ifdef USE_OMP
#include <gnutls/gnutls.h>
#endif

#ifdef USE_OMP
typedef enum
{
  PROTOCOL_OMP,
  PROTOCOL_OTP
} protocol_t;
#endif /* USE_OMP */

typedef enum
{
#ifdef USE_OMP
  CONTEXT_SERVERS = -2,
  CONTEXT_SERVER = -1,
#endif
  CONTEXT_GLOBAL = 0, /**< The GLOBAL scope is parent of all other scopes and acts to define defaults. */
  CONTEXT_TASK,
  CONTEXT_SCOPE,
  CONTEXT_REPORT,
  CONTEXT_ENUM
} context_type;


typedef enum {
  CONTEXT_IDLE  = 0,
  CONTEXT_SCANNING
} context_action;

/**
 * @brief A context bundles information and Gtk-Widgets (if USE_GTK).
 *
 * Basically everything that needs to be accessed from any point in the client
 * is in here.
 * It implements a tree and a list, as it has a parent, a child and a next.
 * So a contexts children are its primary child and the 'next's of its primary
 * child.
 * Practically, as you decend the tree, you also descend in the context_type
 * (root is CONTEXT_GLOBAL, roots children are CONTEXT_TASKs and so on).
 *
 * The currently active context can be normally accessed as through the global
 * variable Context.
 *
 * Contexts are reflected on disk in seperate folders (.openvas keeps the global
 * one), each keeping its very own openvasrc and eventually a plugin cache.
 */
struct context {
  context_type type;
  context_action action;

  struct arglist *prefs;
  char * plugins_md5sum;
  struct openvas_plugin *plugins;
  struct openvas_plugin *scanners;
  struct arglist *dependencies;
  char *dir;
#ifdef USE_OMP
  protocol_t protocol;
  gnutls_session_t session;
#endif
  int socket;
  const char *passwd;
  struct context *parent;
  struct context *children;
  struct context *next;
#ifdef ENABLE_SAVE_TESTS
  GHashTable * sessions;
  int sessions_saved;
  int detached_sessions_saved;
#endif
#ifdef USE_GTK
  GtkTreeRowReference *treerowref;
  GtkWidget *move_menuitem;
  GtkWidget *plugin_prefs_widget;
  GtkWidget *plugin_prefs_cred_widget;
  GtkWidget *pbar;
  GtkTreeStore *plugin_tree_store;
  GtkTreeModel *plugin_tree_model;
  // The PREF_SSH_CREDENTIALS preference of a target is handled differently than
  // other preferences. Since the gui for it is flexible (depends on managed
  // ssh- accounts and defined targets) we need to access it more often then
  // other nvt-preference gui elements. Walking through two arglists each time
  // seems to be an overkill, so the widget is hardwired in a context.
  /** GtkTable with logins per target selection, if triggered by a nvt. */
  GtkTable* gui_sshlogins_per_target;
#endif /* USE_GTK */

  /** Maps certificate_t* (value) to their fingerprints (key). */
  GHashTable* signer_fp_certificates;

  /** Maps names of ssh public key information bundles to openvas_ssh_logins */
  GHashTable* sshkeys;

  /** Maps target names to pointers to ssh_logins. */
  GHashTable* map_target_sshlogin;

  /** Indicates whether the plugin information has been loaded. Useful because
    * Reports may have plugin information too. They can be quite large, and we
    * want to avoid loading them. */
  int plugin_cache_loaded;

  /** Indicates wether a severity_filter was applied to a report (this flag
   * makes only sense for reports).
   */
  gboolean is_severity_mapped;
};

extern struct context *Global;
extern struct context *Context;
#ifdef USE_OMP
extern struct context *Servers;
#endif

void context_init(struct context**, struct context*);
struct context *context_by_type(struct context*, context_type);
void context_reset_plugin_tree(struct context*);
void context_force_plugin_prefs_redraw(struct context *context);
int context_add_plugin(struct context*, struct openvas_plugin*);
void context_load_plugin_cache(struct context *context);
void context_set_plugins_md5sum(struct context *context, const char *md5sum);
void context_reset_plugins(struct context *context);

void context_sync_plugin_prefs(struct context *);

void context_collect (struct context*);
void context_collect_recurse (struct context *context, const char* dir);

#ifdef USE_OMP
void context_collect_servers (struct context *);

/* FIX For monitor_dialog.c.  Should use ~context_add_report. */
void context_remove_child (struct context *, struct context *);
struct context *context_create_child (struct context *);
void context_collect_recurse (struct context *, const char *);
char *context_newpath (const char *, const char *, const char *);
#endif /* USE_OMP */

void context_rename(struct context*, const char*);
int context_move(struct context*, struct context*);
struct context *context_new(struct context*, const char*, const char*);
#ifdef USE_OMP
struct context *context_add(struct context*, const char*, const char*);
int delete_in_client_only;
#endif
void context_delete(struct context*);
void context_save_recurse(struct context*);

/* TODO move to e.g. file_check.c */
int check_exists(const char *);
int check_is_file(const char *);
int check_is_dir(const char *);

#endif /* ifndef CONTEXT_H */
