/* OpenVAS Client
 * Copyright (C) 1998 - 2001 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * @file
 * 
 * Module that deals with representation of (collections of) NVTs.
 * 
 * Uses the cache mechanism of arglists.
 * 
 * @TODO Partly obsoleted by nvti.h/c (although some fields do not correspond).
 */

#include <includes.h>
#include <glib.h>
#include "openvas_plugin.h"

/**
 * @brief Static store of NVTIs.
 *
 * It is never free'd.
 */
static nvtis_t * ni_store = NULL;

/**
 * @brief Initializes a new plugin struct.
 */
struct openvas_plugin *
openvas_plugin_new (char * oid,
                   char * name,
                   char * category,
                   char * copyright,
                   char * description,
                   char * summary,
                   char * family,
                   char * version,
                   char * cve,
                   char * bid,
                   char * xrefs,
                   char * sign_key_ids,
                   char * script_tags
                   )
{
 struct openvas_plugin * np = malloc (sizeof(*np));

 if (np == NULL) abort();

 bzero (np, sizeof(*np));

  if (! ni_store) ni_store = nvtis_new();

  np->ni = nvtis_lookup(ni_store, oid);
  if (! np->ni)
    {
      np->ni = nvti_new();

      nvti_set_oid(np->ni, oid);
      nvti_set_version(np->ni, version);
      nvti_set_name(np->ni, name);
      nvti_set_summary(np->ni, summary);
      nvti_set_description(np->ni, description);
      nvti_set_copyright(np->ni, copyright);
      nvti_set_cve(np->ni, cve);
      nvti_set_bid(np->ni, bid);
      nvti_set_xref(np->ni, xrefs);
      nvti_set_tag(np->ni, script_tags);
      nvti_set_sign_key_ids(np->ni, sign_key_ids);
      nvti_set_family(np->ni, family);

      nvtis_add(ni_store, np->ni);
    }

 np->oid          = cache_inc (oid);
 np->md5sum       = NULL;
 np->category     = cache_inc (category);
 np->next    = NULL;

 return np;
}


/**
 * @brief Traverses a list of NVT and returns the count.
 *
 * @param list List of NVTs, if NULL, returns 0.
 *
 * @return Number of NVTs in list (0 if \ref list equals NULL).
 */
int
openvas_plugin_list_length (struct openvas_plugin* list)
{
  int count = 0;
  struct openvas_plugin * nvt = list;

  while (nvt != NULL)
    {
      ++count;
      nvt = nvt->next;
    }

  return count;
}


/**
 * @brief Set the md5 sum of the plugin (parameter is copied).
 *
 * @param plugin The NVT to get a new md5sum.
 * @param md5sum New md5sum of a plugin (will be copied).
 */
void
openvas_plugin_set_md5sum (struct openvas_plugin * plugin, const char * md5sum)
{
  efree (&plugin->md5sum);
  if (md5sum != NULL)
    plugin->md5sum = estrdup (md5sum);
}


/**
 * @brief Returns the first plugin in the plugin set \<plugins\> with the name
 * @brief \<name\>.
 *
 * @param plugins List of plugins.
 * @param name    Name of plugin to find.
 *
 * @return First plugin in plugin set with given name.
 */
struct openvas_plugin *
openvas_plugin_get_by_name (struct openvas_plugin * plugins, char * name)
{
  if (!name)
    return NULL;

  while (plugins != NULL)
    {
      if (nvti_name(plugins->ni) && strcmp (nvti_name (plugins->ni), name) == 0)
        return plugins;
      plugins = plugins->next;
    }

  return NULL;
}

/**
 * @brief Returns the first plugin in the plugin set \ref plugins with the OID
 * @brief \ref oid.
 *
 * @param plugins List of plugins.
 * @param name    OID of plugin to find.
 *
 * @return First plugin in plugin set with given OID.
 */
struct openvas_plugin *
openvas_plugin_get_by_oid (struct openvas_plugin * plugins, const char * oid)
{
  while (plugins != NULL)
    {
      if (!strcmp (plugins->oid, oid))
        return plugins;

      plugins = plugins->next;
    }

  return NULL;
}

/**
 * @brief Returns a duplicate of a plugin.
 * 
 * Preferences are deep- copied.
 * 
 * @param plugin The plugin to duplicate.
 * 
 * @return Duplicate of plugin parameter.
 */
struct openvas_plugin*
openvas_plugin_duplicate (struct openvas_plugin *plugin)
{
  struct openvas_plugin * copy = openvas_plugin_new (plugin->oid,
                           nvti_name (plugin->ni), plugin->category,
                           nvti_copyright (plugin->ni),
                           nvti_description (plugin->ni),
                           nvti_summary (plugin->ni), nvti_family (plugin->ni),
                           nvti_version (plugin->ni), nvti_cve (plugin->ni),
                           nvti_bid (plugin->ni), nvti_xref (plugin->ni),
                           nvti_sign_key_ids (plugin->ni), nvti_tag (plugin->ni));
  copy->enabled = plugin->enabled;
  openvas_plugin_set_md5sum(copy, plugin->md5sum);
  if (plugin->plugin_prefs != NULL)
    {
      copy->plugin_prefs = emalloc(sizeof(struct arglist));
      arg_dup(copy->plugin_prefs, plugin->plugin_prefs);
    }
  return copy;
}
