/* OpenVAS Client
 * Copyright (C) 1998,1999,2000 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**
 * @file
 * Functions to shows detail information on a NVT (description, tags, copyright
 * etc) and set its timeout.
 */

#include <includes.h>

#include <openvas/base/certificate.h> /* for certificate_t */

#ifdef USE_GTK
#include "gtk-compat.h"
#include <gtk/gtk.h>

#include "context.h"

#include "openvas_plugin.h"
#include "openvas_i18n.h"
#include "globals.h"
#include "preferences.h"
#include "error_dlg.h"
#include "plugin_infos.h"

/**
 * @brief Opens a dialog to show the dependencies.
 * 
 * @param foo  ignored (callback).
 * @param name Plugin name.
 */
static void
show_deps (GtkWidget * foo, char * name)
{
  GtkWindow * window = GTK_WINDOW(arg_get_value(MainDialog, "WINDOW"));
  GtkWidget * dialog;
  GtkWidget * vbox;
  GtkWidget * w;
  struct arglist * deps;
  gchar* lbl = NULL;
  char * label = NULL;
  lbl = g_strdup_printf (_("Dependencies of Plugin '%s'"), name);

  dialog = gtk_dialog_new_with_buttons(lbl, window,
                                       GTK_DIALOG_DESTROY_WITH_PARENT,
                                       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                                       NULL);
  gtk_window_set_resizable(GTK_WINDOW(dialog), TRUE);
  gtk_widget_realize(dialog);

  /* The overall vbox */
  vbox = GTK_DIALOG(dialog)->vbox;

  /* The titel as label again */
  w = gtk_label_new (lbl);
  g_free (lbl);
  gtk_box_pack_start(GTK_BOX(vbox), w, TRUE, TRUE, 5);
  gtk_misc_set_alignment((GtkMisc *)w, 0, 1);
  gtk_widget_show(w);

  /* A horizontal separator */
  w = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), w, TRUE, TRUE, 5);
  gtk_widget_show(w);

  deps = arg_get_value(Context->dependencies, name);

  /* In case we find no dependencies, create a hint on this */
  if (!deps || !deps->next)
  {
    w = gtk_label_new (_("No dependencies found."));
    gtk_box_pack_start(GTK_BOX(vbox), w,   TRUE, TRUE, 2);
    gtk_misc_set_alignment((GtkMisc *)w, 0, 1);
    gtk_widget_show(w);
  }

  while (deps && deps->next)
  {
    struct openvas_plugin * p = openvas_plugin_get_by_name (Context->plugins, deps->name);

    if(p == NULL)
      p = openvas_plugin_get_by_name(Context->scanners, deps->name);

    if (p != NULL )
      {
        // Like: "Identify unknown Services (Sevice Detection), currently disabled."
        label = g_strdup_printf ("%s (%s), %s %s.", deps->name,
                                                    nvti_family(p->ni),
                                                    _("currently"),
                                                    (p->enabled != 0) ? _("enabled") : _("disabled"));
        w = gtk_label_new (label);
      }
    else // like "Identify unknown Services"
      w = gtk_label_new (deps->name);

    gtk_box_pack_start(GTK_BOX(vbox), w,   TRUE, TRUE, 2);
    gtk_misc_set_alignment((GtkMisc *)w, 0, 1);
    gtk_widget_show(w);
    deps = deps->next;
    g_free (label);
  }

  /* execute and later remove the dialog */
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
}

/**
 * @brief Sets a timeout value to the server preferences in the currently
 *        selected context.
 *
 * @param oid     OID of NVT to set timeout for.
 * @param timeout Ideally, timeout in seconds (string will be copied).
 *
 * A timeout value is a key/value ['timeout.OID'/'5'] pair.
 */
static void 
do_set_timeout (char* oid, const char* timeout)
{
  struct arglist * serv_prefs = arg_get_value(Context->prefs, "SERVER_PREFS");
  char * pref;
  int type;

  pref = g_strdup_printf ("timeout.%s", oid);
  if ((type = arg_get_type(serv_prefs, pref)) >= 0)
    {
      char * old = arg_get_value(serv_prefs, pref);
      if (type == ARG_STRING)
        efree(&old);
      arg_set_type (serv_prefs, pref, ARG_STRING);
      arg_set_value (serv_prefs, pref, sizeof(int), estrdup(timeout));
    }
  else
    arg_add_value(serv_prefs, pref, ARG_STRING, sizeof(int), estrdup(timeout));
}

/**
 * @brief Sets the timeout registered in the preferences for a plugin to a
 * @brief GtkSpinButton.
 * 
 * @param spinner SpinButton to set to value timeout.
 * @param oid     OID of the NVT of which timeout to look up.
 */
static void
spin_to_timeout (GtkSpinButton * spinner, const char * oid)
{
  struct arglist * serv_prefs = arg_get_value (Context->prefs, "SERVER_PREFS");
  gchar * name = g_strdup_printf ("timeout.%s", oid);
  char * timeout = NULL;

  if (arg_get_type (serv_prefs, name) == ARG_STRING)
    {
      timeout = arg_get_value (serv_prefs, name);
      gtk_spin_button_set_value (spinner, atoi(timeout));
    }

  g_free (name);
}


/**
 * Callback for button click on "Show", bound to one certificate.
 * Shows a dialog with a text area (displays a public key) and a close button.
 *
 * @param fpr Fingerprint of certificate, used to retrieve the certificate from
 *            the contexts signer_fp_certificates hashtable.
 */
static void
showcert (GtkWidget* parent, certificate_t* cert)
{
  GtkWidget * win;
  GtkWidget * label, *separator, *textview, *content_area;
  GtkTextBuffer* textbuffer;
  int BUFFSIZE = 4096;
  char labelbuf[4096];

  snprintf(labelbuf, BUFFSIZE ,_("OpenVAS Certificate View: %s"), cert->owner);
  win = gtk_dialog_new_with_buttons(labelbuf,
                                         NULL,
                                         GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_STOCK_CLOSE,
                                         GTK_RESPONSE_ACCEPT,
                                         NULL);
  g_signal_connect_swapped (win,
                             "response",
                             G_CALLBACK (gtk_widget_destroy),
                             win);
  content_area = GTK_DIALOG (win)->vbox;

  /* Set up the label with owner information */
  snprintf(labelbuf, BUFFSIZE ,_("Owner Name: %s"), cert->owner);
  label = gtk_label_new(  labelbuf );
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_container_add (GTK_CONTAINER (content_area), label);

  snprintf(labelbuf, BUFFSIZE ,_("Fingeprint: %s"), cert->fingerprint);
  label = gtk_label_new( labelbuf );
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_container_add (GTK_CONTAINER (content_area), label);

  /* Add separator */
  separator = gtk_hseparator_new();
  gtk_container_add (GTK_CONTAINER (content_area), separator);

  /* Set up textview with public key text */
  textview = gtk_text_view_new();
  gtk_text_view_set_editable(GTK_TEXT_VIEW(textview), FALSE);
  textbuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
  gtk_text_buffer_set_text(textbuffer, cert->public_key, -1);
  gtk_container_add (GTK_CONTAINER (content_area), textview);

  gtk_widget_show_all(win);
  gtk_window_set_modal(GTK_WINDOW(win), TRUE);
}

/**
 * @brief Creates an untrusted certificate of owner "Unknown" with fingerprint fpr.
 *
 * @param fpr Fingerprint of certificate.
 *
 * @return Fresh underspecified certificate.
 */
static certificate_t*
new_unknown_cert (char* fpr)
{
  /** @todo Attention, creates a fresh one each time. */
  return certificate_create_full (fpr, _("Unknown"), NULL, FALSE);
}

/**
 * @brief Retrieves certificate_ts from the Contexts index and returns
 *        a list of these.
 *
 * @param txt Comma-separated list of fingerprints.
 *
 * @return A list of certificates, as found in the Contexts 
 *         signer_fp_certificates hashtable, or certificates with owner
 *         "Unknown" when not found.
 */
static GSList*
create_certificate_list (char* fprlist)
{
  /** @todo consider using certificates_t from base/certificates.c */
  GSList* certlist = NULL;

  // Init the hashtable if still not done.
  if (Context->signer_fp_certificates == NULL)
    Context->signer_fp_certificates = g_hash_table_new_full(g_str_hash,
                   g_str_equal, NULL, (GDestroyNotify) certificate_free);

  // Search each signature in Contexts store, if not found add a "Unknown" one.
  if (fprlist != NULL)
    {
      gchar** fprs = g_strsplit_set(fprlist, ",", -1);
      int idx = 0;

      // Look up fingerprint, add certificate to list
      while (fprs[idx] != NULL)
        {
          int key_offset = 0;

          // Client-side we index the certificate by the last 16 chars of its
          //  fingerprint. Sometimes it comes with more.
          if ( strlen (fprs[idx]) > 16)
            key_offset = strlen (fprs[idx]) - 16;

          certificate_t* cert = g_hash_table_lookup (Context->signer_fp_certificates,
                                                     fprs[idx] + key_offset);
          if (cert != NULL)
            certlist = g_slist_prepend (certlist, cert);
          else
            certlist = g_slist_prepend (certlist, new_unknown_cert(fprs[idx]));
          idx ++;
        }
      g_strfreev(fprs);
    }

  return certlist;
}


/**
 * @brief Callback, shows info dialog, using current Context as plugin source.
 *
 * If either no plugin list in current context is found or no name is provided,
 * does nothing.
 *
 * @param emitter ignored (callback)
 * @param name    Name of plugin to show info of.
 */
void
plugin_info_window_show_cb (GtkWidget* emitter, char* name)
{
  if (name != NULL || Context->plugins != NULL)
    plugin_info_window_setup (Context->plugins, name);
}


/**
 * @brief Draws the window which contains information about a plugin.
 *
 * @param res  Pointer to a openvas_plugin struct (list).
 * @param name The plugins name.
 */
void
plugin_info_window_setup (struct openvas_plugin * res, char* pluginname)
{
  GtkWindow * window = GTK_WINDOW(arg_get_value(MainDialog, "WINDOW"));
  GtkWidget * dialog;
  GtkWidget * vbox;
  GtkWidget * hbox;
  GtkWidget * label;
  GtkWidget * scrolledwindow;
  GtkWidget * textview;
  GtkWidget * timeoutspin;
  GtkWidget * timeoutlabel;
  GtkWidget * button;
  GtkWidget * separator;
  GtkTooltips * plugin_info_tips;

  char buf[4096];
  struct openvas_plugin * plugin;
  char * txt;

  plugin = openvas_plugin_get_by_name (res, pluginname);
  if (plugin == NULL)
  {
#ifdef DEBUG
    fprintf(stderr, _("Error ! Plugin selected not found ?!\n"));
#endif
    return;
  }

  dialog = gtk_dialog_new_with_buttons(pluginname, window,
                                       GTK_DIALOG_DESTROY_WITH_PARENT,
                                       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                                       NULL);
  gtk_window_set_resizable(GTK_WINDOW(dialog), TRUE);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);
  gtk_window_set_default_size(GTK_WINDOW(dialog), 350, 450);
  gtk_widget_realize(dialog);

  /* The overall vbox */
  vbox = GTK_DIALOG(dialog)->vbox;

  /* The plugin name */
  label = gtk_label_new(pluginname);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_label_set_selectable (GTK_LABEL(label), TRUE);
  gtk_widget_show(label);

  /* The copyright notice */
  label = gtk_label_new(nvti_copyright(plugin->ni));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE,FALSE,0);
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_label_set_selectable (GTK_LABEL(label), TRUE);
  gtk_widget_show(label);

  /* A separator */
  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE,0);
  gtk_widget_show(separator);

  /* The Family */
  snprintf(buf, sizeof(buf), _("Family: %s"), nvti_family(plugin->ni));
  label = gtk_label_new(buf);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_label_set_selectable (GTK_LABEL(label), TRUE);
  gtk_widget_show(label);

  /* The Category */
  snprintf(buf, sizeof(buf), _("Category: %s"), plugin->category);
  label = gtk_label_new(buf);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_label_set_selectable (GTK_LABEL(label), TRUE);
  gtk_widget_show(label);

  /* The NVT OID */
  snprintf(buf, sizeof(buf), _("OpenVAS NVT OID: %s"), plugin->oid);
  label = gtk_label_new(buf);
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_label_set_selectable (GTK_LABEL(label), TRUE);
  gtk_widget_show(label);

  /* The CVE ID (in case the plugin provides one) */
  txt = nvti_cve(plugin->ni);
  if( txt != NULL && txt[0] != '\0' && strcmp(txt, "NOCVE") != 0 )
  {
    snprintf(buf, sizeof(buf), _("CVE: %s"), txt);
    label = gtk_label_new(buf);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
    gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
    gtk_label_set_selectable (GTK_LABEL(label), TRUE);
    gtk_widget_show(label);
  }

  /* The Bugtraq ID (in case the plugin provides one) */
  txt = nvti_bid(plugin->ni);
  if( txt != NULL  && txt[0] != '\0' && strcmp(txt, "NOBID") != 0 )
  {
    snprintf(buf, sizeof(buf), _("Bugtraq ID: %s"), txt);
    label = gtk_label_new(buf);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
    gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
    gtk_label_set_selectable (GTK_LABEL(label), TRUE);
    gtk_widget_show(label);
  }

  /* The other cross references (in case the plugin provides one) */
  txt = nvti_xref(plugin->ni);
  if( txt != NULL && txt[0] != '\0' && strcmp(txt, "NOXREF") != 0 )
  {
    snprintf(buf, sizeof(buf), _("Other references: %s"), txt);
    label = gtk_label_new(buf);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
    gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
    gtk_label_set_selectable (GTK_LABEL(label), TRUE);
    gtk_widget_show(label);
  }

  /* The Version */
  txt = nvti_version(plugin->ni);
  if( txt != NULL  && txt[0] != '\0' )
  {
    snprintf(buf, sizeof(buf), _("Plugin Version: %s"), txt);
    label = gtk_label_new(buf);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE,FALSE,0);
    gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
    gtk_label_set_selectable (GTK_LABEL(label), TRUE);
    gtk_widget_show(label);
  }

  /* Script Tags */
  txt = nvti_tag(plugin->ni);
  if(txt != NULL && txt[0] != '\0' && strcmp(txt, "NOTAG") != 0)
  {
    snprintf(buf, sizeof(buf), _("Script tags: %s"), txt);
    label = gtk_label_new(buf);
    gtk_box_pack_start(GTK_BOX(vbox), label, FALSE,FALSE,0);
    gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
    gtk_label_set_selectable (GTK_LABEL(label), TRUE);
    gtk_widget_show(label);
  }

  /* A separator */
  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE,0);
  gtk_widget_show(separator);

  /* The titel for the scrolled text */
  label = gtk_label_new(_("Plugin description:"));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE,0);
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_widget_show(label);

  /* The scrolled window */
  scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow, TRUE, TRUE, 5);
  gtk_widget_show(scrolledwindow);

  /* The textview with the plugin description */
  textview = gtk_text_view_new();
  gtk_container_add(GTK_CONTAINER(scrolledwindow), textview);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(textview), FALSE);
  gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD);
  {
   char * s = nvti_description(plugin->ni);
  if ( s != NULL )
    gtk_text_buffer_set_text(gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview)),s, -1);
  }
  gtk_widget_show (textview);

  /* Add Separator */
  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);
  gtk_widget_show(separator);

  /* A hbox for server trust level information */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE,FALSE, 0);
  gtk_widget_show(hbox);

  /* Server trust level information */
  txt = nvti_sign_key_ids(plugin->ni);
  if(txt && strcmp(txt, "NOSIGNKEYS") == 0)
    txt = NULL;

  // Build up certificate list
  GSList* certificates = create_certificate_list (txt);

  // Create text for label 
  if(Context->socket <= 0)
    {
    snprintf(buf, sizeof(buf), _("Signature information available on server connection."));
    }
  else if(g_slist_length(certificates) == 0 && txt != NULL && strcmp(txt, "") != 0)
    {
    snprintf(buf, sizeof(buf), _("Signatures:\n\tUnknown signature(s)."));
    }
  else if(txt == NULL || strcmp(txt, "") == 0)
    {
    snprintf(buf, sizeof(buf), _("Signatures:\n\tNVT is not signed."));
    }
  else
    {
    struct arglist * serv_prefs = arg_get_value(Context->prefs, "SERVER_PREFS");
    // FIXME This defaults to a server that looks as if it verifies the 
    // signature. Due to double negation (no_check) and the representation of no
    // to 0 it is more or less impossible to check whether the preferences has
    // actually been set by the server or not.
    if ( serv_prefs != NULL
         && preferences_yes_or_one (serv_prefs, "nasl_no_signature_check") == FALSE)
      { // Server does not ignore signatures
      snprintf(buf, sizeof(buf), _("Signatures:"));
      }
    else // Server DOES ignore signatures
      snprintf(buf, sizeof(buf), _("Signatures (NOT verified):"));
    }

  label = gtk_label_new(buf);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
  gtk_misc_set_alignment((GtkMisc *)label, 0, 1);
  gtk_widget_show(label);

  // List of Certificates and buttons to view them, if any.
  while (certificates != NULL)
    {
    certificate_t* cert = certificates->data;
    // Label with certificate owner name
    hbox = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
    gtk_widget_show(hbox);
    char* trusted = (cert->trusted == TRUE)?
                      _("<span color=\"green\">trusted</span>"):
                      _("<span color=\"red\">untrusted</span>");
    snprintf(buf, sizeof(buf), "\t(%s) %s", trusted, cert->owner);
    label = gtk_label_new(NULL);
    gtk_label_set_markup(GTK_LABEL(label), buf);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
    gtk_widget_show(label);

    // Button to view Certificate
    if (strcmp(cert->owner, _("Unknown")) != 0)
      {
        button = gtk_button_new_with_label(_("View"));
        g_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(showcert),
                      cert);
        gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 5);
        gtk_widget_show(button);
      }
    certificates = g_slist_next (certificates);
    }

  /* Add Separator */
  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);
  gtk_widget_show(separator);

  /* A hbox for the timeout and dependency buttons */
  hbox = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
  gtk_widget_show(hbox);

  /* The label and spinner for setting the plugin timeout */
  timeoutlabel = gtk_label_new (_("Timeout in seconds"));
  gtk_box_pack_start(GTK_BOX(hbox), timeoutlabel, FALSE, FALSE, 5);
  gtk_widget_show (timeoutlabel);
  timeoutspin = gtk_spin_button_new_with_range (0.0, 60000.0, 2.0);
  gtk_box_pack_start(GTK_BOX(hbox), timeoutspin, FALSE, FALSE, 5);
  gtk_widget_show (timeoutspin);
  spin_to_timeout (GTK_SPIN_BUTTON(timeoutspin), plugin->oid);

  /* The button for displaying the dependencies */
  button = gtk_button_new_with_label(_("Show dependencies"));
  g_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(show_deps),
                   nvti_name(plugin->ni));
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 5);
  gtk_widget_show(button);
  if(!arg_get_value(Context->dependencies, pluginname))
    gtk_widget_set_sensitive(button,FALSE);

  // Set tooltip for timeout spinner
  plugin_info_tips = gtk_tooltips_new ();
  gtk_tooltips_set_tip (GTK_TOOLTIPS (plugin_info_tips), timeoutspin,
                                _("Set the timeout for this plugin, where zero "
                                   "means 'no timeout'. Note that a zero value "
                                   "can be overridden by the server."),"");
  /* Execute */
  gtk_dialog_run(GTK_DIALOG(dialog));
  /* Set the selected timeout and remove the dialog */
  do_set_timeout (plugin->oid, gtk_entry_get_text (GTK_ENTRY(timeoutspin)));
  gtk_widget_destroy(dialog);
}

#endif
