/* OpenVAS-Client
 * $Id$
 * Description: Implementation of the Severity Filter Managere Dialog
 *
 * Authors:
 * Jan-Oliver Wagner <jan-oliver.wagner@intevation.de>
 *
 * Copyright:
 * Copyright (C) 2009 Greenbone Networks GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * or, at your option, any later version as published by the Free
 * Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/** @file
 * 
 * Implementation of Severity Filter Manager Dialog.
 */

/* only for arglists */
#include <includes.h>

#include <gtk/gtk.h>
#include "context.h"

#include <openvas/base/severity_filter.h> /* for severity_filter_remove */

/* e.g. arg_get_value() */
#include <includes.h>

#include "error_dlg.h"
#include "openvas_i18n.h"
#include "listnotebook.h"
/* for prefs_get_openvashome */
#include "preferences.h"
#include "prefs_scope_tree.h"
#include "severity_override_form.h"

/* Needed to access the scopeview to influence GUI (icon,numbers) when a filter is applied */
extern struct arglist* MainDialog;
extern struct context* Context;

/**
 * @brief Add a page showing an override to a listnotebook.
 * 
 * @param override     severity_override to show information of.
 * @param listnotebook Listnotebook to add page to.
 */
static void
severityfiltermngr_dlg_add_override (const severity_override_t* override,
                                 GtkWidget* listnotebook)
{
  gchar* text = g_strdup_printf ("Name: %s\nOID: %s\nHost: %s\nPort: %s\nReason: %s\n"
                                 "mapped from %s to %s.",
                                  override->name, override->OID, override->host,
                                  override->port, override->reason,
                                  override->severity_from, override->severity_to);

  severity_override_form_t* form = severity_override_form_from_override (override, SOF_VIEW);
  gtk_widget_show_all (form->vbox);
  listnotebook_add_page (listnotebook, form->vbox, override->name, NULL);
  g_free (text);
}

/**
 * @brief Launches the dialog to manage the severity filter as menu call-back.
 * 
 * @param menuitem Call-back parameter for respective widget.
 * 
 * @param ctrls Call-back parameter (usually an arglist struct
 *              where the "WINDOW" is stored).
 */
void
severityfiltermanager_dialog (GtkMenuItem * menuitem, gpointer ctrls)
{
  GtkWindow *window = GTK_WINDOW(arg_get_value(ctrls, "WINDOW"));
  GtkWidget *dialog;
  GtkWidget* overrides_notebook;
  gint response;

  dialog = gtk_dialog_new_with_buttons (_("Severity Filter Manager"),
                                         window, GTK_DIALOG_DESTROY_WITH_PARENT,
                                         GTK_STOCK_DELETE, 12,
                                         GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                                         NULL);

  gtk_window_set_transient_for(GTK_WINDOW(dialog), window);

  gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);

  // Create the notebook showing overrides
  overrides_notebook = listnotebook_new(TRUE, TRUE);
  // For each override, add a notebookpage
  if (global_filter)
    g_slist_foreach (global_filter->overrides,
                     (GFunc) severityfiltermngr_dlg_add_override,
                     overrides_notebook);

  // Add listnotebook with keys
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), overrides_notebook,
                      TRUE, TRUE, 5);
  gtk_widget_show(overrides_notebook);

  response = gtk_dialog_run (GTK_DIALOG (dialog));

  // Handle the response
  while (response == 12) // Click on "Delete"
    {
      char* selected_override_name = listnotebook_get_selected (overrides_notebook);
      // Prototypical removal (@TODO Do not mess around with strings but manage list of overrides in the first place)
      GSList* walk = global_filter->overrides;
      while (selected_override_name && walk)
        {
          severity_override_t* override = walk->data;
          if (override && !strcmp(override->name, selected_override_name))
            {
              if (severity_filter_remove (global_filter, override) == FALSE)
                show_error (_("Error while removing override %s from global filter."), override->name);
              listnotebook_remove_page (overrides_notebook, selected_override_name);
              break;
            }
          walk = g_slist_next (walk);
        }
      response = gtk_dialog_run (GTK_DIALOG (dialog));
    }

  gtk_widget_destroy (dialog);

  // Refresh scope- tree
  prefs_scope_tree_update_report ();
}
