/* OpenVAS-Client
 * $Id$
 * Description: Common functions for lsc target preparation (e.g. credentials
 *              package generation).
 *
 * Authors:
 * Felix Wolfsteller <felix.wolfsteller@intevation.de>
 *
 * Copyright:
 * Copyright (C) 2009 Greenbone Networks GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * or, at your option, any later version as published by the Free
 * Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "error_dlg.h"
#include "file_utils.h"
#include "openvas_i18n.h"
#include "openvas_lsc_target_prep.h"
#include "openvas_lsc_user_deb.h"
#include "openvas_lsc_user_makensis.h"
#include "openvas_lsc_user_rpm.h"
#include "openvas_ssh_login.h"
#include "preferences.h" /* For prefs_get_openvashome */

/**
 * @brief Chars used with g_strcanon to produce valid file paths.
 * (duplicate in ssh_key_info_form)
 */
#define LEGAL_FILENAME_CHARS "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVXYZ1234567890_./"


/**
 * @brief Creates a temporary directory as working directory for installer and
 * @brief package generators.
 * 
 * Default directory is .openvas/lsc-mngt/.tmp .
 * 
 * @return Path to the temporary directory if directory successfully created or
 * @return exists, NULL in case of error (and will show the error).
 */
gchar*
openvas_lsc_target_prep_create_tmp_dir ()
{
  gchar* tmpdir = g_build_filename (prefs_get_openvashome(), ".openvas", "lsc-mngt",
                                    ".tmp", NULL);
  if (file_utils_ensure_dir(tmpdir) == FALSE)
    {
      show_error ("Could not create temporary dir %s.", tmpdir);
      g_free (tmpdir);
      tmpdir = NULL;
    }
  return tmpdir;
}


/**
 * @brief Creates a path for a installer or package for credentials for local
 * @brief security checks.
 *
 * Installers will be placed to or read from this location. (Default
 * $(HOME).openvas/lsc-credentials/openvas_lsc_target_preparation_accountname.extension)
 *
 * @param Name Name of the account (non-letters except for underscores and dots
 *             are replaced by underscores).
 * @param extension File extension for this installer/package.
 * 
 * @return Path to installer/package for given openvas_ssh_login, has to be
 *         freed using g_free, NULL in case of errors.
 */
gchar*
openvas_lsc_target_prep_filename (const char* accountname,
                                  const char* extension)
{
  gchar* filename;
  gchar* dir;
  gchar* path;

  if (accountname == NULL || extension == NULL)
    return NULL;

  filename = g_strdup_printf ("openvas_lsc_target_preparation_%s.%s", accountname, extension);
  g_strcanon (filename, LEGAL_FILENAME_CHARS, '_');

  dir = g_build_filename (prefs_get_openvashome(), ".openvas", "lsc-mngt", NULL);
  if (file_utils_ensure_dir (dir) == FALSE)
    {
      g_free (dir);
      g_free (filename);
      return NULL;
    }

  path = g_build_filename (dir, filename, NULL);
  g_free (filename);
  g_free (dir);
  return path;
}

/**
 * @brief Removes any installer or package that was generated for a given
 * @brief login.
 * 
 * @param login Login for which to remove all packages/installer.
 */
void
openvas_lsc_target_prep_remove (openvas_ssh_login* login)
{
      gchar* packname = NULL;
      packname = openvas_lsc_user_rpm_filename (login->name);
      remove (packname);
      g_free (packname);
      packname = openvas_lsc_user_deb_filename (login->name);
      remove (packname);
      g_free (packname);
      packname = openvas_lsc_user_makensis_filename (login->name);
      remove (packname);
      g_free (packname);
}
