###############################################################################
# 
#  Copyright (2008) Alexander Stukowski
#
#  This file is part of OVITO (Open Visualization Tool).
#
#  OVITO is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  OVITO is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
###############################################################################

# This is the CMake file that is used by the OVITO core as well as 
# by third-party plugin projects to adopt the core's build settings.

# Make sure we have a new version of CMake
CMAKE_MINIMUM_REQUIRED(VERSION 2.4 FATAL_ERROR)
IF(COMMAND cmake_policy)
	CMAKE_POLICY(SET CMP0003 NEW)
	IF(POLICY CMP0011)
		CMAKE_POLICY(SET CMP0011 NEW)
	ENDIF(POLICY CMP0011)
ENDIF(COMMAND cmake_policy)

# Include the OVITO revision information.
GET_FILENAME_COMPONENT(OVITO_BUILD_BASE_DIR ${CMAKE_CURRENT_LIST_FILE} PATH)
INCLUDE(${OVITO_BUILD_BASE_DIR}/cmake/Version.cmake)

IF(UNIX)
	IF(APPLE)
		SET(MACOSX_BUNDLE_NAME "Ovito") 
		# The directory where the main executable goes to.
		SET(OVITO_RELATIVE_BINARY_DIRECTORY ".")
		# The directory where the main libraries go to.
		SET(OVITO_RELATIVE_LIBRARY_DIRECTORY "${MACOSX_BUNDLE_NAME}.app/Contents/MacOS")
		# The directory where the auxiliary files go to.
		SET(OVITO_RELATIVE_SHARE_DIRECTORY "${MACOSX_BUNDLE_NAME}.app/Contents/Resources")
		# The directory where the compiled plugins go to.
		SET(OVITO_RELATIVE_PLUGINS_DIRECTORY "${OVITO_RELATIVE_LIBRARY_DIRECTORY}/plugins")
	ELSE(APPLE)
		# The directory where the main executable goes to.
		SET(OVITO_RELATIVE_BINARY_DIRECTORY "bin")
		# The directory where the main libraries go to.
		SET(OVITO_RELATIVE_LIBRARY_DIRECTORY "lib/ovito")
		# The directory where the auxiliary files go to.
		SET(OVITO_RELATIVE_SHARE_DIRECTORY "share/ovito")
		# The directory where the compiled plugins go to.
		SET(OVITO_RELATIVE_PLUGINS_DIRECTORY "${OVITO_RELATIVE_LIBRARY_DIRECTORY}/plugins")
	ENDIF(APPLE)
ELSE(UNIX)
	# The directory where the main executable goes to.
	SET(OVITO_RELATIVE_BINARY_DIRECTORY ".")
	# The directory where the main libraries go to.
	SET(OVITO_RELATIVE_LIBRARY_DIRECTORY ".")
	# The directory where the auxiliary files go to.
	SET(OVITO_RELATIVE_SHARE_DIRECTORY ".")
	# The directory where the compiled plugins go to.
	SET(OVITO_RELATIVE_PLUGINS_DIRECTORY "${OVITO_RELATIVE_LIBRARY_DIRECTORY}/plugins")
ENDIF(UNIX)

# Use the original build settings used to build OVITO for third-party plugin projects as well. 
IF(${PROJECT_NAME} STREQUAL "ovito")

	# We are building the original OVITO package.
	# So let the user choose the build options. 

	# Static or shared (and extensible) version of Ovito?
	OPTION(OVITO_MONOLITHIC_BUILD "Create one big executable that contains all plugins and third-party libraries." "OFF")
	
	# Let the user enable double-precision floating point values.
	OPTION(OVITO_USE_DOUBLE_PRECISION_FP "Use double precision 64-bit floating point values for all calculations." "OFF")
	
	# Controls the building of the Scripting plugin and the 
	# scripting support of other plugins.
	OPTION(OVITO_BUILD_PLUGIN_SCRIPTING "Build the Scripting plugin." "OFF")
	
	# Controls the building of the graphical user interface of the Scripting plugin.
	OPTION(OVITO_BUILD_PLUGIN_SCRIPTING_GUI "Build the graphical user inface for the Scripting plugin." "OFF")
	
	# Controls the creation of the POV-Ray plugin.
	OPTION(OVITO_BUILD_PLUGIN_POVRAY "Build the POV-Ray plugin." "ON")

	# Controls the creation of the RenderMan plugin.
	OPTION(OVITO_BUILD_PLUGIN_RENDERMAN "Build the RenderMan plugin." "OFF")
	
	# Controls the creation of the standard objects plugin.
	OPTION(OVITO_BUILD_PLUGIN_STDOBJECTS "Build the Standard Objects plugin." "ON")

	# Controls the creation of the AtomViz plugin.
	OPTION(OVITO_BUILD_PLUGIN_ATOMVIZ "Build the AtomViz plugin." "ON")

	# Controls the creation of the CrystalAnalysis plugin.
	OPTION(OVITO_BUILD_PLUGIN_CRYSTALANALYSIS "Build the  CrystalAnalysis plugin." "OFF")
	
	# Controls the generation of the documentation files.
	OPTION(OVITO_BUILD_DOCUMENTATION "Build the documentation files" "OFF")

ELSE(${PROJECT_NAME} STREQUAL "ovito")

	# We are building a plugin project.
	# So just copy the build settings from the OVITO core project. 
	
	LOAD_CACHE("${Ovito_DIR}/" READ_WITH_PREFIX OVITO_ 
		OVITO_MONOLITHIC_BUILD
		OVITO_USE_DOUBLE_PRECISION_FP
		OVITO_BUILD_PLUGIN_SCRIPTING
		OVITO_BUILD_PLUGIN_SCRIPTING_GUI
		OVITO_BUILD_PLUGIN_POVRAY
		OVITO_BUILD_PLUGIN_STDOBJECTS
		OVITO_BUILD_PLUGIN_ATOMVIZ
		ovito_SOURCE_DIR
		ovito_BINARY_DIR
		CMAKE_INSTALL_PREFIX
	)
	SET(OVITO_USE_DOUBLE_PRECISION_FP ${OVITO_OVITO_USE_DOUBLE_PRECISION_FP})	
	SET(OVITO_BUILD_PLUGIN_SCRIPTING ${OVITO_OVITO_BUILD_PLUGIN_SCRIPTING})	
	SET(OVITO_BUILD_PLUGIN_SCRIPTING_GUI ${OVITO_OVITO_BUILD_PLUGIN_SCRIPTING_GUI})
	SET(OVITO_BUILD_PLUGIN_POVRAY ${OVITO_OVITO_BUILD_PLUGIN_POVRAY})	
	SET(OVITO_BUILD_PLUGIN_RENDERMAN ${OVITO_OVITO_BUILD_PLUGIN_RENDERMAN})	
	SET(OVITO_BUILD_PLUGIN_STDOBJECTS ${OVITO_OVITO_BUILD_PLUGIN_STDOBJECTS})	
	SET(OVITO_BUILD_PLUGIN_ATOMVIZ ${OVITO_OVITO_BUILD_PLUGIN_ATOMVIZ})	
	SET(OVITO_BUILD_PLUGIN_CRYSTALANALYSIS ${OVITO_BUILD_PLUGIN_CRYSTALANALYSIS})	
	SET(ovito_SOURCE_DIR ${OVITO_ovito_SOURCE_DIR})
	SET(ovito_BINARY_DIR ${OVITO_ovito_BINARY_DIR})
	SET(CMAKE_INSTALL_PREFIX ${OVITO_CMAKE_INSTALL_PREFIX} CACHE PATH "Installation path for the plugin. This must point to the OVITO installation path." FORCE)

	# Building plugins makes only sense when Ovito has been built as a shared library.
	IF(OVITO_OVITO_MONOLITHIC_BUILD)
		MESSAGE(FATAL_ERROR "OVITO has been built as a monolithic executable. It cannot be extended via plugins in this configuration.")
	ENDIF(OVITO_OVITO_MONOLITHIC_BUILD)
	
	# Add OVITO's library directory to the library path. Plugins will always have
	# to be linked against the "Core" and "Base" Ovito libraries found here.
	LINK_DIRECTORIES(${ovito_BINARY_DIR}/${OVITO_RELATIVE_LIBRARY_DIRECTORY})
	
ENDIF(${PROJECT_NAME} STREQUAL "ovito")

# Create the plugin and core libraries as shared objects if we don't build a monolithic executable.
IF(OVITO_MONOLITHIC_BUILD)
	SET(BUILD_SHARED_LIBS "OFF")

	# Let the source code know that we're building a monolithic executable without
	# plugins residing in shared libraries.
	ADD_DEFINITIONS(-DOVITO_MONOLITHIC_BUILD)
	
	# No need to export Python module initialization functions in plugins.
	ADD_DEFINITIONS(-DBOOST_PYTHON_STATIC_MODULE)

ELSE(OVITO_MONOLITHIC_BUILD)
	SET(BUILD_SHARED_LIBS "ON")

	# Setup RPATH
	SET(CMAKE_SKIP_BUILD_RPATH FALSE)
	# When building, use the install RPATH already
	SET(CMAKE_BUILD_WITH_INSTALL_RPATH TRUE)
	# The RPATH to be used when building and installing
	SET(CMAKE_INSTALL_RPATH "\$ORIGIN/../lib/ovito")
	# add the automatically determined parts of the RPATH
	# which point to directories outside the build tree to the install RPATH
	SET(CMAKE_INSTALL_RPATH_USE_LINK_PATH TRUE)

ENDIF(OVITO_MONOLITHIC_BUILD)

# The directory where the main executable goes to.
SET(OVITO_BINARY_DIRECTORY "${ovito_BINARY_DIR}/${OVITO_RELATIVE_BINARY_DIRECTORY}")
# The directory where the main libraries go to.
SET(OVITO_LIBRARY_DIRECTORY "${ovito_BINARY_DIR}/${OVITO_RELATIVE_LIBRARY_DIRECTORY}")
# The directory where the compiled plugins go to.
SET(OVITO_PLUGINS_DIRECTORY "${ovito_BINARY_DIR}/${OVITO_RELATIVE_PLUGINS_DIRECTORY}")
# The directory where the auxiliary files go to.
SET(OVITO_SHARE_DIRECTORY "${ovito_BINARY_DIR}/${OVITO_RELATIVE_SHARE_DIRECTORY}")
# Add the plugins directory to the library path. Maybe some plugins link against other plugins.
LINK_DIRECTORIES(${OVITO_PLUGINS_DIRECTORY})

# The Eclipse CDT Error Parser cannot handle error messages that span more than one line, 
# which is the default GCC behavior. Force GCC to generate single line error messages 
# with no line wrapping:
IF(CMAKE_COMPILER_IS_GNUCXX)
  SET(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -fmessage-length=0")
ENDIF(CMAKE_COMPILER_IS_GNUCXX)

# Define some macro symbols for debug and release builds.
SET(CMAKE_CXX_FLAGS_DEBUG "${CMAKE_CXX_FLAGS_DEBUG} -D_DEBUG")
SET(CMAKE_CXX_FLAGS_RELEASE "${CMAKE_CXX_FLAGS_RELEASE} -DQT_NO_DEBUG=1 -DQT_NO_DEBUG_OUTPUT=1")
SET(CMAKE_CXX_FLAGS_RELWITHDEBINFO "${CMAKE_CXX_FLAGS_RELWITHDEBINFO} -DQT_NO_DEBUG=1 -DQT_NO_DEBUG_OUTPUT=1")

# Special flag needed to create proper dynamic link libraries using the MinGW compiler.
IF(MINGW)
	SET(CMAKE_SHARED_LINKER_FLAGS "${CMAKE_SHARED_LINKER_FLAGS} -Wl,--disable-auto-import")
ENDIF(MINGW)

# Let the source code know whether we are using 32 or 64 bit floating-point precision numbers.
IF(OVITO_USE_DOUBLE_PRECISION_FP)
	ADD_DEFINITIONS(-DOVITO_USE_DOUBLE_PRECISION_FP)
ENDIF(OVITO_USE_DOUBLE_PRECISION_FP)

# Find and setup Qt4 for this project
SET(QT_USE_QTOPENGL TRUE)
SET(QT_USE_QTXML TRUE)
SET(QT_MIN_VERSION "4.4.0")
# Use the Q_SIGNALS and Q_SLOTS macros to avoid name conflicts with Python.
ADD_DEFINITIONS(-DQT_NO_KEYWORDS)
FIND_PACKAGE(Qt4) 
IF(NOT QT4_FOUND)
	IF(QT4_INSTALLED_VERSION_TOO_OLD)
		MESSAGE(FATAL_ERROR "The installed Qt version ${QTVERSION} is too old, at least version ${QT_MIN_VERSION} is required. Please update the installed Qt package or download and compile Qt manually and specify the path to the 'qmake' program in the QT_QMAKE_EXECUTABLE setting of CMake.")
	ELSE(QT4_INSTALLED_VERSION_TOO_OLD)
		MESSAGE(FATAL_ERROR "The Qt library or the qmake program was not found! Please install Qt manually and specify the path to the 'qmake' program in the QT_QMAKE_EXECUTABLE setting of CMake. You need at least version ${QT_MIN_VERSION} of Qt to build OVITO.")
	ENDIF(QT4_INSTALLED_VERSION_TOO_OLD)
ENDIF(NOT QT4_FOUND)
#IF(OVITO_MONOLITHIC_BUILD)
#	SET(QT_QTOPENGL_LIB_DEPENDENCIES "-Wl,-Bdynamic ${QT_QTOPENGL_LIB_DEPENDENCIES} -Wl,-Bstatic")
#ENDIF(OVITO_MONOLITHIC_BUILD)
INCLUDE(${QT_USE_FILE})	
INCLUDE(${OVITO_BUILD_BASE_DIR}/cmake/CompileQtResource.cmake)

# Find the OpenGL library.
FIND_PACKAGE(OpenGL REQUIRED)
IF(NOT OPENGL_FOUND)
	MESSAGE("OPENGL_gl_LIBRARY=${OPENGL_gl_LIBRARY}")
	MESSAGE("OPENGL_INCLUDE_DIR=${OPENGL_INCLUDE_DIR}")
	MESSAGE(FATAL_ERROR "OpenGL library not found.")
ENDIF(NOT OPENGL_FOUND)
SET(OVITO_INCLUDE_PATHS ${OPENGL_INCLUDE_DIR})

# Find the Boost library.
SET(Boost_ADDITIONAL_VERSIONS "1.41" "1.41.0" "1.42" "1.42.0")
SET(REQUIRED_BOOST_COMPONENTS iostreams)
SET(Boost_USE_STATIC_LIBS ${OVITO_MONOLITHIC_BUILD})
SET(Boost_USE_MULTITHREAD ON)
IF(NOT OVITO_MONOLITHIC_BUILD)
	ADD_DEFINITIONS(-DBOOST_ALL_DYN_LINK)
ENDIF(NOT OVITO_MONOLITHIC_BUILD)
IF(OVITO_BUILD_PLUGIN_SCRIPTING)
	SET(REQUIRED_BOOST_COMPONENTS ${REQUIRED_BOOST_COMPONENTS} python)
ENDIF(OVITO_BUILD_PLUGIN_SCRIPTING)
FIND_PACKAGE(Boost COMPONENTS ${REQUIRED_BOOST_COMPONENTS} REQUIRED)
IF(NOT Boost_FOUND)
	MESSAGE(FATAL_ERROR "The Boost library could not be found on your system. ${Boost_ERROR_REASON}")
ENDIF(NOT Boost_FOUND)
LINK_DIRECTORIES(${Boost_LIBRARY_DIRS})
SET(OVITO_INCLUDE_PATHS ${OVITO_INCLUDE_PATHS} ${Boost_INCLUDE_DIRS})

# The old version CMake 2.4 cannot detect the required Boost library files on its own.
# We have to explicitely specify the library filenames.
IF(NOT ${CMAKE_MAJOR_VERSION} GREATER 2 AND NOT ${CMAKE_MINOR_VERSION} GREATER 4)
	SET(Boost_LIBRARIES "boost_iostreams")
	SET(Boost_PYTHON_LIBRARY "boost_python")
ENDIF(NOT ${CMAKE_MAJOR_VERSION} GREATER 2 AND NOT ${CMAKE_MINOR_VERSION} GREATER 4)

# Add the src/ directory to the global include path because that's the base directory for
# all include statements used in Ovito's source files.
SET(OVITO_INCLUDE_PATHS ${OVITO_INCLUDE_PATHS} "${ovito_SOURCE_DIR}/src")

IF(OVITO_BUILD_PLUGIN_SCRIPTING)
	# Make sure Python is installed.
	FIND_PACKAGE(PythonLibs REQUIRED)
	INCLUDE_DIRECTORIES(${PYTHON_INCLUDE_PATH})
	
	# Modules that use Python have to link libutil too.
	IF(NOT WIN32)
		SET(PYTHON_UTIL_LIBRARY "util" "dl" "pthread")
	ELSE(NOT WIN32)
		# No libutil needed on Windows
		SET(PYTHON_UTIL_LIBRARY "")
	ENDIF(NOT WIN32)

	IF(NOT OVITO_MONOLITHIC_BUILD)
		SET(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -DBOOST_PYTHON_DYNAMIC_LIB -DBOOST_PYTHON_DYNAMIC_MODULE")
	ENDIF(NOT OVITO_MONOLITHIC_BUILD)
	
	IF(CMAKE_COMPILER_IS_GNUCXX)
		SET(CMAKE_CXX_FLAGS "${CMAKE_CXX_FLAGS} -ftemplate-depth-100")
	ENDIF(CMAKE_COMPILER_IS_GNUCXX)	

	ADD_DEFINITIONS(-DOVITO_BUILD_PLUGIN_SCRIPTING=1)
ENDIF(OVITO_BUILD_PLUGIN_SCRIPTING)

# This macro copies a complete directory tree but skips paths with ".svn", which are produced by Subversion.
MACRO(COPY_DIRECTORY_WITHOUT_SVN source destination)
	FILE(MAKE_DIRECTORY "${destination}")
	FILE(GLOB_RECURSE FILES_TO_BE_COPIED "${source}/*")
	FOREACH(path ${FILES_TO_BE_COPIED})	
		GET_FILENAME_COMPONENT(filename ${path} NAME)
		IF(NOT ${path} MATCHES ".svn")
			FILE(RELATIVE_PATH REL_PATH "${source}/" ${path})
			CONFIGURE_FILE(${path} "${destination}/${REL_PATH}" COPYONLY)
		ENDIF(NOT ${path} MATCHES ".svn")
	ENDFOREACH(path)
ENDMACRO(COPY_DIRECTORY_WITHOUT_SVN)
