///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/ApplicationManager.h>
#include <core/viewport/ViewportManager.h>
#include "AtomsRenderer.h"

namespace AtomViz {

/// The size of the texture used for billboard rendering in pixels.
#define BILLBOARD_TEXTURE_SIZE	 128

/// The default chunk size for atom rendering, i.e. the number of atoms that are sent to the
/// graphics card at a time.
#define DEFAULT_RENDER_CHUNK_SIZE	 8000000

bool AtomsRenderer::_enableHWShaders = true;
bool AtomsRenderer::_enablePointExtension = true;
bool AtomsRenderer::_settingsLoaded = false;

/******************************************************************************
* Loads the render settings from the application settings store.
******************************************************************************/
void AtomsRenderer::loadRenderSettings()
{
	// Don't use hardware shaders on Intel graphic chipsets or on the Mesa software rasterizer.
	// The OpenGL GLSL support seems to be buggy.
	if(VIEWPORT_MANAGER.activeViewport()) {
		const QByteArray& rendererName = VIEWPORT_MANAGER.activeViewport()->rendererName();
		if(rendererName.contains("Mesa DRI Intel") || rendererName.contains("Software Rasterizer"))
			_enableHWShaders = false;
	}

	QSettings settings;
	settings.beginGroup("atomviz/rendering/");
	_enableHWShaders = settings.value("UseHWShaders", qVariantFromValue(_enableHWShaders)).toBool();
	_enablePointExtension = settings.value("UsePointExtension", qVariantFromValue(_enablePointExtension)).toBool();
	settings.endGroup();

	_settingsLoaded = true;
}

/******************************************************************************
* Returns whether the use of the OpenGL point parameters extension is enabled for atom rendering.
******************************************************************************/
bool AtomsRenderer::arePointSpritesEnabled()
{
	if(!_settingsLoaded) loadRenderSettings();
	return _enablePointExtension;
}

/******************************************************************************
* Sets whether the use of the OpenGL point parameters extension should be enabled for atom rendering.
******************************************************************************/
void AtomsRenderer::enablePointSprites(bool enable)
{
	if(enable == _enablePointExtension) return;
	_enablePointExtension = enable;
	QSettings settings;
	settings.beginGroup("atomviz/rendering/");
	settings.setValue("UsePointExtension", _enablePointExtension);
	settings.endGroup();
}

/******************************************************************************
* Returns whether the use of OpenGL shader programs is enabled for atom rendering.
******************************************************************************/
bool AtomsRenderer::areHWShadersEnabled()
{
	if(!_settingsLoaded) loadRenderSettings();
	return _enableHWShaders;
}

/******************************************************************************
* Sets whether the use of OpenGL shader programs should be enabled for atom rendering.
******************************************************************************/
void AtomsRenderer::enableHWShaders(bool enable)
{
	if(enable == _enableHWShaders) return;
	_enableHWShaders = enable;
	QSettings settings;
	settings.beginGroup("atomviz/rendering/");
	settings.setValue("UseHWShaders", _enableHWShaders);
	settings.endGroup();
}

/******************************************************************************
* Default constructor.
******************************************************************************/
AtomsRenderer::AtomsRenderer()
	: container(NULL), uniformRadius(0), currentAtom(NULL), hasBeenFilled(false),
	vboVerticesID(0), flatImposterShader(NULL), shadedImposterShader(NULL), raytracedSphereShader(NULL),
	chunkRenderSize(DEFAULT_RENDER_CHUNK_SIZE)
{
	textureID[0] = textureID[1] = 0;
}

/******************************************************************************
* Destructor.
******************************************************************************/
AtomsRenderer::~AtomsRenderer()
{
	reset();
}

/******************************************************************************
* Release all cached OpenGL resources.
******************************************************************************/
void AtomsRenderer::reset()
{
	if(container && (textureID[0] || vboVerticesID)) {
		QGLContext *oldContext = const_cast<QGLContext*>(QGLContext::currentContext());
		container->makeCurrent();
		// Release texture.
		if(textureID[0])
			CHECK_OPENGL(glDeleteTextures(2, textureID));
		// Release vertex buffer object.
		if(vboVerticesID)
			CHECK_OPENGL(container->glDeleteBuffersARB(1, &vboVerticesID));
		container = NULL;
		if(oldContext) oldContext->makeCurrent();
	}
	textureID[0] = 0;
	textureID[1] = 0;
	vboVerticesID = 0;
	flatImposterShader = NULL;
	shadedImposterShader = NULL;
	raytracedSphereShader = NULL;
	hasBeenFilled = false;
}

/******************************************************************************
* Initializes the internal buffers of this helper class if they have not been initialized yet.
* It will be associated with the given rendering window and may subsequently
* only be used with rendering windows that belong to the same Window3DContainer.
******************************************************************************/
void AtomsRenderer::prepare(Window3D* win, bool flatShading, bool useImposters)
{
	OVITO_ASSERT_MSG(APPLICATION_MANAGER.guiMode(), "AtomsRenderer::prepare()", "Rendering of atoms is only supported in GUI mode.");
	CHECK_POINTER(win);

	if(flatShading) useImposters = true;
	if(container == NULL ||
			(win->isSharingWith(container) == false && container != win) ||
			useImposters != this->useImposters ||
			flatShading != this->flatShading) {
		reset();
	}
	this->useImposters = useImposters;
	this->flatShading = flatShading;

	container = win;

	// Allocate vertex buffer object.
	container->makeCurrent();
	if(arePointSpritesEnabled() && !vboVerticesID && container->hasVertexBufferObjectsExtension() &&
			container->hasPointParametersExtension() && useImposters) {
		CHECK_OPENGL(container->glGenBuffersARB(1, &vboVerticesID));
	}

	// Create OpenGL shaders.
	if(areHWShadersEnabled() && container->hasShaderObjectsExtension() &&
			container->hasVertexShaderExtension() && container->hasFragmentShaderExtension()) {

		// The shaders are only created once during the lifetime of the rendering window.
		// After that they are stored along with the rendering window and will be used by all
		// instances of the AtomsRenderer class.

		try {
			flatImposterShader = container->findChild<OpenGLShader*>("pointsprite.flat.atom.shader");
			if(flatImposterShader == NULL) {
				flatImposterShader = new OpenGLShader(container, "pointsprite.flat.atom.shader");
				flatImposterShader->loadShader(":/atomviz/pointsprite.atom.vertexshader", ":/atomviz/flat.atom.fragmentshader");
				OVITO_ASSERT(container->findChild<OpenGLShader*>("pointsprite.flat.atom.shader") == flatImposterShader);
			}

			shadedImposterShader = container->findChild<OpenGLShader*>("pointsprite.shaded.atom.shader");
			if(shadedImposterShader == NULL) {
				shadedImposterShader = new OpenGLShader(container, "pointsprite.shaded.atom.shader");
				shadedImposterShader->loadShader(":/atomviz/pointsprite.atom.vertexshader", ":/atomviz/shaded.atom.fragmentshader");
				OVITO_ASSERT(container->findChild<OpenGLShader*>("pointsprite.shaded.atom.shader") == shadedImposterShader);
			}

			raytracedSphereShader = container->findChild<OpenGLShader*>("raytraced.atom.shader");
			if(raytracedSphereShader == NULL) {
				raytracedSphereShader = new OpenGLShader(container, "raytraced.atom.shader");
				raytracedSphereShader->loadShader(":/atomviz/raytraced.atom.vertexshader", ":/atomviz/raytraced.atom.fragmentshader");
				OVITO_ASSERT(container->findChild<OpenGLShader*>("raytraced.atom.shader") == raytracedSphereShader);
			}
		}
		catch(const Exception& ex) {
			ex.logError();
			flatImposterShader = NULL;
			shadedImposterShader = NULL;
			raytracedSphereShader = NULL;
		}
	}
}

/******************************************************************************
* Allocates a memory buffer for the specified number of atoms.
* After all atoms have been specified, EndAtoms() must be called.
******************************************************************************/
void AtomsRenderer::beginAtoms(GLuint numAtoms)
{
	OVITO_ASSERT_MSG(!container.isNull(), "AtomsRenderer::BeginAtoms()", "The atoms renderer has not been initialized using the Prepare() method.");
	if(container == NULL)
		throw Exception("The atoms renderer has not been initialized using the Prepare() method.");
	currentAtom = NULL;

	this->numAtoms = numAtoms;
	OVITO_ASSERT(numAtoms >= 0);
	if(numAtoms == 0) return;

	if(vboVerticesID) {
		// Store atoms in GPU memory.
		internalArray.clear();
		CHECK_OPENGL(container->glBindBufferARB(GL_ARRAY_BUFFER_ARB, vboVerticesID));
		CHECK_OPENGL(container->glBufferDataARB(GL_ARRAY_BUFFER_ARB, numAtoms*sizeof(OpenGLAtom), NULL, GL_STREAM_DRAW_ARB));
		CHECK_OPENGL(currentAtom = (OpenGLAtom*)container->glMapBufferARB(GL_ARRAY_BUFFER_ARB, GL_WRITE_ONLY_ARB));
		CHECK_POINTER(currentAtom);
	}
	else {
		// Use internal atom buffer.
		// Resize internal array.
		if(numAtoms != internalArray.size()) {
			internalArray.clear();
			internalArray.resize(numAtoms);
		}
		currentAtom = internalArray.data();
	}

	uniformRadius = -1;
	maxRadius = 0;
	_boundingBox = Box3();
}

/******************************************************************************
* Stores a single atom in the internal buffer. This function may only be called
* N times between a call to BeginAtoms() and EndAtoms();
******************************************************************************/
void AtomsRenderer::specifyAtom(const Point3& pos, GLubyte r, GLubyte g, GLubyte b, FloatType radius)
{
	OVITO_ASSERT_MSG(currentAtom != NULL, "AtomsRenderer::specifyAtom()", "This function may only be called between a call to beginAtoms() and endAtoms().");

	currentAtom->x = (float)pos.X;
	currentAtom->y = (float)pos.Y;
	currentAtom->z = (float)pos.Z;
	currentAtom->r = r;
	currentAtom->g = g;
	currentAtom->b = b;
	currentAtom->a = 255;
	currentAtom->radius = (float)radius;

	if(uniformRadius == -1) uniformRadius = radius;
	else if(uniformRadius != radius) uniformRadius = 0;

	if(radius > maxRadius) maxRadius = radius;

	_boundingBox.addPoint(pos);

	++currentAtom;
}

/******************************************************************************
* This method must be called after BeginAtoms() has been called and all atom
* data has been written to the supplied buffer.
******************************************************************************/
void AtomsRenderer::endAtoms()
{
	if(uniformRadius == -1) uniformRadius = 0;
	_boundingBox = _boundingBox.padBox(maxRadius);

	if(currentAtom) {
		if(vboVerticesID) {
			container->glUnmapBufferARB(GL_ARRAY_BUFFER_ARB);
			container->glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
		}
		currentAtom = NULL;
	}

	hasBeenFilled = true;
}

/******************************************************************************
* Renders the buffered atoms in the given rendering window.
******************************************************************************/
void AtomsRenderer::render(Window3D* win)
{
	CHECK_POINTER(win);
	OVITO_ASSERT_MSG(!container.isNull(), "AtomsRenderer::render()", "The atoms renderer has not been initialized using the Prepare() method.");
	OVITO_ASSERT_MSG(container == win, "AtomsRenderer::render()", "The AtomsRenderer object may only be used with the associated Window3D.");
	OVITO_ASSERT_MSG(isFilled(), "AtomsRenderer::render()", "Do not call AtomsRenderer::render() before filling the render buffer with atoms.");

	if(!win->isRendering())
		return;

	if(container == NULL)
		throw Exception("The atoms renderer has not been initialized using the prepare() method.");
	if(container != win)
		throw Exception("The AtomsRenderer object may only be used with the associated Window3D.");

	if(numAtoms == 0) return;

	// Choose rendering method.
	if(useImposters && container->hasPointParametersExtension() && shadedImposterShader && shadedImposterShader->isValid()) {
		initializeBillboardTextures(false);
		renderInternalImpostersWithShader(win->isPerspectiveProjection(), win->projectionMatrix(), win->viewportRectangle().height());
	}
	else if(!useImposters && raytracedSphereShader && raytracedSphereShader->isValid())
		renderInternalRaytracedWithShaders(win->isPerspectiveProjection());
	else {
		initializeBillboardTextures();
		if(arePointSpritesEnabled() && container->hasPointParametersExtension())
			renderInternalArrayPointSprites(win->isPerspectiveProjection(), win->projectionMatrix(), win->viewportRectangle().height());
		else
			renderInternalArrayQuads();
	}

	win->enlargeSceneExtentOS(boundingBox());
}

/******************************************************************************
* Render the buffered atoms to an offscreen buffer.
* This function is used by the ambient lighting modifier.
******************************************************************************/
void AtomsRenderer::renderOffscreen(bool isPerspective, const Matrix4& projMatrix, QSize windowSize)
{
	OVITO_ASSERT_MSG(!container.isNull(), "AtomsRenderer::renderOffscreen()", "The atoms renderer has not been initialized using the Prepare() method.");

	if(numAtoms == 0) return;

	// Choose rendering method.
	if(useImposters && container->hasPointParametersExtension() && shadedImposterShader && shadedImposterShader->isValid()) {
		initializeBillboardTextures(false);
		renderInternalImpostersWithShader(isPerspective, projMatrix, windowSize.height());
	}
	else if(!useImposters && raytracedSphereShader && raytracedSphereShader->isValid())
		renderInternalRaytracedWithShaders(isPerspective);
	else {
		initializeBillboardTextures();
		if(container->hasPointParametersExtension())
			renderInternalArrayPointSprites(isPerspective, projMatrix, windowSize.height());
		else
			renderInternalArrayQuads();
	}
}

/******************************************************************************
* Creates and activates the texture used for billboard rendering of atoms.
******************************************************************************/
void AtomsRenderer::initializeBillboardTextures(bool withAlpha)
{
	glEnable(GL_TEXTURE_2D);
	glAlphaFunc(GL_GREATER, 0);
	glEnable(GL_ALPHA_TEST);

	// Generate texture image.
	static GLubyte shadedImage[BILLBOARD_TEXTURE_SIZE][BILLBOARD_TEXTURE_SIZE][4];
	static GLubyte flatImage[BILLBOARD_TEXTURE_SIZE][BILLBOARD_TEXTURE_SIZE][4];
	static bool generatedImage = false;
	if(generatedImage == false) {
		generatedImage = true;
		GLubyte* shadedPixel = shadedImage[0][0];
		GLubyte* flatPixel = flatImage[0][0];
		for(int dx=-BILLBOARD_TEXTURE_SIZE/2; dx<BILLBOARD_TEXTURE_SIZE/2; dx++) {
			for(int dy=-BILLBOARD_TEXTURE_SIZE/2; dy<BILLBOARD_TEXTURE_SIZE/2; dy++, shadedPixel += 4, flatPixel += 4) {
				int dist = dx*dx + dy*dy;
				int diff = BILLBOARD_TEXTURE_SIZE*BILLBOARD_TEXTURE_SIZE/4 - dist;
				if(diff >= 0 || !withAlpha) {

					Vector3 normal(dx, dy, 0);

					normal.Z = sqrt((FloatType)max(diff, 0));
					normal = Normalize(normal);
					Vector3 camera_lightdir(Normalize(Vector3(-0.3, -0.3, 1.0)));
					FloatType diffuse = abs(DotProduct(normal, camera_lightdir)) * 0.8;
					FloatType shininess = 6.0;
					Vector3 reflect = camera_lightdir - camera_lightdir * (2.0 * DotProduct(camera_lightdir, normal));
					FloatType specular = pow(max((FloatType)0.0, -reflect.Z), shininess) * 0.25;
					FloatType ambient = 0.1;
					FloatType colorValue = (diffuse + ambient);
					if(withAlpha) colorValue += specular;

					GLubyte color = (GLubyte)(min(colorValue, (FloatType)1.0) * 255.0);
					shadedPixel[0] = shadedPixel[1] = shadedPixel[2] = color;
					if(!withAlpha)
						shadedPixel[3] = GLubyte(min(specular, (FloatType)1.0) * 255.0);
					else
						shadedPixel[3] = 255;
					flatPixel[0] = flatPixel[1] = flatPixel[2] = flatPixel[3] = 255;
				}
				else {
					shadedPixel[0] = shadedPixel[1] = shadedPixel[2] = shadedPixel[3] = 0;
					flatPixel[0] = flatPixel[1] = flatPixel[2] = flatPixel[3] = 0;
				}
			}
		}
	}

	// Create OpenGL textures
	if(!textureID[0]) {
		glGenTextures(2, textureID);

		CHECK_OPENGL(glBindTexture(GL_TEXTURE_2D, textureID[0]));
		CHECK_OPENGL(glTexImage2D(GL_TEXTURE_2D, 0, 4, BILLBOARD_TEXTURE_SIZE, BILLBOARD_TEXTURE_SIZE, 0, GL_RGBA, GL_UNSIGNED_BYTE, shadedImage));
	//	CHECK_OPENGL(glTexImage2D(GL_TEXTURE_2D, 0, 4, BILLBOARD_TEXTURE_SIZE, BILLBOARD_TEXTURE_SIZE, 0, GL_RGBA, GL_FLOAT, shadedImage));
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

		CHECK_OPENGL(glBindTexture(GL_TEXTURE_2D, textureID[1]));
		CHECK_OPENGL(glTexImage2D(GL_TEXTURE_2D, 0, 4, BILLBOARD_TEXTURE_SIZE, BILLBOARD_TEXTURE_SIZE, 0, GL_RGBA, GL_UNSIGNED_BYTE, flatImage));
	//	CHECK_OPENGL(glTexImage2D(GL_TEXTURE_2D, 0, 4, BILLBOARD_TEXTURE_SIZE, BILLBOARD_TEXTURE_SIZE, 0, GL_RGBA, GL_FLOAT, flatImage));
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	}

	glBindTexture(GL_TEXTURE_2D, textureID[flatShading ? 1 : 0]);
}

/******************************************************************************
* Renders the atoms stored in the internal atoms array
* using the GL_ARB_point_parameters OpenGL extension.
******************************************************************************/
void AtomsRenderer::renderInternalArrayPointSprites(bool isPerspective, const Matrix4& projMatrix, float windowHeight)
{
	OVITO_ASSERT_MSG(windowHeight > 0, "AtomsRenderer::renderInternalArrayPointSprites()", "The viewport window has zero height.");

	// Disable lighting.
	glPushAttrib(GL_LIGHTING_BIT);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);

	// Use point sprites.
	CHECK_OPENGL(glEnable(GL_POINT_SPRITE_ARB));

	// This is how our point sprite's size will be modified by its
	// distance from the viewer
	if(isPerspective) {
		float quadratic[] = {0.0f, 0.0f, 100.0f / (projMatrix(1,1)*projMatrix(1,1) * windowHeight * windowHeight) };
		container->glPointParameterfvARB(GL_POINT_DISTANCE_ATTENUATION_ARB, quadratic);
		if(uniformRadius > 0)
			CHECK_OPENGL(glPointSize(uniformRadius * 10.0));
	}
	else {
		float constant[] = {1.0f, 0.0f, 0.0f};
		container->glPointParameterfvARB(GL_POINT_DISTANCE_ATTENUATION_ARB, constant);
		if(uniformRadius > 0)
			CHECK_OPENGL(glPointSize(uniformRadius * projMatrix(1,1) * windowHeight));
	}
	// No fading of small points.
	CHECK_OPENGL(container->glPointParameterfARB(GL_POINT_FADE_THRESHOLD_SIZE_ARB, 0.0f));
	CHECK_OPENGL(container->glPointParameterfARB(GL_POINT_SIZE_MIN_ARB, 0.01f));

	// Specify point sprite texture coordinate replacement mode for each texture unit.
	glTexEnvf(GL_POINT_SPRITE_ARB, GL_COORD_REPLACE_ARB, GL_TRUE);

	// Setup data arrays for atom position, color and radius.
	CHECK_OPENGL(glEnableClientState(GL_VERTEX_ARRAY));
	CHECK_OPENGL(glEnableClientState(GL_COLOR_ARRAY));
	if(vboVerticesID) {
		CHECK_OPENGL(container->glBindBufferARB(GL_ARRAY_BUFFER_ARB, vboVerticesID));
		CHECK_OPENGL(glVertexPointer(3, GL_FLOAT, sizeof(OpenGLAtom), (const char*)NULL +  offsetof(OpenGLAtom, x)));
		CHECK_OPENGL(glColorPointer(3, GL_UNSIGNED_BYTE, sizeof(OpenGLAtom), (const char*)NULL + offsetof(OpenGLAtom, r)));
	}
	else {
		CHECK_OPENGL(glVertexPointer(3, GL_FLOAT, sizeof(OpenGLAtom), &internalArray.data()->x));
		CHECK_OPENGL(glColorPointer(3, GL_UNSIGNED_BYTE, sizeof(OpenGLAtom), &internalArray.data()->r));
	}
	if(container->hasCompiledVertexArraysExtension())
		CHECK_OPENGL(container->glLockArrays(0, (GLsizei)numAtoms));

	if(chunkRenderSize == 0) {
		// Render atoms all at once.
		CHECK_OPENGL(glDrawArrays(GL_POINTS, 0, (GLsizei)numAtoms));
	}
	else {
		// Render atoms in chunks.
		for(GLuint index = 0; index < numAtoms; index += chunkRenderSize) {
			CHECK_OPENGL(glDrawArrays(GL_POINTS, index, min(chunkRenderSize, numAtoms-index)));
		}
	}

	if(container->hasCompiledVertexArraysExtension())
		container->glUnlockArrays();

	if(vboVerticesID) {
		container->glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
	}

	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);

	// Cleanup
	glDisable(GL_POINT_SPRITE_ARB);
	glDisable(GL_ALPHA_TEST);
	glDisable(GL_TEXTURE_2D);
	glPopAttrib();
}

/******************************************************************************
* Renders the atoms stored in the internal atoms array
* using the custom vertex and fragment shaders.
******************************************************************************/
void AtomsRenderer::renderInternalImpostersWithShader(bool isPerspective, const Matrix4& projMatrix, float windowHeight)
{
	OVITO_ASSERT(shadedImposterShader && shadedImposterShader->isValid());
	OVITO_ASSERT(flatImposterShader && flatImposterShader->isValid());
	OVITO_ASSERT_MSG(windowHeight > 0, "AtomsRenderer::renderInternalImpostersWithShader()", "The viewport window has zero height.");

	// Disable lighting.
	glPushAttrib(GL_LIGHTING_BIT);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);
	glDisable(GL_ALPHA_TEST);

	// Use point sprites.
	glEnable(GL_POINT_SPRITE_ARB);

	// This is how our point sprite's size will be modified by its
	// distance from the viewer
	if(isPerspective) {
		float quadratic[] = {0.0f, 0.0f, 100.0f / (projMatrix(1,1)*projMatrix(1,1) * windowHeight * windowHeight) };
		CHECK_OPENGL(container->glPointParameterfvARB(GL_POINT_DISTANCE_ATTENUATION_ARB, quadratic));
		if(uniformRadius > 0)
			CHECK_OPENGL(glPointSize(uniformRadius * 10.0));
	}
	else {
		float constant[] = {1.0f, 0.0f, 0.0f};
		CHECK_OPENGL(container->glPointParameterfvARB(GL_POINT_DISTANCE_ATTENUATION_ARB, constant));
		if(uniformRadius > 0)
			CHECK_OPENGL(glPointSize(uniformRadius * projMatrix(1,1) * windowHeight));
	}
	// No fading of small points.
	CHECK_OPENGL(container->glPointParameterfARB(GL_POINT_FADE_THRESHOLD_SIZE_ARB, 0.0f));
	CHECK_OPENGL(container->glPointParameterfARB(GL_POINT_SIZE_MIN_ARB, 0.01f));

	// Specify point sprite texture coordinate replacement mode for each texture unit.
	glTexEnvf(GL_POINT_SPRITE_ARB, GL_COORD_REPLACE_ARB, GL_TRUE);

	OpenGLShader* shader = flatShading ? flatImposterShader : shadedImposterShader;
	shader->setEnabled(true);
	// Let the vertex shader compute the point size.
	glEnable(GL_VERTEX_PROGRAM_POINT_SIZE);
	shader->sendUniform1i("isPerspective", isPerspective);
	shader->sendUniform1f("basePointSize", (float)(projMatrix(1,1) * windowHeight));

	// Setup data arrays for atom position, color and radius.
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	if(vboVerticesID) {
		CHECK_OPENGL(container->glBindBufferARB(GL_ARRAY_BUFFER_ARB, vboVerticesID));
		glVertexPointer(3, GL_FLOAT, sizeof(OpenGLAtom), (const char*)NULL + offsetof(OpenGLAtom, x));
		glColorPointer(3, GL_UNSIGNED_BYTE, sizeof(OpenGLAtom), (const char*)NULL + offsetof(OpenGLAtom, r));
		// The fog coord array is abused to pass the atom radii to the shader.
		if(container->hasFogCoordExtension() && shader) {
			glEnableClientState(GL_FOG_COORD_ARRAY);
			container->glFogCoordPointerEXT(GL_FLOAT, sizeof(OpenGLAtom), (const char*)NULL + offsetof(OpenGLAtom, radius));
		}
	}
	else {
		glVertexPointer(3, GL_FLOAT, sizeof(OpenGLAtom), &internalArray.data()->x);
		glColorPointer(3, GL_UNSIGNED_BYTE, sizeof(OpenGLAtom), &internalArray.data()->r);
		// The fog coord array is abused to pass the atom radii to the shader.
		if(container->hasFogCoordExtension() && shader) {
			glEnableClientState(GL_FOG_COORD_ARRAY);
			container->glFogCoordPointerEXT(GL_FLOAT, sizeof(OpenGLAtom), &internalArray.data()->radius);
		}
	}
	if(container->hasCompiledVertexArraysExtension()) CHECK_OPENGL(container->glLockArrays(0, (GLsizei)numAtoms));

	if(chunkRenderSize == 0) {
		// Render atoms all at once.
		CHECK_OPENGL(glDrawArrays(GL_POINTS, 0, (GLsizei)numAtoms));
	}
	else {
		// Render atoms in chunks.
		for(GLuint index = 0; index < numAtoms; index += chunkRenderSize) {
			CHECK_OPENGL(glDrawArrays(GL_POINTS, index, min(chunkRenderSize, numAtoms-index)));
		}
	}

	if(container->hasCompiledVertexArraysExtension()) container->glUnlockArrays();

	shader->setEnabled(false);
	glDisable(GL_VERTEX_PROGRAM_POINT_SIZE);

	if(vboVerticesID) {
		container->glBindBufferARB(GL_ARRAY_BUFFER_ARB, 0);
	}

	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
	if(container->hasFogCoordExtension() && shader)
		glDisableClientState(GL_FOG_COORD_ARRAY);

	// Cleanup
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_POINT_SPRITE_ARB);
	glPopAttrib();
}

/******************************************************************************
* Renders the atoms stored in the internal atoms array
* using the custom vertex and fragment shaders.
******************************************************************************/
void AtomsRenderer::renderInternalRaytracedWithShaders(bool isPerspective)
{
	OVITO_ASSERT(raytracedSphereShader && raytracedSphereShader->isValid());

	// Disable lighting.
	glPushAttrib(GL_LIGHTING_BIT);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);
	glDisable(GL_ALPHA_TEST);

	raytracedSphereShader->setEnabled(true);
	raytracedSphereShader->sendUniform1i("isPerspective", isPerspective);

	GLint viewportCoords[4];
	glGetIntegerv(GL_VIEWPORT, viewportCoords);
	raytracedSphereShader->sendUniform2f("viewport_origin", (float)viewportCoords[0], (float)viewportCoords[1]);
	raytracedSphereShader->sendUniform2f("inverse_viewport_size", 2.0f / (float)viewportCoords[2], 2.0f / (float)viewportCoords[3]);

	GLint particleRadiusLocation = raytracedSphereShader->getAttribLocation("particle_radius");
	GLint particlePosLocation = raytracedSphereShader->getAttribLocation("particle_pos");

	const OpenGLAtom* atom = internalArray.constData();
	const OpenGLAtom* lastAtom = atom + internalArray.size();

	glBegin(GL_QUADS);
	for(; atom != lastAtom; ++atom) {

		float xmin = atom->x - atom->radius;
		float ymin = atom->y - atom->radius;
		float zmin = atom->z - atom->radius;
		float xmax = atom->x + atom->radius;
		float ymax = atom->y + atom->radius;
		float zmax = atom->z + atom->radius;

		glColor4ubv(&atom->r);
		container->glVertexAttrib1fARB(particleRadiusLocation, atom->radius);
		container->glVertexAttrib3fvARB(particlePosLocation, &atom->x);

		// -Z face
		glVertex3f(xmin, ymax, zmin);
		glVertex3f(xmax, ymax, zmin);
		glVertex3f(xmax, ymin, zmin);
		glVertex3f(xmin, ymin, zmin);

		// +Z face
		glVertex3f(xmin, ymin, zmax);
		glVertex3f(xmax, ymin, zmax);
		glVertex3f(xmax, ymax, zmax);
		glVertex3f(xmin, ymax, zmax);

		// -Y face
		glVertex3f(xmin, ymin, zmin);
		glVertex3f(xmax, ymin, zmin);
		glVertex3f(xmax, ymin, zmax);
		glVertex3f(xmin, ymin, zmax);

		// +Y face
		glVertex3f(xmin, ymax, zmax);
		glVertex3f(xmax, ymax, zmax);
		glVertex3f(xmax, ymax, zmin);
		glVertex3f(xmin, ymax, zmin);

		// -X face
		glVertex3f(xmin, ymax, zmin);
		glVertex3f(xmin, ymin, zmin);
		glVertex3f(xmin, ymin, zmax);
		glVertex3f(xmin, ymax, zmax);

		// +X face
		glVertex3f(xmax, ymax, zmax);
		glVertex3f(xmax, ymin, zmax);
		glVertex3f(xmax, ymin, zmin);
		glVertex3f(xmax, ymax, zmin);

	}
	glEnd();

	raytracedSphereShader->setEnabled(false);

	// Cleanup
	glPopAttrib();
}

/******************************************************************************
* Renders the atoms stored in the internal atoms array
* without the GL_ARB_point_parameters OpenGL extension.
******************************************************************************/
void AtomsRenderer::renderInternalArrayQuads()
{
	// Disable lighting.
	glPushAttrib(GL_LIGHTING_BIT);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);

	float m[4][4];

	glMatrixMode(GL_MODELVIEW);
	glGetFloatv(GL_MODELVIEW_MATRIX, &m[0][0]);
	glPushMatrix();
	glLoadIdentity();

	const OpenGLAtom* atom = internalArray.constData();
	const OpenGLAtom* lastAtom = atom + internalArray.size();

	glBegin(GL_QUADS);
	for(; atom != lastAtom; ++atom) {
		float x = m[0][0]*atom->x + m[1][0]*atom->y + m[2][0]*atom->z + m[3][0];
		float y = m[0][1]*atom->x + m[1][1]*atom->y + m[2][1]*atom->z + m[3][1];
		float z = m[0][2]*atom->x + m[1][2]*atom->y + m[2][2]*atom->z + m[3][2];

		glColor4ubv(&atom->r);
		glTexCoord2f(0, 0);
		glVertex3f(x - atom->radius, y - atom->radius, z);
		glTexCoord2f(1, 0);
		glVertex3f(x + atom->radius, y - atom->radius, z);
		glTexCoord2f(1, 1);
		glVertex3f(x + atom->radius, y + atom->radius, z);
		glTexCoord2f(0, 1);
		glVertex3f(x - atom->radius, y + atom->radius, z);
	}
	glEnd();

	// Cleanup
	glPopMatrix();
	glDisable(GL_ALPHA_TEST);
	glDisable(GL_TEXTURE_2D);
	glPopAttrib();
}

/******************************************************************************
* Calculates the lighting for the atoms using an ambient occlusion algorithm.
******************************************************************************/
bool AtomsRenderer::calculateAmbientOcclusion(Window3D* glcontext)
{
	// Make sure that hardware-accelerated offscreen rendering is supported on this platform.
	if(!APPLICATION_MANAGER.guiMode())
		throw Exception("Cannot calculate ambient occlusion in console mode.");
	if(!glcontext->hasFrameBufferExtension())
		throw Exception("Cannot calculate ambient occlusion values. Your graphics card doesn't support the OpenGL GL_EXT_framebuffer_object extension.");

    // Create the offscreen framebuffer object - make sure to have a current
    // context before creating it.
	glcontext->makeCurrent();

	GLuint fbWidth = 512;
	GLuint fbHeight = 512;
	GLuint oglFrameBuffer = 0;
	GLuint oglRenderBuffer = 0;
	GLuint oglDepthRenderBuffer = 0;

	// Create a frame-buffer object and a render-buffer object for the depth channel.
	CHECK_OPENGL(glcontext->glGenFramebuffersEXT(1, &oglFrameBuffer));
	CHECK_OPENGL(glcontext->glGenRenderbuffersEXT(1, &oglRenderBuffer));
	CHECK_OPENGL(glcontext->glGenRenderbuffersEXT(1, &oglDepthRenderBuffer));

	CHECK_OPENGL(glcontext->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, oglDepthRenderBuffer));
	glcontext->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT24, fbWidth, fbHeight);
	GLenum errorCode = glGetError();
	if(errorCode == GL_NO_ERROR) {
		CHECK_OPENGL(glcontext->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, oglRenderBuffer));
		glcontext->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_RGBA8, fbWidth, fbHeight);
		errorCode = glGetError();
	}
	if(errorCode != GL_NO_ERROR) {
		// Clean up
		glcontext->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
		glcontext->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
		glcontext->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);
		throw Exception("Cannot calculate ambient occlusion. Failed to reserve an OpenGL offscreen buffer with the requested size.");
	}

	// Activate the off-screen frame-buffer and associate the render-buffer object.
	GLint oldFrameBuffer;
	glGetIntegerv(GL_FRAMEBUFFER_BINDING_EXT, &oldFrameBuffer);
	CHECK_OPENGL(glcontext->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, oglFrameBuffer));
	CHECK_OPENGL(glcontext->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_RENDERBUFFER_EXT, oglRenderBuffer));
	CHECK_OPENGL(glcontext->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, oglDepthRenderBuffer));

	// Check for errors...
	GLenum status = glcontext->glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	switch(status) {
	    case GL_FRAMEBUFFER_UNSUPPORTED_EXT:
	        throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Unsupported OpenGL framebuffer format.");
	    case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT:
	    	throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete attachment.");
	    case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT:
	    	throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing attachment.");
#ifdef GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT
	    case GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT:
	    	throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, duplicate attachment.");
#endif
	    case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
	    	throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, attached images must have same dimensions.");
	    case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
	    	throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, attached images must have same format.");
	    case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT:
	    	throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing draw buffer.");
	    case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT:
	    	throw Exception("Cannot calculate ambient occlusion. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing read buffer.");
	    case GL_NO_ERROR:
	    case GL_FRAMEBUFFER_COMPLETE_EXT:
	    	break;
	    default:
	    	throw Exception(QString("Cannot calculate ambient occlusion. An undefined error has occurred: status=%1").arg(status));
    }

	CHECK_OPENGL(glReadBuffer(GL_COLOR_ATTACHMENT0_EXT));

	// Prepare OpenGL rendering state.
	glPushAttrib(GL_LIGHTING_BIT|GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_ENABLE_BIT|GL_VIEWPORT_BIT|GL_TRANSFORM_BIT);
	glDisable(GL_SCISSOR_TEST);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glDepthMask(GL_TRUE);
	glViewport(0, 0, fbWidth, fbHeight);
	glClearColor(1.0f, 1.0f, 1.0f, 1.0f);

	// Setup view and projection matrix.
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	Vector3 center = boundingBox().center() - ORIGIN;
	FloatType diameter = Length(boundingBox().size());
    FloatType znear = -diameter / 2.0;
	FloatType zfar  = +diameter / 2.0;
	FloatType zoom  =  diameter / 2.0;
	Matrix4 projMatrix = Matrix4::ortho(-zoom, zoom, -zoom, zoom, znear, zfar);
	glLoadMatrix(projMatrix.constData());
	glMatrixMode(GL_MODELVIEW);

	bool oldFlatShading = flatShading;
	flatShading = true;
	QVector<float> depthBuffer(fbWidth*fbHeight);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	glTranslatef(-(GLfloat)center.X, -(GLfloat)center.Y, -(GLfloat)center.Z);

	// Choose rendering method.
	if(useImposters && container->hasPointParametersExtension() && flatImposterShader && flatImposterShader->isValid())
		renderInternalImpostersWithShader(false, projMatrix, fbHeight);
	else {
		initializeBillboardTextures();
		if(container->hasPointParametersExtension())
			renderInternalArrayPointSprites(false, projMatrix, fbHeight);
		else
			renderInternalArrayQuads();
	}
	glFlush();

	// Read out the OpenGL framebuffer.
	CHECK_OPENGL(glReadPixels(0, 0, fbWidth, fbHeight, GL_DEPTH_COMPONENT, GL_FLOAT, depthBuffer.data()));

	/*
	QImage::Format image_format = QImage::Format_RGB32;
	QImage img(fbWidth, fbHeight, image_format);
	for(int y=0; y<fbHeight; y++)
		for(int x=0; x<fbWidth; x++) {
			int v = (int)(depthBuffer[y*fbWidth+x] * 255.0);
			img.setPixel(x, y, qRgb(v,v,v));
		}
	img.save("ao.png");
	*/

	// Restore old settings.
	flatShading = oldFlatShading;
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
	glPopAttrib();

	// Make the old framebuffer the target again.
	glcontext->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, oldFrameBuffer);

	// Clean up
	glcontext->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
	glcontext->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
	glcontext->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);

	return true;
}

};	// End of namespace AtomViz
