///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file CNAModifier.h
 * \brief Contains the definition of the AtomViz::CommonNeighborAnalysisModifier class.
 */

#ifndef __COMMON_NEIGHBOR_ANALYSIS_MODIFIER_H
#define __COMMON_NEIGHBOR_ANALYSIS_MODIFIER_H

#include <core/Core.h>
#include <core/gui/properties/RefTargetListParameterUI.h>
#include <core/viewport/Viewport.h>

#include <atomviz/AtomViz.h>
#include <atomviz/modifier/AtomsObjectAnalyzerBase.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

/**
 * \brief A modifier that performs the Common Neighbor Analysis to classify atoms
 *        according to their local crystalline structure.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT CommonNeighborAnalysisModifier : public AtomsObjectAnalyzerBase
{
public:

	/// The types of crystalline structures recognized by the CNA algorithm.
	enum CNAAtomType {
		FCC = 0,				//< Face-centered cubic
		HCP = 1,				//< Hexagonal close packed
		BCC = 2,				//< Body-centered cubic
		ICOSAHEDRAL = 3,		//< Icosahedral structure
		OTHER = 4,				//< All other atoms which could not be assigned a crystalline structure
		CG_N = 5,				//< Cubic gauge nitrogen
		DIAMOND = 6,			//< Diamond
		HEX_DIAMOND = 7,		//< Hexagonal diamond
		DIAMOND_STACKINGFAULT = 8,	//< Diamond stacking fault atoms
		BCC_TWIN = 9,			// BCC twin plane atoms
		NumCNAAtomTypes //< This just counts the number of defined CNA atom types.
	};

	/// Default constructor.
	CommonNeighborAnalysisModifier(bool isLoading = false);

	/// Returns the array of atom types that are assigned to the atoms by this modifier.
	const AtomTypeList& atomTypes() const { return _atomTypesList; }

	/// Returns the data channel that stores the computed CNA atom types.
	AtomTypeDataChannel* cnaAtomTypes() const { return _cnaChannel; }

	/// \brief Returns whether the analysis results are saved along with the scene.
	/// \return \c true if data is stored in the scene file; \c false if the data needs to be recomputed after loading the scene file.
	bool storeResultsWithScene() const { return cnaAtomTypes() ? cnaAtomTypes()->serializeData() : false; }

	/// \brief Returns whether analysis results are saved along with the scene.
	/// \param on \c true if data should be stored in the scene file; \c false if the data needs to be recomputed after loading the scene file.
	/// \undoable
	void setStoreResultsWithScene(bool on) { if(cnaAtomTypes()) cnaAtomTypes()->setSerializeData(on); }

public:

	Q_PROPERTY(bool storeResultsWithScene READ storeResultsWithScene WRITE setStoreResultsWithScene)

protected:

	/// This is the actual analysis method.
	/// It is responsible for storing the analysis results in the results container object that
	/// will be stored along with the modifier application object.
	virtual EvaluationStatus doAnalysis(TimeTicks time, bool suppressDialogs);

	/// Applies the previously calculated analysis results to the atoms object.
	virtual EvaluationStatus applyResult(TimeTicks time, TimeInterval& validityInterval);

	/// Performs the analysis.
	/// Throws an exception on error.
	/// Returns false when the operation has been canceled by the user.
	bool calculate(AtomsObject* atomsObject, bool suppressDialogs = false);

	/// This stores the computed CNA atom types.
	ReferenceField<AtomTypeDataChannel> _cnaChannel;

	/// Contains the atom types created by the analysis modifier.
	VectorReferenceField<AtomType> _atomTypesList;

	/// This helper class is used to split up the computation into small
	/// operations that can be performed on multiple processors in parallel.
	class Kernel {
	public:
		// Constructor that takes references to the input and output arrays.
		Kernel(const NearestNeighborList& _nnlist, AtomTypeDataChannel* _atomTypeIndices) : nnlist(_nnlist), atomTypeIndices(*_atomTypeIndices) {
			// This call is necessary to deep copy the memory array of the position channel before accessing it from multiple threads.
			atomTypeIndices.dataInt();
		}

		// The actual kernel function that is called by the Qt concurrent framework for each atom.
		void operator()(int atomIndex);

	private:
		/// Input nearest-neighbor list
		const NearestNeighborList& nnlist;
		/// Output data channel
		AtomTypeDataChannel& atomTypeIndices;
	};

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(CommonNeighborAnalysisModifier)
	DECLARE_VECTOR_REFERENCE_FIELD(_atomTypesList)
	DECLARE_REFERENCE_FIELD(_cnaChannel)
};

/**
 * \brief A properties editor for the CommonNeighborAnalysisModifier class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the modifier's parameters.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT CommonNeighborAnalysisModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Is called when the user presses the Recalculate button.
	void onRecalculate();

	/// Is called when the user has double-clicked on one of the CNA
	/// atom types in the list widget.
	void onDoubleClickAtomType(const QModelIndex& index);

private:

	/// The parameter UI for the list of atom types.
	RefTargetListParameterUI* atomsTypesPUI;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(CommonNeighborAnalysisModifierEditor)
};

};	// End of namespace AtomViz

#endif // __COMMON_NEIGHBOR_ANALYSIS_MODIFIER_H
