///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <boost/iterator/counting_iterator.hpp>

#include "CoordinationNumberModifier.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CoordinationNumberModifier, AtomsObjectAnalyzerBase)
DEFINE_REFERENCE_FIELD(CoordinationNumberModifier, DataChannel, "CoordinationChannel", _coordinationChannel)
DEFINE_REFERENCE_FIELD(CoordinationNumberModifier, DataChannel, "BondsChannel", _bondsChannel)
DEFINE_PROPERTY_FIELD(CoordinationNumberModifier, "GenerateBonds", _generateBonds)
DEFINE_PROPERTY_FIELD(CoordinationNumberModifier, "MaxBonds", _maxBonds)
SET_PROPERTY_FIELD_LABEL(CoordinationNumberModifier, _generateBonds, "Generate bonds")
SET_PROPERTY_FIELD_LABEL(CoordinationNumberModifier, _maxBonds, "Maximum # of bonds per atom")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
CoordinationNumberModifier::CoordinationNumberModifier(bool isLoading)
	: AtomsObjectAnalyzerBase(isLoading), _generateBonds(false), _maxBonds(16)
{
	INIT_PROPERTY_FIELD(CoordinationNumberModifier, _coordinationChannel);
	INIT_PROPERTY_FIELD(CoordinationNumberModifier, _bondsChannel);
	INIT_PROPERTY_FIELD(CoordinationNumberModifier, _generateBonds);
	INIT_PROPERTY_FIELD(CoordinationNumberModifier, _maxBonds);

	if(!isLoading) {
		// Create data channels for the computation results.
		_coordinationChannel = new DataChannel(DataChannel::CoordinationChannel);
		_bondsChannel = new BondsDataChannel(DataChannel::BondsChannel);
	}
}

/******************************************************************************
* Applies the previously calculated analysis results to the atoms object.
******************************************************************************/
EvaluationStatus CoordinationNumberModifier::applyResult(TimeTicks time, TimeInterval& validityInterval)
{
	// Check if it is still valid.
	if(input()->atomsCount() != coordinationChannel()->size())
		throw Exception(tr("Number of atoms of input object has changed. Analysis results became invalid."));

	// Create a copy of the coordination channel that stores the calculated atom coordination numbers.
	CloneHelper cloneHelper;
	DataChannel::SmartPtr channelClone = cloneHelper.cloneObject(coordinationChannel(), true);

	// Put the new channel into the output object.
	output()->insertDataChannel(channelClone);

	// Put the new channel into the output object.
	if(_generateBonds && input()->atomsCount() == bondsChannel()->size()) {
		DataChannel::SmartPtr bondsClone = cloneHelper.cloneObject(bondsChannel(), true);
		output()->insertDataChannel(bondsClone);
	}
	else bondsChannel()->setSize(0);

	QString statusMessage;
	statusMessage += tr("Minimum coordination: %1\n").arg(minimumCoordinationNumber());
	statusMessage += tr("Maximum coordination: %1").arg(maximumCoordinationNumber());
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);
}

/******************************************************************************
* This is the actual analysis method.
******************************************************************************/
EvaluationStatus CoordinationNumberModifier::doAnalysis(TimeTicks time, bool suppressDialogs)
{
	// Perform analysis.
	if(calculate(input(), suppressDialogs))
		return EvaluationStatus();
	else
		return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR, tr("Calculation has been canceled by the user."));
}

/******************************************************************************
* Performs the analysis.
* Throws an exception on error.
* Returns false when the operation has been canceled by the user.
******************************************************************************/
bool CoordinationNumberModifier::calculate(AtomsObject* atomsObject, bool suppressDialogs)
{
	ProgressIndicator progress(tr("Calculating coordination (on %n processor(s))", NULL, QThread::idealThreadCount()), atomsObject->atomsCount(), suppressDialogs);

	// Prepare the neighbor list.
	OnTheFlyNeighborList neighborList(atomsObject, nearestNeighborList()->nearestNeighborCutoff());

	// Prepare the output channels.
	coordinationChannel()->setSize(atomsObject->atomsCount());
	if(_generateBonds) {
		size_t componentCount = (size_t)max((int)_maxBonds, 0);
		bondsChannel()->setComponentCount(componentCount);
		bondsChannel()->setSize(atomsObject->atomsCount());
		bondsChannel()->clearBonds();
	}
	else bondsChannel()->setSize(0);

	// Measure computation time.
	QTime timer;
	timer.start();

	// Execute CNA analysis code for each atom in a parallel fashion.
	Kernel kernel(neighborList, coordinationChannel(), _generateBonds ? bondsChannel() : NULL);
	boost::counting_iterator<int> firstAtom(0);
	boost::counting_iterator<int> lastAtom(atomsObject->atomsCount());
	QFuture<void> future = QtConcurrent::map(firstAtom, lastAtom, kernel);
	progress.waitForFuture(future);

	// Throw away results obtained so far if the user cancels the calculation.
	if(future.isCanceled()) {
		coordinationChannel()->setSize(0);
		bondsChannel()->setSize(0);
		minCoordination = maxCoordination = 0;
		return false;
	}

	// Determine minimum and maximum of coordination number distribution.
	minCoordination = 10000000;
	maxCoordination = 0;
	const int* c = coordinationChannel()->constDataInt();
	const int* cend = c + coordinationChannel()->size();
	for(; c != cend; ++c) {
		if(*c < minCoordination) minCoordination = *c;
		if(*c > maxCoordination) maxCoordination = *c;
	}

	// Special case of no atoms in the system.
	if(coordinationChannel()->size() == 0)
		minCoordination = maxCoordination = 0;

	VerboseLogger() << "Coordination number calculation took" << (timer.elapsed()/1000) << "sec." << endl;

	return true;
}

/******************************************************************************
* Counts the number of nearest-neighbors for a single atom.
******************************************************************************/
void CoordinationNumberModifier::Kernel::operator()(int atomIndex)
{
	int coordination = 0;
	for(OnTheFlyNeighborList::iterator neighborIter(nnlist, atomIndex); !neighborIter.atEnd(); neighborIter.next()) {
		if(bondsChannel) bondsChannel->setHalfBond(atomIndex, coordination, neighborIter.current());
		coordination++;
	}
	coordinationChannel->setInt(atomIndex, coordination);
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void CoordinationNumberModifier::saveToStream(ObjectSaveStream& stream)
{
	AtomsObjectAnalyzerBase::saveToStream(stream);

	stream.beginChunk(0x66EAAF);
	stream << minCoordination;
	stream << maxCoordination;
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void CoordinationNumberModifier::loadFromStream(ObjectLoadStream& stream)
{
	AtomsObjectAnalyzerBase::loadFromStream(stream);

	stream.expectChunk(0x66EAAF);
	stream >> minCoordination;
	stream >> maxCoordination;
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr CoordinationNumberModifier::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	CoordinationNumberModifier::SmartPtr clone = static_object_cast<CoordinationNumberModifier>(AtomsObjectAnalyzerBase::clone(deepCopy, cloneHelper));

	clone->minCoordination = this->minCoordination;
	clone->maxCoordination = this->maxCoordination;

	return clone;
}

IMPLEMENT_PLUGIN_CLASS(CoordinationNumberModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Constructor that sets up the UI widgets of the editor.
******************************************************************************/
void CoordinationNumberModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Coordination analysis"), rolloutParams);

    // Create the rollout contents.
	QVBoxLayout* layout1 = new QVBoxLayout(rollout);
	layout1->setContentsMargins(4,4,4,4);
	layout1->setSpacing(0);

	BooleanPropertyUI* autoUpdateUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _autoUpdateOnTimeChange));
	layout1->addWidget(autoUpdateUI->checkBox());

	BooleanPropertyUI* saveResultsUI = new BooleanPropertyUI(this, "storeResultsWithScene", tr("Save results in scene file"));
	layout1->addWidget(saveResultsUI->checkBox());

	BooleanPropertyUI* generateBondsUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(CoordinationNumberModifier, _generateBonds));
	layout1->addWidget(generateBondsUI->checkBox());

    // Create the rollout contents.
	QGridLayout* layout2 = new QGridLayout();
	layout2->setContentsMargins(4,4,4,4);
	layout2->setHorizontalSpacing(0);
	layout2->setVerticalSpacing(2);
	layout2->setColumnStretch(1, 1);
	layout1->addLayout(layout2);

	IntegerPropertyUI* maxBondsPUI = new IntegerPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(CoordinationNumberModifier, _maxBonds));
	layout2->addWidget(maxBondsPUI->label(), 0, 0);
	layout2->addWidget(maxBondsPUI->textBox(), 0, 1);
	layout2->addWidget(maxBondsPUI->spinner(), 0, 2);
	maxBondsPUI->setMinValue(1);

	QPushButton* recalcButton = new QPushButton(tr("Calculate"), rollout);
	layout1->addSpacing(6);
	layout1->addWidget(recalcButton);
	connect(recalcButton, SIGNAL(clicked(bool)), this, SLOT(onRecalculate()));

	// Status label.
	layout1->addSpacing(10);
	layout1->addWidget(statusLabel());

	// Open a sub-editor for the NearestNeighborList sub-object.
	new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(AtomsObjectAnalyzerBase, _nearestNeighborList), rolloutParams.before(rollout));

	// Open a sub-editor for the bonds channel.
	new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CoordinationNumberModifier, _bondsChannel), rolloutParams.after(rollout));
}

/******************************************************************************
* Is called when the user presses the Recalculate button.
******************************************************************************/
void CoordinationNumberModifierEditor::onRecalculate()
{
	if(!editObject()) return;
	CoordinationNumberModifier* modifier = static_object_cast<CoordinationNumberModifier>(editObject());
	try {
		modifier->performAnalysis(ANIM_MANAGER.time());
	}
	catch(Exception& ex) {
		ex.prependGeneralMessage(tr("Failed to calculate coordination."));
		ex.showError();
	}
}

};	// End of namespace AtomViz
