///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/undo/UndoManager.h>
#include <core/gui/properties/BooleanPropertyUI.h>

#include "SelectAtomTypeModifier.h"
#include <atomviz/utils/DataChannelComboBox.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(SelectAtomTypeModifier, SelectionModifierBase)

/******************************************************************************
* Sets the identifier of the data channel that contains the type for each atom.
******************************************************************************/
void SelectAtomTypeModifier::setSourceDataChannel(const DataChannelReference& c)
{
	if(_dataChannel == c) return;

	qRegisterMetaType<DataChannelReference>();
	if(UNDO_MANAGER.isRecording())
		UNDO_MANAGER.addOperation(new SimplePropertyChangeOperation(this, "sourceDataChannel"));

	_dataChannel = c;
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* Sets the list of atom type identifiers to select.
******************************************************************************/
void SelectAtomTypeModifier::setSelectedAtomTypes(const QSet<int>& types)
{
	if(_selectedAtomTypes == types) return;

	class SelectAtomTypesOperation : public UndoableOperation {
	public:
		SelectAtomTypesOperation(SelectAtomTypeModifier* _mod) : mod(_mod), oldTypes(_mod->selectedAtomTypes()) {}
		virtual void undo() {
			QSet<int> temp = mod->selectedAtomTypes();
			mod->setSelectedAtomTypes(oldTypes);
			oldTypes = temp;
		}
		virtual void redo() { undo(); }
		virtual QString displayName() const { return "Select Atom Type"; }
	private:
		SelectAtomTypeModifier::SmartPtr mod;
		QSet<int> oldTypes;
	};

	if(UNDO_MANAGER.isRecording())
		UNDO_MANAGER.addOperation(new SelectAtomTypesOperation(this));

	_selectedAtomTypes = types;
	notifyDependents(REFTARGET_CHANGED);
}

/******************************************************************************
* This modifies the input object.
******************************************************************************/
EvaluationStatus SelectAtomTypeModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	// Get the atom type channel.
	AtomTypeDataChannel* typeChannel = dynamic_object_cast<AtomTypeDataChannel>(input()->lookupDataChannel(sourceDataChannel()));
	if(typeChannel == NULL)
		throw Exception(tr("The selection source channel is not present in the input object."));
	OVITO_ASSERT(typeChannel->componentCount() == 1);
	OVITO_ASSERT(typeChannel->type() == qMetaTypeId<int>());

	QString statusMessage = tr("%n input atoms", 0, input()->atomsCount());

	// Get the deep copy of the selection channel.
	DataChannel* selChannel = outputStandardChannel(DataChannel::SelectionChannel);
	selChannel->setVisible(selectionShown());

	// The number of selected atoms.
	size_t nSelected = 0;

	// Create a selection based on the atomic types.
	const int* t = typeChannel->constDataInt();
	int* s = selChannel->dataInt();
	for(size_t i = selChannel->size(); i != 0; i--, t+=typeChannel->componentCount()) {
		if(selectedAtomTypes().contains((*t))) {
			*s++ = 1;
			nSelected++;
		}
		else *s++ = 0;
	}

	statusMessage += tr("\n%n atoms selected", 0, nSelected);
	return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, QString(), statusMessage);
}

/******************************************************************************
* Saves the class' contents to the given stream.
******************************************************************************/
void SelectAtomTypeModifier::saveToStream(ObjectSaveStream& stream)
{
	SelectionModifierBase::saveToStream(stream);
	stream.beginChunk(0x10000000);
	stream << _dataChannel;
	stream << _selectedAtomTypes;
	stream.endChunk();
}

/******************************************************************************
* Loads the class' contents from the given stream.
******************************************************************************/
void SelectAtomTypeModifier::loadFromStream(ObjectLoadStream& stream)
{
	SelectionModifierBase::loadFromStream(stream);
	stream.expectChunk(0x10000000);
	stream >> _dataChannel;
	stream >> _selectedAtomTypes;
	stream.closeChunk();
}

/******************************************************************************
* Creates a copy of this object.
******************************************************************************/
RefTarget::SmartPtr SelectAtomTypeModifier::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	SelectAtomTypeModifier::SmartPtr clone = static_object_cast<SelectAtomTypeModifier>(SelectionModifierBase::clone(deepCopy, cloneHelper));

	clone->_dataChannel = this->_dataChannel;
	clone->_selectedAtomTypes = this->_selectedAtomTypes;

	return clone;
}

IMPLEMENT_PLUGIN_CLASS(SelectAtomTypeModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void SelectAtomTypeModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	QWidget* rollout = createRollout(tr("Select Atom Type"), rolloutParams, "atomviz.modifiers.select_atom_type");

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);

	dataChannelBox = new DataChannelComboBox();
	layout->addWidget(new QLabel(tr("Use following data channel:"), rollout));
	layout->addWidget(dataChannelBox);

	class MyListWidget : public QListWidget {
	public:
		MyListWidget() : QListWidget() {}
		virtual QSize sizeHint() const { return QSize(256, 92); }
	};
	atomTypesBox = new MyListWidget();
	atomTypesBox->setSelectionMode(QAbstractItemView::ExtendedSelection);
	layout->addWidget(new QLabel(tr("Atom type to select:"), rollout));
	layout->addWidget(atomTypesBox);

	layout->addSpacing(2);
	BooleanPropertyUI* showSelUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(SelectionModifierBase, _selectionShown));
	layout->addWidget(showSelUI->checkBox());

	// Update channel list if another modifier has been loaded into the editor.
	connect(this, SIGNAL(contentsReplaced(RefTarget*)), this, SLOT(updateDataChannelList()));

	// Status label.
	layout->addSpacing(12);
	layout->addWidget(new QLabel(tr("Status:")));
	layout->addWidget(statusLabel());
}

/******************************************************************************
* Updates the contents of the combo box.
******************************************************************************/
void SelectAtomTypeModifierEditor::updateDataChannelList()
{
	disconnect(dataChannelBox, SIGNAL(activated(int)), this, SLOT(onDataChannelSelected(int)));
	dataChannelBox->clear();

	SelectAtomTypeModifier* mod = static_object_cast<SelectAtomTypeModifier>(editObject());
	if(!mod) {
		dataChannelBox->setEnabled(false);
	}
	else {
		dataChannelBox->setEnabled(true);

		// Populate data channel list based on input object.
		PipelineFlowState inputState = mod->getModifierInput();
		AtomsObject* inputObj = dynamic_object_cast<AtomsObject>(inputState.result());
		if(inputObj) {
			Q_FOREACH(DataChannel* channel, inputObj->dataChannels()) {
				AtomTypeDataChannel* atypeChannel = dynamic_object_cast<AtomTypeDataChannel>(channel);
				if(atypeChannel != NULL && atypeChannel->atomTypes().empty() == false && atypeChannel->componentCount() == 1) {
					dataChannelBox->addItem(atypeChannel);
				}
			}
		}

		dataChannelBox->setCurrentChannel(mod->sourceDataChannel());
	}
	connect(dataChannelBox, SIGNAL(activated(int)), this, SLOT(onDataChannelSelected(int)));

	updateAtomTypeList();
}

/******************************************************************************
* Updates the contents of the list box.
******************************************************************************/
void SelectAtomTypeModifierEditor::updateAtomTypeList()
{
	disconnect(atomTypesBox, SIGNAL(itemSelectionChanged()), this, SLOT(onAtomTypeSelected()));
	atomTypesBox->setUpdatesEnabled(false);
	atomTypesBox->clear();

	SelectAtomTypeModifier* mod = static_object_cast<SelectAtomTypeModifier>(editObject());
	if(!mod) {
		atomTypesBox->setEnabled(false);
	}
	else {
		atomTypesBox->setEnabled(true);

		// Populate atom types list based on the input object's data channel.
		PipelineFlowState inputState = mod->getModifierInput();
		AtomsObject* inputObj = dynamic_object_cast<AtomsObject>(inputState.result());
		if(inputObj) {
			AtomTypeDataChannel* atomTypeChannel = dynamic_object_cast<AtomTypeDataChannel>(inputObj->lookupDataChannel(mod->sourceDataChannel()));
			if(atomTypeChannel) {
				for(int i=0; i<atomTypeChannel->atomTypes().size(); i++) {
					AtomType* atype = atomTypeChannel->atomTypes()[i];
					if(!atype) continue;
					QListWidgetItem* item = new QListWidgetItem(atype->name(), atomTypesBox);
					item->setData(Qt::UserRole, i);
					if(mod->selectedAtomTypes().contains(i))
						item->setSelected(true);
				}
			}
		}
	}

	connect(atomTypesBox, SIGNAL(itemSelectionChanged()), this, SLOT(onAtomTypeSelected()));
	atomTypesBox->setUpdatesEnabled(true);
}

/******************************************************************************
* This is called when the user has selected another item in the data channel list.
******************************************************************************/
void SelectAtomTypeModifierEditor::onDataChannelSelected(int index)
{
	SelectAtomTypeModifier* mod = static_object_cast<SelectAtomTypeModifier>(editObject());
	if(!mod) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Select Data Channel"));
	mod->setSourceDataChannel(dataChannelBox->currentChannel());
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* This is called when the user has selected another atom type.
******************************************************************************/
void SelectAtomTypeModifierEditor::onAtomTypeSelected()
{
	SelectAtomTypeModifier* mod = static_object_cast<SelectAtomTypeModifier>(editObject());
	if(!mod) return;

	QSet<int> types;
	Q_FOREACH(QListWidgetItem* item, atomTypesBox->selectedItems()) {
		types.insert(item->data(Qt::UserRole).toInt());
	}

	UNDO_MANAGER.beginCompoundOperation(tr("Select Atom Type"));
	mod->setSelectedAtomTypes(types);
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool SelectAtomTypeModifierEditor::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == editObject() && msg->type() == REFTARGET_CHANGED) {
		updateDataChannelList();
	}
	return AtomsObjectModifierEditorBase::onRefTargetMessage(source, msg);
}

};	// End of namespace AtomViz
