///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __SELECT_ATOM_TYPE_MODIFIER_H
#define __SELECT_ATOM_TYPE_MODIFIER_H

#include <core/Core.h>

#include <atomviz/AtomViz.h>
#include "SelectionModifierBase.h"

namespace AtomViz {

class DataChannelComboBox; 		// defined in DataChannelComboBox.h

/******************************************************************************
* Selects atoms of one atom type.
******************************************************************************/
class ATOMVIZ_DLLEXPORT SelectAtomTypeModifier : public SelectionModifierBase
{
public:

	/// Default constructor.
	SelectAtomTypeModifier(bool isLoading = false) : SelectionModifierBase(isLoading),
		_dataChannel(DataChannel::AtomTypeChannel) {}

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time) { return TimeForever; }

	/// Returns the data channel that is used as source for the selection.
	const DataChannelReference& sourceDataChannel() const { return _dataChannel; }
	/// Sets the data channel that is used as source for the selection.
	void setSourceDataChannel(const DataChannelReference& c);

	/// Returns the list of atom type identifiers that are selected.
	const QSet<int>& selectedAtomTypes() const { return _selectedAtomTypes; }
	/// Sets the list of atom type identifiers to select.
	void setSelectedAtomTypes(const QSet<int>& types);
	/// Sets a single atom type identifier to be selected.
	void setSelectedAtomType(int type) { QSet<int> temp; temp.insert(type); setSelectedAtomTypes(temp); }

public:

	Q_PROPERTY(AtomViz::DataChannelReference sourceDataChannel READ sourceDataChannel WRITE setSourceDataChannel)

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);
	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// Modifies the atoms object.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// The data channel that is used as source for the selection.
	DataChannelReference _dataChannel;

	/// The identifiers (indices) of the atom types to select.
	QSet<int> _selectedAtomTypes;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(SelectAtomTypeModifier)
};

/******************************************************************************
* A properties editor for the SelectAtomTypeModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT SelectAtomTypeModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

protected Q_SLOTS:

	/// Updates the contents of the data channel combo box.
	void updateDataChannelList();
	/// Updates the contents of the atom types list box.
	void updateAtomTypeList();

	/// This is called when the user has selected another item in the data channel list.
	void onDataChannelSelected(int index);
	/// This is called when the user has selected another atom type.
	void onAtomTypeSelected();

protected:

	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

private:

	/// The list of data channels.
	DataChannelComboBox* dataChannelBox;
	/// The list of atom types.
	QListWidget* atomTypesBox;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(SelectAtomTypeModifierEditor)
};

};	// End of namespace AtomViz

#endif // __SELECT_ATOM_TYPE_MODIFIER_H
