///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/BooleanRadioButtonPropertyUI.h>
#include <core/gui/properties/AffineTransformationPropertyUI.h>
#include <core/undo/UndoManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/objects/ModifiedObject.h>

#include <boost/iterator/counting_iterator.hpp>

#include "AffineTransformationModifier.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(AffineTransformationModifier, AtomsObjectModifierBase)
DEFINE_PROPERTY_FIELD(AffineTransformationModifier, "Transformation", _transformationTM)
DEFINE_PROPERTY_FIELD(AffineTransformationModifier, "ApplyToAtoms", _applyToAtoms)
DEFINE_PROPERTY_FIELD(AffineTransformationModifier, "ToSelectionOnly", _toSelectionOnly)
DEFINE_PROPERTY_FIELD(AffineTransformationModifier, "ApplyToSimulationBox", _applyToSimulationBox)
DEFINE_PROPERTY_FIELD(AffineTransformationModifier, "DestinationCell", _destinationCell)
DEFINE_PROPERTY_FIELD(AffineTransformationModifier, "RelativeMode", _relativeMode)
SET_PROPERTY_FIELD_LABEL(AffineTransformationModifier, _transformationTM, "Transformation")
SET_PROPERTY_FIELD_LABEL(AffineTransformationModifier, _applyToAtoms, "Apply transformation to atoms")
SET_PROPERTY_FIELD_LABEL(AffineTransformationModifier, _toSelectionOnly, "Apply to selected atoms only")
SET_PROPERTY_FIELD_LABEL(AffineTransformationModifier, _applyToSimulationBox, "Apply transformation to simulation box")
SET_PROPERTY_FIELD_LABEL(AffineTransformationModifier, _destinationCell, "Destination cell geometry")
SET_PROPERTY_FIELD_LABEL(AffineTransformationModifier, _relativeMode, "Relative transformation")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
AffineTransformationModifier::AffineTransformationModifier(bool isLoading) : AtomsObjectModifierBase(isLoading),
	_applyToAtoms(true), _toSelectionOnly(false), _applyToSimulationBox(false), _transformationTM(IDENTITY), _destinationCell(NULL_MATRIX), _relativeMode(true)
{
	INIT_PROPERTY_FIELD(AffineTransformationModifier, _transformationTM);
	INIT_PROPERTY_FIELD(AffineTransformationModifier, _applyToAtoms);
	INIT_PROPERTY_FIELD(AffineTransformationModifier, _toSelectionOnly);
	INIT_PROPERTY_FIELD(AffineTransformationModifier, _applyToSimulationBox);
	INIT_PROPERTY_FIELD(AffineTransformationModifier, _destinationCell);
	INIT_PROPERTY_FIELD(AffineTransformationModifier, _relativeMode);
}

/******************************************************************************
* Asks the modifier for its validity interval at the given time.
******************************************************************************/
TimeInterval AffineTransformationModifier::modifierValidity(TimeTicks time)
{
	return TimeForever;
}

/******************************************************************************
* This method is called by the system when the modifier has been inserted into a ModifiedObject.
******************************************************************************/
void AffineTransformationModifier::initializeModifier(ModifiedObject* modObject, ModifierApplication* modApp)
{
	// Take the simulation cell from the input object as the default destination cell geometry for absolute scaling.
	if((AffineTransformation)_destinationCell == NULL_MATRIX) {
		PipelineFlowState input = modObject->evalObject(ANIM_MANAGER.time(), modApp, false);
		AtomsObject* inputObject = dynamic_object_cast<AtomsObject>(input.result());
		if(inputObject != NULL) {
			_destinationCell = inputObject->simulationCell()->cellMatrix();
		}
	}
}

/******************************************************************************
* Modifies the atoms object. The time interval passed
* to the function is reduced to the interval where the modified object is valid/constant.
******************************************************************************/
EvaluationStatus AffineTransformationModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	AffineTransformation tm;
	if(_relativeMode) {
		tm = _transformationTM;
		if(applyToSimulationBox()) {
			AffineTransformation cell = output()->simulationCell()->cellMatrix();
			AffineTransformation deformedCell = tm * cell;
			output()->simulationCell()->setCellMatrix(deformedCell);
		}
	}
	else {
		AffineTransformation oldCell = input()->simulationCell()->cellMatrix();
		if(oldCell.determinant() == 0.0)
			throw Exception(tr("Input simulation cell is degenerate."));
		tm = (AffineTransformation)_destinationCell * oldCell.inverse();
		if(applyToSimulationBox())
			output()->simulationCell()->setCellMatrix(_destinationCell);
	}

	if(applyToAtoms()) {
		expectStandardChannel(DataChannel::PositionChannel);
		DataChannel* posChannel = outputStandardChannel(DataChannel::PositionChannel);

		if(toSelectionOnly()) {
			DataChannel* selChannel = inputStandardChannel(DataChannel::SelectionChannel);
			if(selChannel) {
				boost::counting_iterator<int> begin(0);
				boost::counting_iterator<int> end(posChannel->size());
				QtConcurrent::blockingMap(begin, end, KernelWithSelection(tm, posChannel, selChannel));
			}
		}
		else {
			Point3* pbegin = posChannel->dataPoint3();
			Point3* pend = pbegin + posChannel->size();
			QtConcurrent::blockingMap(pbegin, pend, Kernel(tm));
		}
	}

	return EvaluationStatus();
}

IMPLEMENT_PLUGIN_CLASS(AffineTransformationModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void AffineTransformationModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the first rollout.
	QWidget* rollout = createRollout(tr("Affine transformation"), rolloutParams, "atomviz.modifiers.affine_transformation");

    QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setHorizontalSpacing(0);
	layout->setVerticalSpacing(0);
	layout->setColumnStretch(0, 5);
	layout->setColumnStretch(1, 95);

	BooleanPropertyUI* applyToSimulationBoxUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AffineTransformationModifier, _applyToSimulationBox));
	layout->addWidget(applyToSimulationBoxUI->checkBox(), 0, 0, 1, 2);

	BooleanPropertyUI* applyToAtomsUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AffineTransformationModifier, _applyToAtoms));
	layout->addWidget(applyToAtomsUI->checkBox(), 1, 0, 1, 2);

	BooleanRadioButtonPropertyUI* selectionUI = new BooleanRadioButtonPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AffineTransformationModifier, _toSelectionOnly));

	selectionUI->buttonFalse()->setText(tr("All atoms"));
	selectionUI->buttonFalse()->setEnabled(false);
	layout->addWidget(selectionUI->buttonFalse(), 2, 1);
	connect(applyToAtomsUI->checkBox(), SIGNAL(toggled(bool)), selectionUI->buttonFalse(), SLOT(setEnabled(bool)));

	selectionUI->buttonTrue()->setText(tr("Only to selected atoms"));
	selectionUI->buttonTrue()->setEnabled(false);
	layout->addWidget(selectionUI->buttonTrue(), 3, 1);
	connect(applyToAtomsUI->checkBox(), SIGNAL(toggled(bool)), selectionUI->buttonTrue(), SLOT(setEnabled(bool)));

	// Create the second rollout.
	rollout = createRollout(tr("Transformation"), rolloutParams.after(rollout), "atomviz.modifiers.affine_transformation");

	QVBoxLayout* topLayout = new QVBoxLayout(rollout);

	BooleanRadioButtonPropertyUI* relativeModeUI = new BooleanRadioButtonPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AffineTransformationModifier, _relativeMode));

	relativeModeUI->buttonTrue()->setText(tr("Relative transformation matrix:"));
	topLayout->addWidget(relativeModeUI->buttonTrue());

    layout = new QGridLayout();
	layout->setContentsMargins(30,4,4,4);
	layout->setHorizontalSpacing(0);
	layout->setVerticalSpacing(2);
	topLayout->addLayout(layout);

	layout->addWidget(new QLabel(tr("Rotate/Scale/Shear:")), 0, 0, 1, 8);
	for(int col=0; col<3; col++) {
		layout->setColumnStretch(col*3 + 0, 1);
		if(col < 2) layout->setColumnMinimumWidth(col*3 + 2, 4);
		for(int row=0; row<4; row++) {
			QLineEdit* lineEdit = new QLineEdit(rollout);
			SpinnerWidget* spinner = new SpinnerWidget(rollout);
			if(row < 3) {
				elementSpinners[row][col] = spinner;
				spinner->setProperty("column", col);
				spinner->setProperty("row", row);
			}
			else {
				elementSpinners[col][row] = spinner;
				spinner->setProperty("column", row);
				spinner->setProperty("row", col);
			}
			spinner->setTextBox(lineEdit);

			int gridRow = (row == 3) ? 5 : (row+1);
			layout->addWidget(lineEdit, gridRow, col*3 + 0);
			layout->addWidget(spinner, gridRow, col*3 + 1);

			connect(spinner, SIGNAL(spinnerValueChanged()), this, SLOT(onSpinnerValueChanged()));
			connect(spinner, SIGNAL(spinnerDragStart()), this, SLOT(onSpinnerDragStart()));
			connect(spinner, SIGNAL(spinnerDragStop()), this, SLOT(onSpinnerDragStop()));
			connect(spinner, SIGNAL(spinnerDragAbort()), this, SLOT(onSpinnerDragAbort()));
		}
	}
	layout->addWidget(new QLabel(tr("Translation:")), 4, 0, 1, 8);

	relativeModeUI->buttonFalse()->setText(tr("Scale to fixed box size:"));
	topLayout->addWidget(relativeModeUI->buttonFalse());

    layout = new QGridLayout();
	layout->setContentsMargins(30,4,4,4);
	layout->setHorizontalSpacing(0);
	layout->setVerticalSpacing(2);
	layout->setColumnStretch(0, 1);
	layout->setColumnStretch(3, 1);
	layout->setColumnStretch(6, 1);
	layout->setColumnMinimumWidth(2, 4);
	layout->setColumnMinimumWidth(5, 4);
	topLayout->addLayout(layout);
	AffineTransformationPropertyUI* destinationCellUI;

	for(size_t v = 0; v < 3; v++) {
		layout->addWidget(new QLabel(tr("Cell vector %1:").arg(v+1)), v*2, 0, 1, 8);
		for(size_t r = 0; r < 3; r++) {
			destinationCellUI = new AffineTransformationPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AffineTransformationModifier, _destinationCell), r, v);
			layout->addWidget(destinationCellUI->textBox(), v*2+1, r*3+0);
			layout->addWidget(destinationCellUI->spinner(), v*2+1, r*3+1);
		}
	}

	layout->addWidget(new QLabel(tr("Cell origin:")), 6, 0, 1, 8);
	for(size_t r = 0; r < 3; r++) {
		destinationCellUI = new AffineTransformationPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(AffineTransformationModifier, _destinationCell), r, 3);
		layout->addWidget(destinationCellUI->textBox(), 7, r*3+0);
		layout->addWidget(destinationCellUI->spinner(), 7, r*3+1);
	}

	// Update spinner values when a new object has been loaded into the editor.
	connect(this, SIGNAL(contentsChanged(RefTarget*)), this, SLOT(updateUI()));

	// Also update the displayed values when the animation time has changed.
	connect(&ANIM_MANAGER, SIGNAL(timeChanged(TimeTicks)), this, SLOT(updateUI()));
}

/******************************************************************************
* This method updates the displayed matrix values.
******************************************************************************/
void AffineTransformationModifierEditor::updateUI()
{
	AffineTransformationModifier* mod = dynamic_object_cast<AffineTransformationModifier>(editObject());
	if(mod == NULL) return;

	const AffineTransformation& tm = mod->transformation();

	for(int row=0; row<3; row++) {
		for(int column=0; column<4; column++) {
			if(!elementSpinners[row][column]->isDragging())
				elementSpinners[row][column]->setFloatValue(tm(row, column));
		}
	}
}

/******************************************************************************
* Is called when the spinner value has changed.
******************************************************************************/
void AffineTransformationModifierEditor::onSpinnerValueChanged()
{
	ViewportSuspender noVPUpdate;
	if(!UNDO_MANAGER.isRecording()) {
		UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));
		updateParameterValue();
		UNDO_MANAGER.endCompoundOperation();
	}
	else {
		UNDO_MANAGER.currentCompoundOperation()->clear();
		updateParameterValue();
	}
}

/******************************************************************************
* Takes the value entered by the user and stores it in transformation controller.
******************************************************************************/
void AffineTransformationModifierEditor::updateParameterValue()
{
	AffineTransformationModifier* mod = dynamic_object_cast<AffineTransformationModifier>(editObject());
	if(mod == NULL) return;

	// Get the spinner whose value has changed.
	SpinnerWidget* spinner = qobject_cast<SpinnerWidget*>(sender());
	CHECK_POINTER(spinner);

	AffineTransformation tm = mod->transformation();

	int column = spinner->property("column").toInt();
	int row = spinner->property("row").toInt();

	tm(row, column) = spinner->floatValue();
	mod->setTransformation(tm);
}

/******************************************************************************
* Is called when the user begins dragging the spinner interactively.
******************************************************************************/
void AffineTransformationModifierEditor::onSpinnerDragStart()
{
	OVITO_ASSERT(!UNDO_MANAGER.isRecording());
	UNDO_MANAGER.beginCompoundOperation(tr("Change Parameter"));
}

/******************************************************************************
* Is called when the user stops dragging the spinner interactively.
******************************************************************************/
void AffineTransformationModifierEditor::onSpinnerDragStop()
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Is called when the user aborts dragging the spinner interactively.
******************************************************************************/
void AffineTransformationModifierEditor::onSpinnerDragAbort()
{
	OVITO_ASSERT(UNDO_MANAGER.isRecording());
	UNDO_MANAGER.currentCompoundOperation()->clear();
	UNDO_MANAGER.endCompoundOperation();
}

};	// End of namespace AtomViz
