///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file AffineTransformationModifier.h
 * \brief Contains the definition of the AtomViz::AffineTransformationModifier class.
 */

#ifndef __AFFINE_TRANSFORMATION_MODIFIER_H
#define __AFFINE_TRANSFORMATION_MODIFIER_H

#include <core/Core.h>
#include <core/gui/SpinnerWidget.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/controller/TransformationController.h>
#include <atomviz/AtomViz.h>
#include <atomviz/modifier/AtomsObjectModifierBase.h>

namespace AtomViz {

/**
 * \brief This modifier applies an arbitrary affine transformation to the atoms and/or the simulation box.
 *
 * The affine transformation is specified as a 3x4 Matrix.
 * The position vector of each atom gets multiplied by this matrix when the modifiers is
 * applied to an AtomsObject. The user can specify whether the transformation is only
 * applied to the atoms or to the SimulationCell too.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AffineTransformationModifier : public AtomsObjectModifierBase
{
public:
	/// \brief Constructs a new instance of this class.
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	AffineTransformationModifier(bool isLoading = false);

	/// \brief Asks the modifier for its validity interval at the given time.
	///
	/// This method returns the maximum time interval during which
	/// the object's parameters stay constant and that includes the given
	/// point in time.
	virtual TimeInterval modifierValidity(TimeTicks time);

	/// \brief This method is called by the system when the modifier has been inserted into a ModifiedObject.
	/// \param modObject The ModifiedObject into which the modifier has been inserted.
	/// \param modApp The ModifiedApplication object that has been created for the modifier.
	virtual void initializeModifier(ModifiedObject* modObject, ModifierApplication* modApp);

	// Property access functions:

	/// Returns the affine transformation matrix at the current animation time.
	const AffineTransformation& transformation() const { return _transformationTM; }
	/// Sets the affine transformation at the current animation time.
	void setTransformation(const AffineTransformation& tm) { _transformationTM = tm; }

	/// Returns whether the transformation is applied to the atoms of the input object.
	/// \sa setApplyToAtoms()
	bool applyToAtoms() const { return _applyToAtoms; }
	/// Sets whether the transformation is applied to the atoms of the input object.
	/// \sa applyToAtoms()
	void setApplyToAtoms(bool apply) { _applyToAtoms = apply; }

	/// Returns whether the transformation is applied only to the selected atoms of the input object.
	/// \sa setToSelectionOnly()
	bool toSelectionOnly() const { return _toSelectionOnly; }
	/// Sets whether the transformation is applied only to the selected atoms of the input object.
	/// \sa toSelectionOnly()
	void setToSelectionOnly(bool onlySelected) { _toSelectionOnly = onlySelected; }

	/// Returns whether the transformation is applied to the simulation box of the input object.
	/// \sa setApplyToSimulationBox()
	bool applyToSimulationBox() const { return _applyToSimulationBox; }
	/// Sets whether the transformation is applied to the simulation box of the input object.
	/// \sa applyToSimulationBox()
	void setApplyToSimulationBox(bool apply) { _applyToSimulationBox = apply; }

public:

	Q_PROPERTY(bool applyToAtoms READ applyToAtoms WRITE setApplyToAtoms)
	Q_PROPERTY(bool toSelectionOnly READ toSelectionOnly WRITE setToSelectionOnly)
	Q_PROPERTY(bool applyToSimulationBox READ applyToSimulationBox WRITE setApplyToSimulationBox)

protected:

	/// \brief Modifies the atoms object.
	///
	/// The time interval passed to the function is cut down by this method to the interval
	/// where the output object produced by this modifier is valid/constant.
	/// This method is part of the implementation of the abstract AtomsObjectModifierBase class.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// This property fields stores the transformation matrix (used in 'relative' mode).
	PropertyField<AffineTransformation> _transformationTM;

	/// This property fields stores the simulation cell geometry  (used in 'absolute' mode).
	PropertyField<AffineTransformation> _destinationCell;

	/// This controls whether the transformation is applied to the atoms.
	PropertyField<bool> _applyToAtoms;

	/// This controls whether the transformation is applied only to the selected atoms.
	PropertyField<bool> _toSelectionOnly;

	/// This controls whether the transformation is applied to the simulation box.
	PropertyField<bool> _applyToSimulationBox;

	/// This controls whether a relative transformation is applied to the simulation box or
	/// the absolute cell geometry has been specified.
	PropertyField<bool> _relativeMode;

private:

	/// This helper class is used to split up the computation into small
	/// operations that can be performed on multiple processors in parallel.
	class Kernel {
	public:
		// Constructor that takes references to the input and output arrays.
		Kernel(const AffineTransformation& _tm) : tm(_tm) {}

		// The actual kernel function that is called by the Qt concurrent framework for each atom.
		void operator()(Point3& p) const { p = tm * p; }

	private:
		const AffineTransformation tm;
	};

	/// This helper class is used to split up the computation into small
	/// operations that can be performed on multiple processors in parallel.
	class KernelWithSelection {
	public:
		// Constructor that takes references to the input and output arrays.
		KernelWithSelection(const AffineTransformation& _tm, DataChannel* _posChannel, DataChannel* _selChannel) : tm(_tm), posChannel(_posChannel), selChannel(_selChannel) {
			// This call is necessary to deep copy the memory array of the position channel before accessing it from multiple threads.
			posChannel->dataPoint3();
		}

		// The actual kernel function that is called by the Qt concurrent framework for each atom.
		void operator()(int atomIndex) const {
			OVITO_ASSERT(posChannel != NULL && selChannel != NULL);
			OVITO_ASSERT(atomIndex < posChannel->size() && atomIndex < selChannel->size());
			if(selChannel->getInt(atomIndex))
				posChannel->setPoint3(atomIndex, tm * posChannel->getPoint3(atomIndex));
		}

	private:
		const AffineTransformation tm;
		DataChannel* posChannel;
		DataChannel* selChannel;
	};

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AffineTransformationModifier)
	DECLARE_PROPERTY_FIELD(_transformationTM)
	DECLARE_PROPERTY_FIELD(_applyToAtoms)
	DECLARE_PROPERTY_FIELD(_toSelectionOnly)
	DECLARE_PROPERTY_FIELD(_applyToSimulationBox)
	DECLARE_PROPERTY_FIELD(_destinationCell)
	DECLARE_PROPERTY_FIELD(_relativeMode)
};

/**
 * \brief A properties editor for the AffineTransformationModifier class.
 *
 * This editor class creates and manages the user interface through which the
 * user can alter the modifier's parameters.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AffineTransformationModifierEditor : public AtomsObjectModifierEditorBase
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private Q_SLOTS:

	/// Is called when the spinner value has changed.
	void onSpinnerValueChanged();

	/// Is called when the user begins dragging the spinner interactively.
	void onSpinnerDragStart();

	/// Is called when the user stops dragging the spinner interactively.
	void onSpinnerDragStop();

	/// Is called when the user aborts dragging the spinner interactively.
	void onSpinnerDragAbort();

	/// This method updates the displayed matrix values.
	void updateUI();

private:

	/// Takes the value entered by the user and stores it in transformation controller.
	void updateParameterValue();

	SpinnerWidget* elementSpinners[3][4];

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AffineTransformationModifierEditor)
};

};	// End of namespace AtomViz

#endif // __AFFINE_TRANSFORMATION_MODIFIER_H
