///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>

#include "ColumnChannelMapping.h"
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

/******************************************************************************
 * Resizes the internal mapping array to include the specified number of data columns.
 *****************************************************************************/
void ColumnChannelMapping::setColumnCount(int numberOfColumns, const QStringList& columnNames)
{
	// Expand column array if necessary and initialize all new columns to their default values.
	while(numberOfColumns >= columnCount()) {
		MapEntry newEntry;
		newEntry.dataChannelId = DataChannel::UserDataChannel;
		newEntry.type = QMetaType::Void;
		newEntry.vectorComponent = 0;
		columns.append(newEntry);
	}
	columns.resize(numberOfColumns);

	for(int i=0; i<columnNames.size() && i<numberOfColumns; i++)
		columns[i].columnName = columnNames[i];
}

/******************************************************************************
 * Associates a column in the data file with a custom DataChannel in the AtomsObject.
 *****************************************************************************/
void ColumnChannelMapping::defineColumn(int columnIndex, DataChannel::DataChannelIdentifier channelId, const QString& channelName, int dataType, size_t vectorComponent, const QString& columnName)
{
	OVITO_ASSERT(columnIndex >= 0);

	// Expand column array if necessary and initialize all new columns to their default values.
	if(columnIndex >= columnCount())
		setColumnCount(columnIndex+1);

	columns[columnIndex].dataChannelId = channelId;
	columns[columnIndex].dataChannelName = channelName;
	columns[columnIndex].columnName = columnName;
	columns[columnIndex].type = dataType;
	columns[columnIndex].vectorComponent = vectorComponent;
}

/******************************************************************************
 * Associates a column in the data file with a standard DataChannel in the AtomsObject.
 *****************************************************************************/
void ColumnChannelMapping::defineStandardColumn(int columnIndex, DataChannel::DataChannelIdentifier channel, size_t vectorComponent, const QString& columnName)
{
	defineColumn(columnIndex, channel, DataChannel::standardChannelName(channel), DataChannel::standardChannelType(channel), vectorComponent, columnName);
}

/******************************************************************************
 * Ignores a column in the data file.
 *****************************************************************************/
void ColumnChannelMapping::ignoreColumn(int columnIndex, const QString& columnName)
{
	OVITO_ASSERT(columnIndex >= 0);
	if(columnIndex < columnCount()) {
		columns[columnIndex].dataChannelId = DataChannel::UserDataChannel;
		columns[columnIndex].dataChannelName = QString();
		columns[columnIndex].columnName = columnName;
		columns[columnIndex].type = QMetaType::Void;
		columns[columnIndex].vectorComponent = 0;
	}
	else {
		// Expand column array if necessary and initialize all new columns to their default values.
		setColumnCount(columnIndex+1);
		columns[columnIndex].columnName = columnName;
	}
}

/******************************************************************************
 * Saves the mapping to the given stream.
 *****************************************************************************/
void ColumnChannelMapping::saveToStream(SaveStream& stream) const
{
	stream.beginChunk(0x11000000);
	stream << remapAtomIndices;
	stream << columns.size();
	for(QVector<MapEntry>::const_iterator entry = columns.constBegin(); entry != columns.constEnd(); ++entry) {
		stream << entry->columnName;
		stream.writeEnum(entry->dataChannelId);
		stream << entry->dataChannelName;
		stream.writeEnum(entry->type);
		stream.writeSizeT(entry->vectorComponent);
	}
	stream.endChunk();
}

/******************************************************************************
 * Loads the mapping from the given stream.
 *****************************************************************************/
void ColumnChannelMapping::loadFromStream(LoadStream& stream)
{
	stream.expectChunk(0x11000000);
	stream >> remapAtomIndices;
	int numColumns;
	stream >> numColumns;
	columns.resize(numColumns);
	for(QVector<MapEntry>::iterator entry = columns.begin(); entry != columns.end(); ++entry) {
		stream >> entry->columnName;
		stream.readEnum(entry->dataChannelId);
		stream >> entry->dataChannelName;
		stream.readEnum(entry->type);
		if(entry->type == qMetaTypeId<float>() || entry->type == qMetaTypeId<double>())
			entry->type = qMetaTypeId<FloatType>();
		stream.readSizeT(entry->vectorComponent);
	}
	stream.closeChunk();
}

/******************************************************************************
 * Saves the mapping into a byte array.
 *****************************************************************************/
QByteArray ColumnChannelMapping::toByteArray() const
{
	QByteArray buffer;
	QDataStream dstream(&buffer, QIODevice::WriteOnly);
	SaveStream stream(dstream);
	saveToStream(stream);
	stream.close();
	return buffer;
}

/******************************************************************************
 * Loads the mapping from a byte array.
 *****************************************************************************/
void ColumnChannelMapping::fromByteArray(const QByteArray& array)
{
	QDataStream dstream(array);
	LoadStream stream(dstream);
	loadFromStream(stream);
	stream.close();
}

/******************************************************************************
 * Saves the mapping the application's settings store.
 *****************************************************************************/
void ColumnChannelMapping::savePreset(const QString& presetName) const
{
	QSettings settings;
	settings.beginGroup("atomviz/io/columnmapping/presets");
	settings.beginGroup(presetName);
	settings.setValue("name", presetName);
	settings.setValue("data", toByteArray());
	settings.endGroup();
	settings.endGroup();
}

/******************************************************************************
 * Loads a mapping from the application's settings store.
 *****************************************************************************/
void ColumnChannelMapping::loadPreset(const QString& presetName)
{
	QSettings settings;
	settings.beginGroup("atomviz/io/columnmapping/presets");
	settings.beginGroup(presetName);
	if(settings.value("name").toString() != presetName)
		throw Exception(tr("No preset found with the name: %1").arg(presetName));
	fromByteArray(settings.value("data").toByteArray());
}

/******************************************************************************
 * Returns a list of all presets found in the
 *****************************************************************************/
QStringList ColumnChannelMapping::listPresets()
{
	QStringList list;
	QSettings settings;
	settings.beginGroup("atomviz/io/columnmapping/presets");
	// Find preset with the given name.
	Q_FOREACH(QString group, settings.childGroups()) {
		settings.beginGroup(group);
		list.push_back(settings.value("name").toString());
		settings.endGroup();
	}
	return list;
}

/******************************************************************************
 * Deletes a mapping from the application's settings store.
 *****************************************************************************/
void ColumnChannelMapping::deletePreset(const QString& presetName)
{
	QSettings settings;
	settings.beginGroup("atomviz/io/columnmapping/presets");
	// Find preset with the given name.
	Q_FOREACH(QString group, settings.childGroups()) {
		settings.beginGroup(group);
		if(settings.value("name").toString() == presetName) {
			settings.endGroup();
			settings.remove(group);
			return;
		}
		settings.endGroup();
	}
	throw Exception(tr("No preset found with the name: %1").arg(presetName));
}

/******************************************************************************
 * Makes a copy of the mapping object.
 *****************************************************************************/
ColumnChannelMapping& ColumnChannelMapping::operator=(const ColumnChannelMapping& other)
{
	this->columns = other.columns;
	this->remapAtomIndices = other.remapAtomIndices;
	return *this;
}

/******************************************************************************
 * Initializes the helper object.
 *****************************************************************************/
DataRecordParserHelper::DataRecordParserHelper(const ColumnChannelMapping* mapping, AtomsObject* destination)
	: _coordinatesOutOfRange(false),
	  intMetaTypeId(qMetaTypeId<int>()),
	  floatMetaTypeId(qMetaTypeId<FloatType>())
{
	CHECK_POINTER(mapping);
	CHECK_OBJECT_POINTER(destination);

	this->mapping = mapping;
	this->destination = destination;
	this->atomIndexColumn = -1;

	if(mapping->columnCount() > MAXIMUM_ATOM_COLUMN_COUNT)
		throw Exception(tr("Cannot parse more than %1 data columns from the input file.").arg(MAXIMUM_ATOM_COLUMN_COUNT));

	// Create data channels in the destination object.
	for(int i=0; i<mapping->columnCount(); i++) {
		DataChannel* channel = NULL;
		int channelType = mapping->getChannelType(i);
		size_t vectorComponent = mapping->getVectorComponent(i);

		if(channelType != QMetaType::Void) {
			DataChannel::DataChannelIdentifier channelId = mapping->getChannelId(i);
			QString channelName = mapping->getChannelName(i);

			size_t dataTypeSize;
			if(channelType == qMetaTypeId<int>()) {
				dataTypeSize = sizeof(int);
			}
			else if(channelType == qMetaTypeId<FloatType>()) {
				dataTypeSize = sizeof(FloatType);
			}
			else throw Exception(tr("Invalid custom data channel type %1 selected for column %2").arg(channelType).arg(i+1));

			if(channelId != DataChannel::UserDataChannel) {	// standard data channel
				if(channelId == DataChannel::AtomIndexChannel && mapping->atomIndexRemappingEnabled())
					atomIndexColumn = i;
				else
					channel = destination->createStandardDataChannel(channelId);
			}
			else {	// non-standard data channel
				// Look for existing data channel with that name
				channel = destination->findDataChannelByName(channelName);
				if(channel == NULL) {
					// Create a new data channel for the column.
					channel = destination->createCustomDataChannel(channelType, dataTypeSize, vectorComponent+1);
				}
				else if(channel->type() != channelType && channel->componentCount() <= vectorComponent) {
					// Replace old channel with a new one that has the correct data type.
					DataChannel* oldChannel = channel;
					channel = new DataChannel(channelType, dataTypeSize, vectorComponent+1);
					destination->replaceDataChannel(oldChannel, channel);
				}
			}
			if(channel) {
				channel->setName(channelName);
				OVITO_ASSERT(channel->size() == destination->atomsCount());
				CHECK_OBJECT_POINTER(channel);
			}
		}

		// Build internal list of channel objects for fast look up during parsing.
		channels.push_back(channel);
	}

	// Remove unused data channels from the destination object.
	Q_FOREACH(DataChannel* channel, destination->dataChannels()) {
		if(!channels.contains(channel))
			destination->removeDataChannel(channel);
	}

	for(QVector<DataChannel*>::iterator channel = channels.begin(); channel != channels.end(); ++channel) {
		if(*channel == NULL) continue;
		CHECK_OBJECT_POINTER(*channel);
	}
}

/******************************************************************************
 * Parses the string tokens from one line of the input file and stores the values
 * in the data channels of the destination AtomsObject.
 *****************************************************************************/
void DataRecordParserHelper::storeAtom(int atomIndex, char* dataLine)
{
	// Divide string into tokens.
	const char* tokens[MAXIMUM_ATOM_COLUMN_COUNT];
	int ntokens = 0;
	while(ntokens < MAXIMUM_ATOM_COLUMN_COUNT) {
		while(*dataLine != '\0' && (*dataLine == ' ' || *dataLine == '\t'))
			++dataLine;
		tokens[ntokens] = dataLine;
		while(*dataLine != '\0' && *dataLine != ' ' && *dataLine != '\t')
			++dataLine;
		if(dataLine != tokens[ntokens]) ntokens++;
		if(*dataLine == '\0') break;
		*dataLine = '\0';
		dataLine++;
	}

	storeAtom(atomIndex, ntokens, tokens);
}

/******************************************************************************
 * Parses the string tokens from one line of the input file and stores the values
 * in the data channels of the destination AtomsObject.
 *****************************************************************************/
void DataRecordParserHelper::storeAtom(int atomIndex, int ntokens, const char* tokens[])
{
	OVITO_ASSERT(channels.size() == mapping->columnCount());
	if(ntokens < channels.size())
		throw Exception(tr("Data line in input file contains not enough items. Expected %1 data columns but found only %2.").arg(channels.size()).arg(ntokens));

	if(atomIndex >= (int)destination->atomsCount())
		throw Exception(tr("Too many data lines in input file. Expected only %1 lines.").arg(destination->atomsCount()));

	QVector<DataChannel*>::iterator channel = channels.begin();
	const char** token = tokens;

	int d;
	char* endptr;

	// Parse the atom index column.
	if(atomIndexColumn >= 0) {
		atomIndex = strtoul(tokens[atomIndexColumn], &endptr, 10);
		if(*endptr) throw Exception(tr("Invalid integer value in column %1 (atom index): \"%2\"").arg(atomIndexColumn+1).arg(tokens[atomIndexColumn]));
		if(atomIndex > (int)destination->atomsCount())
			throw Exception(tr("Atom index is out of range. This error is usually caused by partial data files, which contain only a subset of the total number of atoms in the system. To avoid this error, do NOT read in the atom IDs/indices during import. The erroneous atom index is %1, but number of atoms in input file is only %2.").arg(tokens[atomIndexColumn]).arg(destination->atomsCount()));
		else if(atomIndex < 1)
			throw Exception(tr("Found non-positive atom index: %1.").arg(tokens[atomIndexColumn]));
		atomIndex--;
	}

	for(int columnIndex = 0; channel != channels.end(); ++columnIndex, ++token, ++channel) {
		if(*channel == NULL) continue;
		CHECK_OBJECT_POINTER(*channel);
		OVITO_ASSERT(destination->dataChannels().contains(*channel));
		OVITO_ASSERT((*channel)->size() == destination->atomsCount());

		if((*channel)->type() == floatMetaTypeId) {
			double f = strtod(*token, &endptr);
			if(*endptr) throw Exception(tr("Invalid floating-point value in column %1 (%2): \"%3\"").arg(columnIndex+1).arg((*channel)->name()).arg(*token));
			size_t vectorComponent = mapping->getVectorComponent(columnIndex);

			// Do a quick range check for atom coordinates.
			if((*channel)->id() == DataChannel::PositionChannel) {
				OVITO_ASSERT(vectorComponent < 3);
				if(f == numeric_limits<double>::signaling_NaN() || abs(f) > FLOATTYPE_MAX * 0.5) {
					_coordinatesOutOfRange = true;
				}
				if(f < _boundingBox.minc[vectorComponent]) _boundingBox.minc[vectorComponent] = f;
				if(f > _boundingBox.maxc[vectorComponent]) _boundingBox.maxc[vectorComponent] = f;
			}

			(*channel)->setFloatComponent(atomIndex, vectorComponent, (FloatType)f);
		}
		else if((*channel)->type() == intMetaTypeId) {
			d = strtol(*token, &endptr, 10);
			AtomTypeDataChannel* atypeChannel = dynamic_object_cast<AtomTypeDataChannel>(*channel);
			if(!atypeChannel) {
				if(*endptr) throw Exception(tr("Invalid integer value in column %1 (%2): \"%3\"").arg(columnIndex+1).arg((*channel)->name()).arg(*token));
				(*channel)->setIntComponent(atomIndex, mapping->getVectorComponent(columnIndex), d);
			}
			else {
				// Automatically create an atom type if a new type number is encountered.
				if(!*endptr) {
					atypeChannel->setIntComponent(atomIndex, mapping->getVectorComponent(columnIndex), d);
					atypeChannel->createAtomType(d);
				}
				else {
					d = atypeChannel->findAtomTypeIndexByName(*token);
					if(d < 0) {
						d = atypeChannel->atomTypes().size();
						atypeChannel->createAtomType(d)->setName(*token);
						OVITO_ASSERT(atypeChannel->atomTypes()[d]->name() == *token);
					}
					atypeChannel->setIntComponent(atomIndex, mapping->getVectorComponent(columnIndex), d);
				}
			}
		}
	}
}

/******************************************************************************
 * Stores the columns values for one atom in the data channels of the destination AtomsObject.
 *****************************************************************************/
void DataRecordParserHelper::storeAtom(int atomIndex, const double* values, int nvalues)
{
	OVITO_ASSERT(channels.size() == mapping->columnCount());
	if(nvalues < channels.size())
		throw Exception(tr("Data row in input file contains to few items. Expected %1 data columns but found only %2.").arg(channels.size(), nvalues));

	if(atomIndex >= (int)destination->atomsCount())
		throw Exception(tr("Too many data rows in input file. Expected only %1 rows.").arg(destination->atomsCount()));

	QVector<DataChannel*>::iterator channel = channels.begin();
	const double* value = values;

	int d;
	double f;

	// Parse the atom index column.
	if(atomIndexColumn >= 0 && atomIndexColumn < nvalues) {
		atomIndex = (int)values[atomIndexColumn];
		if(atomIndex > (int)destination->atomsCount())
			throw Exception(tr("Atom index is out of range. This error is usually caused by partial data files, which contain only a subset of the total number of atoms in the system. To avoid this error, do NOT read in the atom IDs/indices during import. The erroneous atom index is %1, but number of atoms in input file is only %2.").arg(atomIndex).arg(destination->atomsCount()));
		else if(atomIndex < 1)
			throw Exception(tr("Found non-positive atom index: %1.").arg(atomIndex));
		atomIndex--;
	}

	// Parse data values and store them in the appropriate data channels.
	for(int columnIndex = 0; channel != channels.end(); ++columnIndex, ++value, ++channel) {

		if(*channel == NULL) continue;
		CHECK_OBJECT_POINTER(*channel);
		OVITO_ASSERT(destination->dataChannels().contains(*channel));
		OVITO_ASSERT((*channel)->size() == destination->atomsCount());

		if((*channel)->type() == floatMetaTypeId) {
			f = *value;
			size_t vectorComponent = mapping->getVectorComponent(columnIndex);

			// Do a quick range check for atom coordinates.
			if((*channel)->id() == DataChannel::PositionChannel) {
				OVITO_ASSERT(vectorComponent < 3);
				if(f == numeric_limits<double>::signaling_NaN() || abs(f) > FLOATTYPE_MAX * 0.5) {
					_coordinatesOutOfRange = true;
				}
				if(f < _boundingBox.minc[vectorComponent]) _boundingBox.minc[vectorComponent] = f;
				if(f > _boundingBox.maxc[vectorComponent]) _boundingBox.maxc[vectorComponent] = f;
			}

			(*channel)->setFloatComponent(atomIndex, vectorComponent, (FloatType)f);
		}
		else if((*channel)->type() == intMetaTypeId) {
			d = (int)*value;
			(*channel)->setIntComponent(atomIndex, mapping->getVectorComponent(columnIndex), d);
			AtomTypeDataChannel* atypeChannel = dynamic_object_cast<AtomTypeDataChannel>(*channel);
			if(atypeChannel)
				atypeChannel->createAtomType(d);
		}
	}
}

};	// End of namespace AtomViz
