///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>
#include <core/gui/ApplicationManager.h>
#include <core/utilities/ProgressIndicator.h>
#include "LAMMPSTextDumpParser.h"
#include "../CompressedTextParserStream.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(LAMMPSTextDumpParser, LAMMPSDumpParser)

/******************************************************************************
* Default constructor.
******************************************************************************/
LAMMPSTextDumpParser::LAMMPSTextDumpParser(bool isLoading) : LAMMPSDumpParser(isLoading)
{
}

/******************************************************************************
* Checks if the given file has format that can be read by this importer.
******************************************************************************/
bool LAMMPSTextDumpParser::checkFileFormat(const QString& filepath)
{
	// Open the input file for reading.
	CompressedTextParserStream stream(filepath);

	char buffer[20];
	int count = stream.getline(buffer, sizeof(buffer)/sizeof(buffer[0]));
	if(count<1) return false;

	// Skip initial whitespace.
	char* p = buffer;
	while(isspace(*p)) {
		if(*p == '\0') return false;
		++p;
	}
	if(qstrncmp(p, "ITEM: ", 6) != 0) return false;

	// Search for "ITEM: NUMBER OF ATOMS" within the first 20 lines.
	for(int i=0; i<20; i++) {
		if(stream.eof()) return false;
		if(stream.readline().find("ITEM: NUMBER OF ATOMS") != string::npos)
			return true;
	}

	return false;
}

/******************************************************************************
* Parses the header of the given file and returns the number of data columns contained in the file.
******************************************************************************/
bool LAMMPSTextDumpParser::inspectFileHeader(const QString& filename, int& numberOfColumns, QStringList& columnNames)
{
	// Open input file.
	CompressedTextParserStream stream(filename);

	int numAtoms = 0;
	setlocale(LC_NUMERIC, "C");

	while(!stream.eof()) {

		// Parse next line.
		stream.readline();

		// Skip empty lines
		if(stream.line().find_first_not_of(" \t\n\r") == string::npos)
			continue;

		do {
			if(stream.line().find("ITEM: NUMBER OF ATOMS") != string::npos) {
				// Parse number of atoms.
				if(sscanf(stream.readline().c_str(), "%u", &numAtoms) != 1 || numAtoms < 0 || numAtoms > 1e9)
					throw Exception(tr("LAMMPS dump file parsing error. Invalid number of atoms (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				break;
			}
			else if(stream.line().find("ITEM: ATOMS") != string::npos) {

				// Determine the number of columns from the column name list.
				QStringList tokens = QString(stream.line().c_str()).split(QRegExp("\\s+"));
				OVITO_ASSERT(tokens[0] == "ITEM:");
				OVITO_ASSERT(tokens[1] == "ATOMS");
				int numberOfColumnNames = max(0, tokens.size() - 2);
				if(numberOfColumnNames > 1 && tokens.back().isEmpty()) {
					numberOfColumnNames--;
					tokens.pop_back();
				}

				// NOTE: This is a special hack for Karsten's dump files.
				// Can be safely removed in the future.
				if(numberOfColumnNames == 3 && tokens[2] == "0" && tokens[3] == "0" && tokens[4] == "0")
					numberOfColumnNames = 0;

				if(numAtoms > 0) {
					// Parse first atom line
					stream.readline();

					// Count the number of tokens in the first atom line.
					numberOfColumns = 0;
					const char* start = stream.line().c_str();
					char* end;
					for(;;) {
						double d = strtod(start, &end);
						if(end == start) break;
						numberOfColumns++;
						start = end;
					}
				}
				else numberOfColumns = numberOfColumnNames;

				if(numberOfColumnNames > 0) {
					if(numberOfColumnNames != numberOfColumns)
						throw Exception(tr("LAMMPS dump file parsing error. Invalid number of column names in line %1. It differs from the number of values in the first atom line.").arg(stream.lineNumber()-1));
					columnNames = tokens.mid(2);
				}

				return true;
			}
			else if(stream.line().find("ITEM:") != string::npos) {
				// Skip lines up to next ITEM:
				while(!stream.eof()) {
					stream.readline();
					if(stream.line().find("ITEM:") != string::npos) break;
				}
			}
			else if(stream.line().find_first_not_of(" \t\n\r") == string::npos) {
				// Skip empty lines
				break;
			}
			else {
				throw Exception(tr("LAMMPS dump file parsing error. Invalid line %1 in dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
			}
		}
		while(!stream.eof());
	}
	return true;
}

/******************************************************************************
* Scans an atoms file for the time step frames contained therein.
******************************************************************************/
bool LAMMPSTextDumpParser::scanFileForTimeSteps(const QString& filename, bool suppressDialogs)
{
	// Determine file size.
	QFile input_file(filename);
	qint64 fileSize = input_file.size();
	if(fileSize == 0)
		throw Exception(tr("The LAMMPS dump file %1 contains no data.").arg(filename));

	// Open input file.
	CompressedTextParserStream stream(filename);

	ProgressIndicator progress(tr("Opening LAMMPS dump file '%1'").arg(filename), fileSize, suppressDialogs);

	int timeStepNumber = 0;
	int numAtoms = 0;
	setlocale(LC_NUMERIC, "C");

	while(!stream.eof()) {
		streampos byteOffset = stream.byteOffset();

		// Parse next line.
		stream.readline();

		do {
			int startLineNumber = stream.lineNumber();
			if(stream.line().find("ITEM: TIMESTEP") != string::npos) {
				if(sscanf(stream.readline().c_str(), "%i", &timeStepNumber) != 1)
					throw Exception(tr("LAMMPS dump file parsing error. Invalid timestep number (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				// Create a new record for the time step.
				addTimeStep(filename, byteOffset, startLineNumber);
				progress.setLabelText(tr("Scanning LAMMPS dump file (Frame %1)").arg(timeStepNumber));
				timeStepNumber++;
				break;
			}
			else if(stream.line().find("ITEM: NUMBER OF ATOMS") != string::npos) {
				// Parse number of atoms.
				if(sscanf(stream.readline().c_str(), "%u", &numAtoms) != 1 || numAtoms < 0 || numAtoms > 1e9)
					throw Exception(tr("LAMMPS dump file parsing error. Invalid number of atoms (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				break;
			}
			else if(stream.line().find("ITEM: ATOMS") != string::npos) {
				// Parse one atom per line.
				for(int i = 0; i < numAtoms; i++) {
					stream.readline();

					if(i % 4096 == 0) {
						progress.setValue(stream.compressedByteOffset());
						if(progress.isCanceled()) {
							if(APPLICATION_MANAGER.guiMode() && QMessageBox::question(NULL, tr("Scan operation canceled"), tr("Do you want to keep the time steps that have already been scanned so far?"),
								QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes) == QMessageBox::Yes)
								return true;
							else
								return false;
						}
					}
				}
				break;
			}
			else if(stream.line().find("ITEM:") != string::npos) {
				// Skip lines up to next ITEM:
				while(!stream.eof()) {
					byteOffset = stream.byteOffset();
					stream.readline();
					if(stream.line().find("ITEM:") != string::npos) break;
				}
			}
			else if(stream.line().find_first_not_of(" \t\n\r") == string::npos) {
				// Skip empty lines
				break;
			}
			else {
				throw Exception(tr("LAMMPS dump file parsing error. Invalid line %1 in dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
			}
		}
		while(!stream.eof());
	}
	return true;
}

/******************************************************************************
* Parses the atomic data of a single time step.
******************************************************************************/
EvaluationStatus LAMMPSTextDumpParser::loadTimeStep(AtomsObject* destination, int movieFrame, const QString& filename, streampos byteOffset, int lineNumber, bool suppressDialogs)
{
	CHECK_OBJECT_POINTER(destination);

	// Show the progress indicator.
	ProgressIndicator progress(tr("Opening LAMMPS dump file '%1'").arg(filename), 0, suppressDialogs);

	// Open input file.
	CompressedTextParserStream stream(filename);

	// Seek to the byte offset where the requested movie frame is stored.
	if(byteOffset != streampos(0))
		stream.seek(byteOffset);

	int numAtoms = 0;
	int timeStep = movieFrame;
	Box3 simBox;
	setlocale(LC_NUMERIC, "C");

	// These are the format strings used with the sscanf() parsing function.
	// We have to obey the floating point precision used to compile the program
	// because the destination variables are either single or double precision variables.
#ifdef USE_DOUBLE_PRECISION_FP
	#define FLOAT_SCANF_STRING_1   "%lg"
	#define FLOAT_SCANF_STRING_2   "%lg %lg"
	#define FLOAT_SCANF_STRING_3   "%lg %lg %lg"
#else
	#define FLOAT_SCANF_STRING_1   "%g"
	#define FLOAT_SCANF_STRING_2   "%g %g"
	#define FLOAT_SCANF_STRING_3   "%g %g %g"
#endif

	while(!stream.eof()) {

		// Parse next line.
		stream.readline();

		do {
			if(stream.line().find("ITEM: TIMESTEP") != string::npos) {
				if(sscanf(stream.readline().c_str(), "%i", &timeStep) != 1)
					throw Exception(tr("LAMMPS dump file parsing error. Invalid timestep number (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				break;
			}
			else if(stream.line().find("ITEM: NUMBER OF ATOMS") != string::npos) {
				// Parse number of atoms.
				if(sscanf(stream.readline().c_str(), "%u", &numAtoms) != 1 || numAtoms < 0 || numAtoms > 1e9)
					throw Exception(tr("LAMMPS dump file parsing error. Invalid number of atoms (line %1): %2").arg(stream.lineNumber()).arg(stream.line().c_str()));

				progress.setLabelText(tr("Loading LAMMPS dump file (%1 atoms at time step %2)").arg(numAtoms).arg(timeStep));
				progress.setMaximum(numAtoms);
				break;
			}
			else if(stream.line().find("ITEM: BOX BOUNDS xy xz yz") != string::npos) {
				// Parse triclinic simulation box.
				FloatType tiltFactors[3];
				for(size_t k=0; k<3; k++) {
					if(sscanf(stream.readline().c_str(), FLOAT_SCANF_STRING_3, &simBox.minc[k], &simBox.maxc[k], &tiltFactors[k]) != 3)
						throw Exception(tr("Invalid box size in line %1 of dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				}
				// LAMMPS only stores the outer bounding box of the simulation cell in the dump file.
				// We have to determine the size of the actual triclinic cell.
				simBox.minc.X -= min(min(tiltFactors[0], tiltFactors[1]), (FloatType)0.0);
				simBox.maxc.X -= max(max(tiltFactors[0], tiltFactors[1]), (FloatType)0.0);
				simBox.minc.Y -= min(tiltFactors[2], (FloatType)0.0);
				simBox.maxc.Y -= max(tiltFactors[2], (FloatType)0.0);
				destination->simulationCell()->setCellShape(simBox.minc, Vector3(simBox.sizeX(), 0, 0), Vector3(tiltFactors[0],simBox.sizeY(), 0), Vector3(tiltFactors[1],tiltFactors[2],simBox.sizeZ()));
				break;
			}
			else if(stream.line().find("ITEM: BOX BOUNDS") != string::npos) {
				// Parse orthogonal simulation box size.
				for(size_t k=0; k<3; k++) {
					if(sscanf(stream.readline().c_str(), FLOAT_SCANF_STRING_2, &simBox.minc[k], &simBox.maxc[k]) != 2)
						throw Exception(tr("Invalid box size in line %1 of dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				}
				destination->simulationCell()->setBoxShape(simBox);
				break;
			}
			else if(stream.line().find("ITEM: TILT FACTORS") != string::npos) {
				OVITO_ASSERT(!simBox.isEmpty());
				FloatType xy, xz, yz;
				if(sscanf(stream.readline().c_str(), FLOAT_SCANF_STRING_3, &xy, &xz, &yz) != 3)
					throw Exception(tr("Invalid tilt factors in line %1 of dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				Vector3 a1(simBox.sizeX(), 0, 0);
				Vector3 a2(xy, simBox.sizeY(), 0);
				Vector3 a3(xz, yz, simBox.sizeZ());
				destination->simulationCell()->setCellShape(simBox.minc, a1, a2, a3);
				break;
			}
			else if(stream.line().find("ITEM: PERIODIC BOUNDARY CONDITIONS") != string::npos) {
				// Parse PBC flags
				int pbcFlags[3];
				if(sscanf(stream.readline().c_str(), "%u %u %u", &pbcFlags[0], &pbcFlags[1], &pbcFlags[2]) != 3)
					throw Exception(tr("Invalid periodic boundary condition flags in line %1 of dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
				destination->simulationCell()->setPeriodicity(pbcFlags[0], pbcFlags[1], pbcFlags[2]);
				break;
			}
			else if(stream.line().find("ITEM: ATOMS") != string::npos) {
				// Parse atom coordinates.
				destination->setAtomsCount(numAtoms);

				// Prepare the mapping between input file columns and data channels.
				DataRecordParserHelper recordParser(&columnMapping(), destination);

				// Parse one atom per line.
				for(int i = 0; i < numAtoms; i++) {

					// Update progress indicator.
					if((i % 4096) == 0) {
						progress.setValue(i);
						if(progress.isCanceled()) return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
					}

					stream.readline();
					try {
						recordParser.storeAtom(i, (char*)stream.line().c_str());
					}
					catch(Exception& ex) {
						throw ex.prependGeneralMessage(tr("Parsing error in line %1 of LAMMPS dump file.").arg(stream.lineNumber()));
					}
				}

				if(recordParser.coordinatesOutOfRange()) {
					MsgLogger() << "WARNING: At least some of the atomic coordinates are out of the valid range." << endl;
					if(APPLICATION_MANAGER.guiMode() && QMessageBox::warning(NULL, tr("Warning"), tr("At least some of the atomic coordinates are out of the valid range. Do you want to ignore this and still load the dataset?"),
							QMessageBox::Ignore | QMessageBox::Cancel, QMessageBox::Cancel) == QMessageBox::Cancel)
					return EvaluationStatus(EvaluationStatus::EVALUATION_ERROR);
				}

				DataChannel* posChannel = destination->getStandardDataChannel(DataChannel::PositionChannel);
				if(posChannel && posChannel->size() > 1) {
					// Rescale atoms if they are given in reduced coordinates.
					const Box3& boundingBox = recordParser.boundingBox();
					if(Box3(Point3(-0.05), Point3(1.05)).containsBox(boundingBox)) {
						VerboseLogger() << "Rescaling reduced coordinates in interval [0,1] to absolute box size." << endl;
						AffineTransformation simCell = destination->simulationCell()->cellMatrix();
						Point3* p = posChannel->dataPoint3();
						for(size_t i = posChannel->size(); i != 0; --i, ++p)
							*p = simCell * (*p);
					}
				}

				// Stop parsing here
				destination->invalidate();

				QString statusMessage = tr("%1 atoms at timestep %2").arg(numAtoms).arg(timeStep);
				return EvaluationStatus(EvaluationStatus::EVALUATION_SUCCESS, statusMessage);
			}
			else if(stream.line().find("ITEM:") != string::npos) {
				// Skip lines up to next ITEM:
				while(!stream.eof()) {
					stream.readline();
					if(stream.line().find("ITEM:") != string::npos) break;
				}
			}
			else if(stream.line().find_first_not_of(" \t\n\r") == string::npos) {
				// Skip empty lines
				break;
			}
			else {
				throw Exception(tr("LAMMPS dump file parsing error. Invalid line %1 in dump file: %2").arg(stream.lineNumber()).arg(stream.line().c_str()));
			}
		}
		while(!stream.eof());
	}

	destination->invalidate();
	throw Exception(tr("LAMMPS dump file parsing error. No ATOMS section found in input file."));
}

};	// End of namespace AtomViz
