///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/units/ParameterUnit.h>
#include <atomviz/atoms/AtomsObject.h>
#include <atomviz/atoms/datachannels/PositionDataChannel.h>
#include <core/gui/properties/FloatPropertyUI.h>
#include <boost/iterator/counting_iterator.hpp>

#include "NearestNeighborList.h"
#include "ChemicalElements.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(NearestNeighborList, RefTarget)
DEFINE_PROPERTY_FIELD(NearestNeighborList, "NearestNeighborCutoff", _nearestNeighborCutoff)
SET_PROPERTY_FIELD_LABEL(NearestNeighborList, _nearestNeighborCutoff, "Cutoff radius")
SET_PROPERTY_FIELD_UNITS(NearestNeighborList, _nearestNeighborCutoff, WorldParameterUnit)

/******************************************************************************
* Constructor
******************************************************************************/
NearestNeighborList::NearestNeighborList(bool isLoading) : RefTarget(isLoading),
	_nearestNeighborCutoff(0.0)
{
	INIT_PROPERTY_FIELD(NearestNeighborList, _nearestNeighborCutoff);

	if(!isLoading) {
		// Use the default cutoff radius stored in the application settings.
		QSettings settings;
		settings.beginGroup("atomviz/neigborlist");
		setNearestNeighborCutoff(settings.value("DefaultCutoff", 0.0).value<FloatType>());
		settings.endGroup();
	}
}

/******************************************************************************
* Computes the nearest neighbor list.
* Throws an exception on error.
* Returns false when the operation has been canceled by the user.
******************************************************************************/
bool NearestNeighborList::build(AtomsObject* input, bool noProgressIndicator)
{
	CHECK_OBJECT_POINTER(input);
	clear();

	PositionDataChannel* posChannel = static_object_cast<PositionDataChannel>(input->getStandardDataChannel(DataChannel::PositionChannel));
	if(!posChannel) throw Exception(tr("Input object does not contain atomic positions. Position channel is missing."));

	FloatType cutoffRadius = nearestNeighborCutoff();
	FloatType cutoffRadiusSquared = cutoffRadius * cutoffRadius;

	if(cutoffRadius <= 0.0)
		throw Exception(tr("Invalid parameter: Nearest-neighbor cutoff radius must be positive."));

	AffineTransformation simCell = input->simulationCell()->cellMatrix();
	if(simCell.determinant() <= 1e-5f)
		throw Exception(tr("Simulation cell is degenerate."));
	AffineTransformation simCellInverse = simCell.inverse();
	array<bool,3> pbc = input->simulationCell()->periodicity();

	// Calculate the number of bins required in each spatial direction.
	int binDim[3] = {1,1,1};
	if(cutoffRadius > 0.0) {
		AffineTransformation m = AffineTransformation::scaling(cutoffRadius) * simCellInverse;
		Matrix3 binCell;
		for(size_t i=0; i<3; i++) {
			binDim[i] = (int)(Length(simCell.column(i)) / cutoffRadius);
			binDim[i] = min(binDim[i], (int)(1.0 / Length(m.column(i))));
			binDim[i] = min(binDim[i], 50);

			// Only accept an even number of bins (exception is exactly 1 bin) to avoid problems with the parallel processing and periodic boundary conditions.
			binDim[i] &= ~1;

			if(binDim[i] < 1) {
				if(pbc[i])
					throw Exception(tr("Periodic simulation cell is smaller than the neighbor cutoff radius. Minimum image convention cannot be used with such a small simulation box."));
				binDim[i] = 1;
			}
			binCell.column(i) = simCell.column(i) / (FloatType)binDim[i];
		}
	}

	// Show progress dialog.
	scoped_ptr<ProgressIndicator> progress;
	if(!noProgressIndicator)
		progress.reset(new ProgressIndicator(tr("Building nearest-neighbor lists (using %n processor(s))", NULL, QThread::idealThreadCount())));

	/// An 3d array of cubic bins. Each bin is a linked list of atoms.
	BinsArray bins(extents[binDim[0]][binDim[1]][binDim[2]]);

	// Clear bins.
	typedef BinsArray::iterator iterator1;
	typedef subarray_gen<BinsArray,2>::type::iterator iterator2;
	typedef subarray_gen<BinsArray,1>::type::iterator iterator3;
	for(iterator1 iter1 = bins.begin(); iter1 != bins.end(); ++iter1)
		for(iterator2 iter2 = (*iter1).begin(); iter2 != (*iter1).end(); ++iter2)
			for(iterator3 iter3 = (*iter2).begin(); iter3 != (*iter2).end(); ++iter3)
				(*iter3) = NULL;

	// Allocate output array.
	atoms.resize(input->atomsCount());

	// Measure computation time.
	QTime timer;
	timer.start();

	// Sort atoms into bins.
	const Point3* p = posChannel->constDataPoint3();
	QVector<NeighborListAtom>::iterator a = atoms.begin();
	int atomIndex = 0;
	for(; a != atoms.end(); ++a, ++p, ++atomIndex) {
		a->index = atomIndex;

		// Transform atom position from absolute coordinates to reduced coordinates.
		a->pos = *p;
		Point3 reducedp = simCellInverse * (*p);

		int indices[3];
		for(size_t k=0; k<3; k++) {
			// Shift atom position to make it be inside simulation cell.
			if(pbc[k]) {
				while(reducedp[k] < 0) {
					reducedp[k] += 1;
					a->pos += simCell.column(k);
				}
				while(reducedp[k] > 1) {
					reducedp[k] -= 1;
					a->pos -= simCell.column(k);
				}
			}
			else {
				reducedp[k] = max(reducedp[k], (FloatType)0);
				reducedp[k] = min(reducedp[k], (FloatType)1);
			}

			// Determine the atom's bin from its reduced position in the simulation cell.
			indices[k] = (int)(reducedp[k] * binDim[k]);
			if(indices[k] == binDim[k]) indices[k] = binDim[k]-1;
			OVITO_ASSERT(indices[k] >= 0 && indices[k] < (int)bins.shape()[k]);
		}

		// Put atom into its bin.
		NeighborListAtom** binList = &bins[indices[0]][indices[1]][indices[2]];
		a->nextInBin = *binList;
		*binList = &*a;
	}

	VerboseLogger() << "Neighbor list binning took" << timer.restart() << "msec." << endl;

	Vector3I offset;
	for(offset.X = 0; offset.X <= 1; offset.X++) {
		for(offset.Y = 0; offset.Y <= 1; offset.Y++) {
			for(offset.Z = 0; offset.Z <= 1; offset.Z++) {

				// Put together list of bins to process in this iteration.
				QVector<Point3I> binsToProcess;
				binsToProcess.reserve(binDim[0]*binDim[1]*binDim[2]/8);
				Point3I binIndex;
				for(binIndex.X = offset.X; binIndex.X < binDim[0]; binIndex.X += 2)
					for(binIndex.Y = offset.Y; binIndex.Y < binDim[1]; binIndex.Y += 2)
						for(binIndex.Z = offset.Z; binIndex.Z < binDim[2]; binIndex.Z += 2)
							binsToProcess.push_back(binIndex);

				// Execute neighbor list code for each atom in a parallel fashion.
				Kernel kernel(bins, simCell, pbc, cutoffRadiusSquared, offset);
				QFuture<void> future = QtConcurrent::map(binsToProcess, kernel);
				if(progress) {
					progress->setLabelText(tr("Building nearest-neighbor lists (%1/8) (using %n processor(s))", NULL, QThread::idealThreadCount()).arg(offset.X*4+offset.Y*2+offset.Z+1));
					progress->waitForFuture(future);
				}
				else
					future.waitForFinished();

				// Throw away results obtained so far if the user cancels the calculation.
				if(future.isCanceled()) {
					clear();
					return false;
				}
			}
		}
	}

	VerboseLogger() << "Neighbor list building took" << timer.elapsed() << "msec." << endl;

	return true;
}

static const Vector3I stencils[][2] = {
		{ Vector3I(0,0,0), Vector3I(0,0,0) },
		{ Vector3I(0,0,0), Vector3I(0,0,1) },
		{ Vector3I(0,0,0), Vector3I(0,1,0) },
		{ Vector3I(0,0,0), Vector3I(0,1,1) },
		{ Vector3I(0,0,0), Vector3I(1,0,0) },
		{ Vector3I(0,0,0), Vector3I(1,0,1) },
		{ Vector3I(0,0,0), Vector3I(1,1,0) },
		{ Vector3I(0,0,0), Vector3I(1,1,1) },

		{ Vector3I(1,0,0), Vector3I(0,1,0) },	// -1  1  0
		{ Vector3I(1,0,0), Vector3I(0,0,1) },	// -1  0  1
		{ Vector3I(0,1,0), Vector3I(0,0,1) },   //  0 -1  1

		{ Vector3I(1,0,0), Vector3I(0,1,1) },   // -1  1  1
		{ Vector3I(0,1,0), Vector3I(1,0,1) },   //  1 -1  1
		{ Vector3I(0,0,1), Vector3I(1,1,0) },   //  1  1 -1
};

/******************************************************************************
* Finds the neighbors for all atoms in a single bin.
******************************************************************************/
void NearestNeighborList::Kernel::operator()(const Point3I& binOrigin)
{
	OVITO_ASSERT(binOrigin.X < bins.shape()[0]);
	OVITO_ASSERT(binOrigin.Y < bins.shape()[1]);
	OVITO_ASSERT(binOrigin.Z < bins.shape()[2]);

	for(size_t stencilIndex = 0; stencilIndex < sizeof(stencils)/sizeof(stencils[0]); ++stencilIndex) {
		Point3I bin1 = binOrigin + stencils[stencilIndex][0];
		Point3I bin2 = binOrigin + stencils[stencilIndex][1];
		Vector3 pbcOffset(NULL_VECTOR);
		bool skipStencil = false;
		for(size_t k = 0; k < 3; k++) {
			if(bin1[k] == bins.shape()[k]) {
				if(!pbc[k]) { skipStencil = true; break; }
				bin1[k] = 0;
				pbcOffset += simCell.column(k);
			}
			if(bin2[k] == bins.shape()[k]) {
				if(!pbc[k]) { skipStencil = true; break; }
				bin2[k] = 0;
				pbcOffset -= simCell.column(k);
			}
		}
		if(skipStencil) continue;

		int numNeighborPairs = 0;
		int numAtoms = 0;
		for(NeighborListAtom* atom1 = bins[bin1.X][bin1.Y][bin1.Z]; atom1 != NULL; atom1 = atom1->nextInBin) {
			NeighborListAtom* atom2 = bins[bin2.X][bin2.Y][bin2.Z];
			if(bin1 == bin2) atom2 = atom1->nextInBin;
			for(; atom2 != NULL; atom2 = atom2->nextInBin) {
				Vector3 delta = atom1->pos - atom2->pos + pbcOffset;
				if(LengthSquared(delta) > cutoffRadiusSquared) continue;

				OVITO_ASSERT(atom1 != atom2);
				atom1->neighbors.append(atom2);
				atom2->neighbors.append(atom1);
				numNeighborPairs++;
			}
			numAtoms++;

			if(numNeighborPairs > numAtoms * 200)
				throw Exception(tr("The average number of nearest neighbors per atom exceeds the reasonable limit. Atomic positions seem to be invalid or cutoff radius too large."));
		}
	}
}

IMPLEMENT_PLUGIN_CLASS(NearestNeighborListEditor, PropertiesEditor)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void NearestNeighborListEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Nearest Neighbor List"), rolloutParams);

    // Create the rollout contents.
	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Cutoff parameter.
	FloatPropertyUI* cutoffRadiusPUI = new FloatPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(NearestNeighborList, _nearestNeighborCutoff));
	layout->addWidget(cutoffRadiusPUI->label(), 0, 0);
	layout->addWidget(cutoffRadiusPUI->textBox(), 0, 1);
	layout->addWidget(cutoffRadiusPUI->spinner(), 0, 2);
	cutoffRadiusPUI->setMinValue(0);
	connect(cutoffRadiusPUI->spinner(), SIGNAL(spinnerValueChanged()), this, SLOT(memorizeCutoff()));

	// Selection box for predefined cutoff radii.
	nearestNeighborPresetsBox = new QComboBox(rollout);
	nearestNeighborPresetsBox->addItem(tr("Choose..."));
	for(size_t i=0; i<NumberOfChemicalElements; i++) {
		if(ChemicalElements[i].structure == ChemicalElement::FaceCenteredCubic) {
			FloatType r = ChemicalElements[i].latticeParameter * 0.5 * (1.0 + 1.0/sqrt(2.0));
			nearestNeighborPresetsBox->addItem(QString("%1 (%2) - FCC - %3").arg(ChemicalElements[i].elementName).arg(i).arg(r, 0, 'f', 2), r);
		}
		else if(ChemicalElements[i].structure == ChemicalElement::BodyCenteredCubic) {
			FloatType r = ChemicalElements[i].latticeParameter * (1.0 + (sqrt(2.0)-1.0)*0.5);
			nearestNeighborPresetsBox->addItem(QString("%1 (%2) - BCC - %3").arg(ChemicalElements[i].elementName).arg(i).arg(r, 0, 'f', 2), r);
		}
	}
	layout->addWidget(new QLabel(tr("Presets:")), 1, 0);
	layout->addWidget(nearestNeighborPresetsBox, 1, 1, 1, 2);
	connect(nearestNeighborPresetsBox, SIGNAL(activated(int)), this, SLOT(onSelectNearestNeighborPreset(int)));
}

/******************************************************************************
* Is called when the user has selected an item in the radius preset box.
******************************************************************************/
void NearestNeighborListEditor::onSelectNearestNeighborPreset(int index)
{
	FloatType r = nearestNeighborPresetsBox->itemData(index).value<FloatType>();
	if(r != 0) {
		if(!editObject()) return;
		NearestNeighborList* obj = static_object_cast<NearestNeighborList>(editObject());
		UNDO_MANAGER.beginCompoundOperation(tr("Change Cutoff Radius"));
		obj->setNearestNeighborCutoff(r);
		UNDO_MANAGER.endCompoundOperation();
		memorizeCutoff();
	}
	nearestNeighborPresetsBox->setCurrentIndex(0);
}


/******************************************************************************
* Stores the current cutoff radius in the application settings
* so it can be used as default value for new neighbor lists.
******************************************************************************/
void NearestNeighborListEditor::memorizeCutoff()
{
	if(!editObject()) return;
	NearestNeighborList* nnList = static_object_cast<NearestNeighborList>(editObject());

	QSettings settings;
	settings.beginGroup("atomviz/neigborlist");
	settings.setValue("DefaultCutoff", nnList->nearestNeighborCutoff());
	settings.endGroup();
}

};	// End of namespace AtomViz

