///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ActionManager.h>
#include <core/viewport/Window3DContainer.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/utilities/ProgressIndicator.h>
#include <core/data/DataSetManager.h>
#include <core/rendering/RenderSettings.h>
#include <core/rendering/PluginRenderer.h>
#include <core/rendering/FrameBuffer.h>
#include <core/rendering/FrameBufferWindow.h>
#include "StdActions.h"

namespace Core {

/// Constructs the action objects and registers them with the ActionManager.
RenderingActionsHandler::RenderingActionsHandler()
{
	connect(addCommandAction(ACTION_RENDER_PREVIEW, tr("Render Preview")), SIGNAL(triggered(bool)), this, SLOT(OnRenderPreview()));
	connect(addCommandAction(ACTION_RENDER_ACTIVE_VIEWPORT, tr("Render Active Viewport"), ":/core/rendering/render_active_viewport.png"), SIGNAL(triggered(bool)), this, SLOT(OnRenderActiveViewport()));
	connect(addCommandAction(ACTION_SELECT_RENDERER_DIALOG, tr("Select Renderer")), SIGNAL(triggered(bool)), this, SLOT(OnSelectRendererDialog()));
}

/// Creates and registers a single command action.
ActionProxy* RenderingActionsHandler::addCommandAction(const QString& id, const QString& title, const char* iconPath, const QKeySequence& shortcut)
{
	ActionProxy* proxy = ACTION_MANAGER.addAction(new Action(id), title, iconPath);
	if(!shortcut.isEmpty()) proxy->setShortcut(shortcut);
	return proxy;
}

/// Handles ACTION_RENDER_PREVIEW command.
void RenderingActionsHandler::OnRenderPreview()
{
#if 0
	Window3DContainer* win3DContainer = new Window3DContainer();
	win3DContainer->resize(640, 480);
	Viewport* viewport = new Viewport(win3DContainer);
	viewport->SetRect(win3DContainer->rect());
	viewport->SetViewType(VIEWPRT_PERSPECTIVE);
	viewport->SetShadingMode(VPSHADING_SHADED);
	viewport->GetGrid().SetVisible(true);
	viewport->SetZoom(FLOATTYPE_PI/3.0);
	viewport->SetViewMatrix(Matrix::LookAt(Point3(70,-100,80), ORIGIN, Vector3(0,0,1)));
	qDebug() << "Rendering Pixmap:";

	QPixmap pixmap = win3DContainer->renderPixmap();
	pixmap.save("preview.png");
#endif
}

/// Handles ACTION_RENDER_ACTIVE_VIEWPORT command.
void RenderingActionsHandler::OnRenderActiveViewport()
{
	try {
		// Get the renderer instance.
		RenderSettings* settings = DATASET_MANAGER.currentSet()->renderSettings();
		if(!settings || !settings->renderer()) throw Exception(tr("No renderer has been selected."));
		PluginRenderer* renderer = settings->renderer();

		// Open progress dialog.
		ProgressIndicator progress(tr("Rendering ..."), 1);
		progress.isCanceled();

		// Allocate the frame buffer.
		shared_ptr<FrameBuffer> frameBuffer(new FrameBuffer(settings->imageWidth(), settings->imageHeight()));
		FrameBufferWindow* display = NULL;

		bool ok;
		try {
			// Do not update the viewports while rendering.
			ViewportSuspender noVPUpdates;

			// Initialize the renderer.
			if(ok = renderer->startRender(DATASET_MANAGER.currentSet())) {

				TimeTicks renderTime = ANIM_MANAGER.time();
				if(settings->renderingRangeType() == RenderSettings::ANIMATION_INTERVAL) {
					renderTime = ANIM_MANAGER.animationInterval().start();
					progress.setMaximum(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.animationInterval().duration()) + 1);
				}

				for(;;) {
					if(progress.isCanceled()) break;
					if(settings->renderingRangeType() == RenderSettings::ANIMATION_INTERVAL) {
						progress.setLabelText(tr("Rendering frame %1").arg(ANIM_MANAGER.timeToFrame(renderTime)));
						progress.setValue(ANIM_MANAGER.timeToFrame(renderTime - ANIM_MANAGER.animationInterval().start()));
					}

					// Setup camera view.
					ViewportRecord::SmartPtr viewportSettings;
					if(APPLICATION_MANAGER.guiMode()) {
						Viewport* vp = VIEWPORT_MANAGER.activeViewport();
						if(vp) viewportSettings = vp->settings();
					}
					else {
						viewportSettings = DATASET_MANAGER.currentSet()->viewportConfig()->activeViewportSettings();
					}
					if(!viewportSettings) throw Exception(tr("There is no active viewport."));
					Box3 sceneBoundingBox;
				    SceneRenderer* viewportRenderer = SceneRenderer::activeRenderer();
					if(viewportRenderer)
						sceneBoundingBox = viewportRenderer->sceneExtents(viewportSettings.get(), renderTime, SceneRenderer::RENDERABLE_OBJECTS);
					FloatType aspectRatio = (FloatType)settings->imageHeight() / (FloatType)settings->imageWidth();
					CameraViewDescription view = viewportSettings->getViewDescription(renderTime, aspectRatio, sceneBoundingBox);

					// Render one frame.
					frameBuffer->clear();
					if(renderer->renderFrame(renderTime, view, frameBuffer.get()) == false)
						break;

					// Save rendered image to disk.
					if(settings->imageFilename().isEmpty() == false) {
						int frameNumber = ANIM_MANAGER.timeToFrame(renderTime);
						progress.setLabelText(tr("Saving rendered image to disk (Frame %1)").arg(frameNumber));

						QString imageFilename = settings->imageFilename();
						if(settings->renderingRangeType() != RenderSettings::CURRENT_FRAME) {
							// Append frame number to file name if rendering a movie.
							QFileInfo fileInfo(imageFilename);
							imageFilename = fileInfo.path() + QChar('/') + fileInfo.baseName() + QString("%1.").arg(frameNumber, 4, 10, QChar('0')) + fileInfo.completeSuffix();
						}

						if(!frameBuffer->image().save(imageFilename))
							throw Exception(tr("Failed to save rendered image to image file '%1'.").arg(imageFilename));
					}

					// Open a display window for the rendered frame.
					if(APPLICATION_MANAGER.guiMode()) {
						if(display == NULL) {
							display = new FrameBufferWindow();
							display->setFrameBuffer(frameBuffer);
							display->setWindowTitle(tr("Frame Buffer"));
							display->show();
							display->adjustSize();
						}
						display->update();
					}

					if(settings->renderingRangeType() == RenderSettings::CURRENT_FRAME)
						break;

					// Goto next animation frame.
					renderTime += ANIM_MANAGER.ticksPerFrame();
					if(renderTime > ANIM_MANAGER.animationInterval().end())
						break;
				}
			}
		}
		catch(...) {
			renderer->endRender();
			throw;
		}
		// Release renderer.
		renderer->endRender();
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/// Handles ACTION_SELECT_RENDERER_DIALOG command.
void RenderingActionsHandler::OnSelectRendererDialog()
{
	if(!DATASET_MANAGER.currentSet()) return;
	RenderSettings* renderSettings = DATASET_MANAGER.currentSet()->renderSettings();
	if(!renderSettings) return;

	QList<PluginClassDescriptor*> rendererClasses = PluginRenderer::availableRendererClasses();
	QStringList itemList;
	Q_FOREACH(PluginClassDescriptor* clazz, rendererClasses)
		itemList << clazz->schematicTitle();

	int currentIndex = 0;
	if(renderSettings->rendererClass())
		currentIndex = itemList.indexOf(renderSettings->rendererClass()->schematicTitle());

	bool ok;
	QString selectedClass = QInputDialog::getItem(NULL, tr("Choose renderer"), tr("Select the renderer implementation:"), itemList, currentIndex, false, &ok);
	if(!ok) return;

	UNDO_MANAGER.beginCompoundOperation(tr("Change renderer"));
	try {
		int newIndex = itemList.indexOf(selectedClass);
		if(newIndex != currentIndex)
			renderSettings->setRendererClass(rendererClasses[newIndex]);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
	UNDO_MANAGER.endCompoundOperation();
}

};
