///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/actions/ActionManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/NavigationModes.h>
#include "StdActions.h"

namespace Core {

/// Constructs the action objects and registers them with the ActionManager. 
ViewportActionsHandler::ViewportActionsHandler()
{
	connect(addCommandAction(ACTION_VIEWPORT_MAXIMIZE, tr("Maximize Active Viewport"), ":/core/viewport/maximize_viewport.png"), SIGNAL(triggered(bool)), this, SLOT(OnViewportMaximize()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS, tr("Zoom Scene Extents"), ":/core/viewport/zoom_scene_extents.png"), SIGNAL(triggered(bool)), this, SLOT(OnViewportZoomSceneExtents()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS, tr("Zoom Selection Extents"), ":/core/viewport/zoom_selection_extents.png"), SIGNAL(triggered(bool)), this, SLOT(OnViewportZoomSelectionExtents()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS_ALL, tr("Zoom All Scene Extents"), ":/core/viewport/zoom_all_scene_extents.png"), SIGNAL(triggered(bool)), this, SLOT(OnViewportZoomSceneExtentsAll()));
	connect(addCommandAction(ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS_ALL, tr("Zoom All Selection Extents"), ":/core/viewport/zoom_all_selection_extents.png"), SIGNAL(triggered(bool)), this, SLOT(OnViewportZoomSelectionExtentsAll()));

	addViewportModeAction(ACTION_VIEWPORT_ZOOM, ZoomMode::instance(), tr("Zoom"), ":/core/viewport/mode_zoom.png");
	addViewportModeAction(ACTION_VIEWPORT_PAN, PanMode::instance(), tr("Pan"), ":/core/viewport/mode_pan.png");
	addViewportModeAction(ACTION_VIEWPORT_ORBIT, OrbitMode::instance(), tr("Orbit"), ":/core/viewport/mode_orbit.png");
}

/// Creates and registers a single command action.
ActionProxy* ViewportActionsHandler::addCommandAction(const QString& id, const QString& title, const char* iconPath) 
{
	ActionProxy* proxy = ACTION_MANAGER.addAction(new Action(id), title, iconPath);
	return proxy;
}

/// Creates and registers a single viewport input mode action.
ViewportModeAction* ViewportActionsHandler::addViewportModeAction(const QString& id, ViewportInputHandler* handler, const QString& title, const char* iconPath) 
{
	ViewportModeAction* action = new ViewportModeAction(id, handler, QColor(255, 255, 200));
	ACTION_MANAGER.addAction(action, title, iconPath);
	return action;
}

/// Handles ACTION_VIEWPORT_MAXIMIZE command.
void ViewportActionsHandler::OnViewportMaximize() 
{
	if(VIEWPORT_MANAGER.maximizedViewport())
		VIEWPORT_MANAGER.setMaximizedViewport(NULL);
	else if(VIEWPORT_MANAGER.activeViewport())
		VIEWPORT_MANAGER.setMaximizedViewport(VIEWPORT_MANAGER.activeViewport());
}

/// Handles ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS commands.
void ViewportActionsHandler::OnViewportZoomSceneExtents() 
{
	if(VIEWPORT_MANAGER.activeViewport())
		VIEWPORT_MANAGER.activeViewport()->zoomToExtents(SceneRenderer::ALL);
}

/// Handles ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS commands.
void ViewportActionsHandler::OnViewportZoomSelectionExtents() 
{
	if(VIEWPORT_MANAGER.activeViewport())
		VIEWPORT_MANAGER.activeViewport()->zoomToExtents(SceneRenderer::SELECTED_OBJECTS);
}

/// Handles ACTION_VIEWPORT_ZOOM_SCENE_EXTENTS_ALL commands.
void ViewportActionsHandler::OnViewportZoomSceneExtentsAll() 
{
	Q_FOREACH(Viewport* vp, VIEWPORT_MANAGER.viewports())
		vp->zoomToExtents(SceneRenderer::ALL);
}

/// Handles ACTION_VIEWPORT_ZOOM_SELECTION_EXTENTS_ALL commands.
void ViewportActionsHandler::OnViewportZoomSelectionExtentsAll() 
{
	Q_FOREACH(Viewport* vp, VIEWPORT_MANAGER.viewports())
		vp->zoomToExtents(SceneRenderer::SELECTED_OBJECTS);
}

};
