///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "AnimationSettingsDialog.h"
#include <core/undo/UndoManager.h>
#include <core/data/DataSetManager.h>
#include <core/data/units/ParameterUnit.h>

namespace Core {

/******************************************************************************
* The constructor of the animation settings dialog.
******************************************************************************/
AnimationSettingsDialog::AnimationSettingsDialog(QWidget* parent) : 
	QDialog(parent)	
{
	setWindowTitle(tr("Animation Settings"));
	connect(this, SIGNAL(rejected()), this, SLOT(OnCancel()));
	
	QVBoxLayout* layout1 = new QVBoxLayout(this);
	
	QGridLayout* contentLayout = new QGridLayout();
	contentLayout->setContentsMargins(0,0,0,0);
	contentLayout->setSpacing(0);
	contentLayout->setColumnStretch(1, 1);
	layout1->addLayout(contentLayout);		    
	
	contentLayout->addWidget(new QLabel(tr("Frames Per Second:"), this), 0, 0);
	fpsBox = new QComboBox(this);
	QLocale locale;
	fpsBox->addItem(locale.toString(0.1), TICKS_PER_SECOND * 10);
	fpsBox->addItem(locale.toString(0.2), TICKS_PER_SECOND * 5);
	fpsBox->addItem(locale.toString(0.5), TICKS_PER_SECOND * 2);
	fpsBox->addItem(locale.toString(1), TICKS_PER_SECOND * 1);
	fpsBox->addItem(locale.toString(2), TICKS_PER_SECOND / 2);
	fpsBox->addItem(locale.toString(4), TICKS_PER_SECOND / 4);
	fpsBox->addItem(locale.toString(5), TICKS_PER_SECOND / 5);
	fpsBox->addItem(locale.toString(8), TICKS_PER_SECOND / 8);
	fpsBox->addItem(locale.toString(10), TICKS_PER_SECOND / 10);
	fpsBox->addItem(locale.toString(12), TICKS_PER_SECOND / 12);
	fpsBox->addItem(locale.toString(15), TICKS_PER_SECOND / 15);
	fpsBox->addItem(locale.toString(16), TICKS_PER_SECOND / 16);
	fpsBox->addItem(locale.toString(20), TICKS_PER_SECOND / 20);
	fpsBox->addItem(locale.toString(24), TICKS_PER_SECOND / 24);
	fpsBox->addItem(locale.toString(25), TICKS_PER_SECOND / 25);
	fpsBox->addItem(locale.toString(30), TICKS_PER_SECOND / 30);
	fpsBox->addItem(locale.toString(32), TICKS_PER_SECOND / 32);
	fpsBox->addItem(locale.toString(40), TICKS_PER_SECOND / 40);
	fpsBox->addItem(locale.toString(50), TICKS_PER_SECOND / 50);
	fpsBox->addItem(locale.toString(60), TICKS_PER_SECOND / 60);
	contentLayout->addWidget(fpsBox, 0, 1, 1, 2);
	connect(fpsBox, SIGNAL(activated(int)), this, SLOT(OnFramesPerSecondChanged(int)));

	contentLayout->addWidget(new QLabel(tr("Animation Start:"), this), 1, 0);
	QLineEdit* animStartBox = new QLineEdit(this);
	contentLayout->addWidget(animStartBox, 1, 1);
	animStartSpinner = new SpinnerWidget(this);
	animStartSpinner->setTextBox(animStartBox);
	animStartSpinner->setUnit(UNITS_MANAGER.getTimeParameterUnit());
	contentLayout->addWidget(animStartSpinner, 1, 2);

	contentLayout->addWidget(new QLabel(tr("Animation End:"), this), 2, 0);
	QLineEdit* animEndBox = new QLineEdit(this);
	contentLayout->addWidget(animEndBox, 2, 1);
	animEndSpinner = new SpinnerWidget(this);
	animEndSpinner->setTextBox(animEndBox);
	animEndSpinner->setUnit(UNITS_MANAGER.getTimeParameterUnit());
	contentLayout->addWidget(animEndSpinner, 2, 2);
	
	contentLayout->addWidget(new QLabel(tr("Playback Speed in Viewports:"), this), 3, 0);
	playbackSpeedBox = new QComboBox(this);
	playbackSpeedBox->addItem(tr("x 1/40"), -40);
	playbackSpeedBox->addItem(tr("x 1/20"), -20);
	playbackSpeedBox->addItem(tr("x 1/10"), -10);
	playbackSpeedBox->addItem(tr("x 1/5"), -5);
	playbackSpeedBox->addItem(tr("x 1/2"), -2);
	playbackSpeedBox->addItem(tr("x 1 (Realtime)"), 1);
	playbackSpeedBox->addItem(tr("x 2"), 2);
	playbackSpeedBox->addItem(tr("x 5"), 5);
	playbackSpeedBox->addItem(tr("x 10"), 10);
	playbackSpeedBox->addItem(tr("x 20"), 20);
	contentLayout->addWidget(playbackSpeedBox, 3, 1, 1, 2);
	connect(playbackSpeedBox, SIGNAL(activated(int)), this, SLOT(OnPlaybackSpeedChanged(int)));	
	
	QDialogButtonBox* buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this);
	connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept()));
	connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
	layout1->addWidget(buttonBox); 
	
    UpdateValues();

	UNDO_MANAGER.beginCompoundOperation(tr("Change Animation Settings"));
}

/******************************************************************************
* Destructor.
******************************************************************************/
AnimationSettingsDialog::~AnimationSettingsDialog()
{
	UNDO_MANAGER.endCompoundOperation();
}

/******************************************************************************
* Event handler for the CANCEL button.
******************************************************************************/
void AnimationSettingsDialog::OnCancel()
{
	// Undo all changes done so far.
	OVITO_ASSERT(UNDO_MANAGER.currentCompoundOperation());
	UNDO_MANAGER.currentCompoundOperation()->clear();
}

/******************************************************************************
* Updates the values shown in the dialog.
******************************************************************************/
void AnimationSettingsDialog::UpdateValues()
{
	fpsBox->setCurrentIndex(fpsBox->findData(ANIM_MANAGER.ticksPerFrame()));
	playbackSpeedBox->setCurrentIndex(playbackSpeedBox->findData(ANIM_MANAGER.playbackSpeed()));
	animStartSpinner->setIntValue(ANIM_MANAGER.animationInterval().start());
	animEndSpinner->setIntValue(ANIM_MANAGER.animationInterval().end());
}

/******************************************************************************
* Is called when the user has selected a new value for the frames per seconds.
******************************************************************************/
void AnimationSettingsDialog::OnFramesPerSecondChanged(int index)
{
	int newTicksPerFrame = fpsBox->itemData(index).toInt();
	OVITO_ASSERT(newTicksPerFrame != 0);
	
	// Change the animation speed.
	int oldTicksPerFrame = ANIM_MANAGER.ticksPerFrame();
	ANIM_MANAGER.setTicksPerFrame(newTicksPerFrame);

#if 0	
	// Rescale animation interval and animation keys.
	TimeInterval oldInterval = ANIM_MANAGER.animationInterval();
	TimeInterval newInterval;
	newInterval.setStart(oldInterval.start() * oldTicksPerFrame / newTicksPerFrame);
	newInterval.setEnd(oldInterval.end() * oldTicksPerFrame / newTicksPerFrame);
	ANIM_MANAGER.setAnimationInterval(newInterval);	
	DATASET_MANAGER.currentSet()->rescaleTime(oldInterval, newInterval);
#endif
	
	// Update dialog controls to reflect new values.
	UpdateValues();
}

/******************************************************************************
* Is called when the user has selected a new value for the playback speed.
******************************************************************************/
void AnimationSettingsDialog::OnPlaybackSpeedChanged(int index)
{
	int newPlaybackSpeed = playbackSpeedBox->itemData(index).toInt();
	OVITO_ASSERT(newPlaybackSpeed != 0);
	
	// Change the animation speed.
	ANIM_MANAGER.setPlaybackSpeed(newPlaybackSpeed);
	
	// Update dialog controls to reflect new values.
	UpdateValues();
}

};
