///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_REFTARGET_LIST_PARAMETER_UI_H
#define __OVITO_REFTARGET_LIST_PARAMETER_UI_H

#include <core/Core.h>
#include "ParameterUI.h"
#include "PropertiesEditor.h"

namespace Core {

/******************************************************************************
* A list view that shows the RefTarget items contained in a vector reference field.
******************************************************************************/
class CORE_DLLEXPORT RefTargetListParameterUI : public ParameterUI
{
public:
	/// Constructor.
	RefTargetListParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField,
			const RolloutInsertionParameters& rolloutParams = RolloutInsertionParameters(), PluginClassDescriptor* defaultEditorClass = NULL);
	
	/// Destructor, that releases all GUI controls.
	virtual ~RefTargetListParameterUI();
	
	/// The reference field that specifies the vector reference field of the edited object that
	/// is bound to this parameter UI.
	const PropertyFieldDescriptor& referenceField() const { return _refField; }
	
	/// This returns the list view managed by this ParameterUI.
	QListView* listWidget() const { return _listWidget; }
	
	/// Returns the RefTarget that is currently selected in the UI.
	RefTarget* selectedObject() const;
	
	/// Selectes the given sub-object in the list.
	int setSelectedObject(RefTarget* selObj);

	/// This method is called when a new editable object has been assigned to the properties owner this
	/// parameter UI belongs to. The parameter UI should react to this change appropriately and
	/// show the properties value for the new edit object in the UI.
	virtual void resetUI();

	/// Returns the current sub-editor for the selected sub-object or NULL if there is none.
	PropertiesEditor* subEditor() const { return _subEditor.get(); }

public:
	
	Q_PROPERTY(QListView listWidget READ listWidget)
	Q_PROPERTY(RefTarget selectedObject READ selectedObject)
	
private:

	class ListViewModel : public QAbstractListModel {
	public:
		ListViewModel(RefTargetListParameterUI* owner) : QAbstractListModel(owner) {}
		RefTargetListParameterUI* getOwner() const { return (RefTargetListParameterUI*)QObject::parent(); }
		
		virtual int rowCount(const QModelIndex &parent = QModelIndex()) const { return getOwner()->_rowToTarget.size(); }
		virtual QVariant data(const QModelIndex &index, int role) const;
		
		void updateItem(size_t itemIndex) {
			QModelIndex modelIndex = index(itemIndex);
			dataChanged(modelIndex, modelIndex);
		}
		void resetList() { reset(); }
		
		void beginInsert(int atIndex) {
			beginInsertRows(QModelIndex(), atIndex, atIndex);
		}
		void endInsert() {
			endInsertRows();
		}

		void beginRemove(int atIndex) {
			beginRemoveRows(QModelIndex(), atIndex, atIndex);
		}
		void endRemove() {
			endRemoveRows();
		}
	};
	
protected Q_SLOTS:

	/// Is called when the user has selected an item in the list view.
	void onListSelectionChanged();
	
protected:

	/// This method is called when a reference target changes. 
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

	/// Returns the data stored under the given role for the given RefTarget.
	/// This method is part of the data model used by the list widget and can be overriden
	/// by sub-classes. The default implementation returns the title of the RefTarget 
	/// for the Qt::DisplayRole.
	virtual QVariant getItemData(RefTarget* target, const QModelIndex& index, int role);
	
	/// Opens a sub-editor for the object that is selected in the list view.
	virtual void openSubEditor();

	/// The reference field that specifies the parameter of the edited object that
	/// is bound to this parameter UI.
	const PropertyFieldDescriptor& _refField;
	
	/// The list view control.
	QPointer<QListView> _listWidget;
	
	/// This editor type is show if no entry is selected in the listbox.
	PluginClassDescriptor* _defaultEditorClass;

	/// The internal model used for the list view widget.
	ListViewModel* _model;

	/// The list of items in the list view.
	VectorReferenceField<RefTarget> _targets;

	/// Maps reference field indices to row indices.
	QVector<int> _targetToRow;

	/// Maps row indices to reference field indices.
	QVector<int> _rowToTarget;
	
	/// The editor for the selected sub-object.
	PropertiesEditor::SmartPtr _subEditor;	

	/// Controls where the sub-editor is opened and whether the sub-editor is opened in a collapsed state.
	RolloutInsertionParameters _rolloutParams;
	
private:

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(RefTargetListParameterUI)
	DECLARE_VECTOR_REFERENCE_FIELD(_targets)
};

};

#endif // __OVITO_REFTARGET_LIST_PARAMETER_UI_H
