///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/SubObjectParameterUI.h>
#include <core/gui/properties/ColorControllerUI.h>
#include <core/gui/properties/IntegerPropertyUI.h>
#include <core/gui/properties/StringPropertyUI.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/IntegerRadioButtonPropertyUI.h>
#include <core/rendering/RenderSettings.h>
#include <core/rendering/RenderSettingsEditor.h>

namespace Core {

IMPLEMENT_PLUGIN_CLASS(RenderSettingsEditor, PropertiesEditor)

/******************************************************************************
* Constructor that creates the UI controls for the editor.
******************************************************************************/
void RenderSettingsEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("Render settings"), rolloutParams, "core.rendering.render_settings");
	
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);

	// Rendering range type
	{
		QGroupBox* groupBox = new QGroupBox(tr("Rendering range"));
		layout->addWidget(groupBox);
		QGridLayout* layout2 = new QGridLayout(groupBox);
		layout2->setContentsMargins(4,4,4,4);
		layout2->setSpacing(2);
		IntegerRadioButtonPropertyUI* renderingRangeTypeUI = new IntegerRadioButtonPropertyUI(this, "renderingRangeType");
		QRadioButton* currentFrameButton = renderingRangeTypeUI->addRadioButton(RenderSettings::CURRENT_FRAME);
		currentFrameButton->setText(tr("Current frame"));
		layout2->addWidget(currentFrameButton, 0, 0);
		QRadioButton* animationIntervalButton = renderingRangeTypeUI->addRadioButton(RenderSettings::ANIMATION_INTERVAL);
		animationIntervalButton->setText(tr("Complete animation"));
		layout2->addWidget(animationIntervalButton, 0, 1, 1, 3);
	}

	// Image properties
	{
		QGroupBox* groupBox = new QGroupBox(tr("Image output"));
		layout->addWidget(groupBox);
		QGridLayout* layout2 = new QGridLayout(groupBox);
		layout2->setContentsMargins(4,4,4,4);
		layout2->setSpacing(2);

		// Image Width parameter.
		IntegerPropertyUI* imageWidthUI = new IntegerPropertyUI(this, "imageWidth", tr("Image width:"));
		layout2->addWidget(imageWidthUI->label(), 0, 0);
		layout2->addWidget(imageWidthUI->textBox(), 0, 1);
		layout2->addWidget(imageWidthUI->spinner(), 0, 2);
		imageWidthUI->setMinValue(1);
	
		// Image Height parameter.
		IntegerPropertyUI* imageHeightUI = new IntegerPropertyUI(this, "imageHeight", tr("Image height:"));
		layout2->addWidget(imageHeightUI->label(), 1, 0);
		layout2->addWidget(imageHeightUI->textBox(), 1, 1);
		layout2->addWidget(imageHeightUI->spinner(), 1, 2);
		imageHeightUI->setMinValue(1);
	
		// Background color parameter.
		ColorControllerUI* backgroundColorPUI = new ColorControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _backgroundColor));
		layout2->addWidget(backgroundColorPUI->label(), 2, 0);
		layout2->addWidget(backgroundColorPUI->colorPicker(), 2, 1, 1, 2);
	
		// Alpha channel.
		BooleanPropertyUI* generateAlphaUI = new BooleanPropertyUI(this, "generateAlphaChannel", tr("Generate alpha channel"));
		layout2->addWidget(generateAlphaUI->checkBox(), 3, 0, 1, 3);
	
		// Output Filename parameter.
		StringPropertyUI* imageFilenameUI = new StringPropertyUI(this, "imageFilename");
		layout2->addWidget(new QLabel(tr("Output filename:")), 4, 0);
		QHBoxLayout* layout3 = new QHBoxLayout();
		layout3->setContentsMargins(0,0,0,0);
		layout3->setSpacing(0);
		layout2->addLayout(layout3, 4, 1, 1, 2);
		layout3->addWidget(imageFilenameUI->textBox(), 1);
		QPushButton* chooseFilenameBtn = new QPushButton("...", groupBox);
		chooseFilenameBtn->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred);
		chooseFilenameBtn->setMaximumWidth(20);
		connect(chooseFilenameBtn, SIGNAL(clicked(bool)), this, SLOT(onChooseImageFilename()));
		layout3->addWidget(chooseFilenameBtn);
	}

	// Open a sub-editor for the internal AtomsObject.
	new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(RenderSettings, _renderer), rolloutParams.after(rollout));
}

/******************************************************************************
* Lets the user choose a filename for the output image.
******************************************************************************/
void RenderSettingsEditor::onChooseImageFilename()
{
	RenderSettings* settings = static_object_cast<RenderSettings>(editObject());
	if(!settings) return;

	QString filter = tr("Images (*.png *.jpg *.jpeg *.bmp *.ppm *.xpm)"); 
	QString path = QFileDialog::getSaveFileName(container(), tr("Output Image"), QString(), filter);
	
	if(!path.isEmpty()) {
		settings->setImageFilename(path);
	}
}

};
