///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/ObjectLoadStream.h>
#include <core/data/ObjectSaveStream.h>
#include <core/data/DataSet.h>
#include <core/rendering/FrameBuffer.h>
#include <core/rendering/RenderSettings.h>
#include <core/utilities/ProgressIndicator.h>
#include <core/reference/CloneHelper.h>
#include <core/gui/ApplicationManager.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/animation/controller/StandardControllers.h>

#include "PreviewRenderer.h"
#include "PreviewRendererEditor.h"

namespace Core {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(PreviewRenderer, PluginRenderer)
DEFINE_PROPERTY_FIELD(PreviewRenderer, "AntialiasingLevel", _antialiasingLevel)
SET_PROPERTY_FIELD_LABEL(PreviewRenderer, _antialiasingLevel, "Antialiasing level")

/******************************************************************************
* Default constructor.
******************************************************************************/
PreviewRenderer::PreviewRenderer(bool isLoading)
	: PluginRenderer(isLoading), _antialiasingLevel(2)
{
	INIT_PROPERTY_FIELD(PreviewRenderer, _antialiasingLevel)
}

/******************************************************************************
* Prepares the renderer for rendering of the given scene.
******************************************************************************/
bool PreviewRenderer::startRender(DataSet* dataset)
{
	this->dataset = dataset;
	return true;
}

/******************************************************************************
* Renders a single animation frame into the given frame buffer.
******************************************************************************/
bool PreviewRenderer::renderFrame(TimeTicks time, CameraViewDescription view, FrameBuffer* frameBuffer)
{
	// Make sure that hardware-accelerated rendering is supported on this platform.
	if(!APPLICATION_MANAGER.guiMode())
		throw Exception(tr("Cannot render an image of the scene in non-gui mode using OpenGL. You have to use another renderer."));
	Window3D* renderingContainer = VIEWPORT_MANAGER.activeViewport();
	if(!renderingContainer || !renderingContainer->hasFrameBufferExtension())
		throw Exception(tr("Cannot render a preview of the scene. Your graphics card doesn't support the OpenGL GL_EXT_framebuffer_object extension."));

    // Create the offscreen framebuffer object - make sure to have a current
    // context before creating it.
	renderingContainer->makeCurrent();

	int antialiasingLevel = 1;
	if(_antialiasingLevel > 1)
		antialiasingLevel = _antialiasingLevel;

	GLuint fbWidth = renderSettings()->imageWidth() * antialiasingLevel;
	GLuint fbHeight = renderSettings()->imageHeight() * antialiasingLevel;
	GLuint oglFrameBuffer;
	GLuint oglRenderBuffer;
	GLuint oglDepthRenderBuffer;

	// Create a frame-buffer object and a render-buffer object for color and depth channel.
	VerboseLogger() << logdate << "Allocating OpenGL off-screen frame buffer:" << fbWidth << "x" << fbHeight << "pixels" << endl;
	CHECK_OPENGL(renderingContainer->glGenFramebuffersEXT(1, &oglFrameBuffer));
	CHECK_OPENGL(renderingContainer->glGenRenderbuffersEXT(1, &oglRenderBuffer));
	CHECK_OPENGL(renderingContainer->glGenRenderbuffersEXT(1, &oglDepthRenderBuffer));

	VerboseLogger() << logdate << "Binding OpenGL framebuffer." << endl;
	CHECK_OPENGL(renderingContainer->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, oglDepthRenderBuffer));
	renderingContainer->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT24, fbWidth, fbHeight);
	GLenum errorCode = glGetError();
	if(errorCode == GL_NO_ERROR) {
		CHECK_OPENGL(renderingContainer->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, oglRenderBuffer));
		renderingContainer->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_RGBA8, fbWidth, fbHeight);
		errorCode = glGetError();
	}
	if(errorCode != GL_NO_ERROR) {
		// Clean up
		renderingContainer->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);
		throw Exception(tr("Cannot render a preview of the scene. Failed to reserve an OpenGL offscreen buffer with the requested size. Please descrease the image resolution or the antialising level."));
	}

	// Activate the off-screen frame-buffer and associate the render-buffer objects.
	CHECK_OPENGL(renderingContainer->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, oglFrameBuffer));
	CHECK_OPENGL(renderingContainer->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_RENDERBUFFER_EXT, oglRenderBuffer));
	CHECK_OPENGL(renderingContainer->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, oglDepthRenderBuffer));

	// Check for errors...
	GLenum status = renderingContainer->glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if(status != GL_NO_ERROR && status != GL_FRAMEBUFFER_COMPLETE_EXT) {
		// Clean up
		renderingContainer->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);
		switch(status) {
			case GL_FRAMEBUFFER_UNSUPPORTED_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Unsupported OpenGL framebuffer format.\nTry to reduce the image resolution or the antialising level to fix this problem.");
			case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete attachment.");
			case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing attachment.");
#ifdef GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT
			case GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, duplicate attachment.");
#endif
			case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, attached images must have same dimensions.");
			case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, attached images must have same format.");
			case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing draw buffer.");
			case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT:
				throw Exception("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing read buffer.");
			default:
				throw Exception(tr("Cannot render a preview of the scene. Failed to initialize the OpenGL framebuffer. An undefined error has occurred: status=%1").arg(status));
		}
	}

	CHECK_OPENGL(glReadBuffer(GL_COLOR_ATTACHMENT0_EXT));

	// Prepare OpenGL rendering state.
	glPushAttrib(GL_LIGHTING_BIT|GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_ENABLE_BIT|GL_VIEWPORT_BIT|GL_TRANSFORM_BIT);
	glDisable(GL_SCISSOR_TEST);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glViewport(0, 0, fbWidth, fbHeight);
	glDisable(GL_ALPHA_TEST);
	glEnable(GL_LIGHTING);

	// Setup projection matrix.
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	Matrix4 tm = view.projectionMatrix;
	glLoadMatrix(tm.constData());
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// Initialize buffer with background color.
	Color backgroundColor(renderSettings()->backgroundColorController()->getValueAtTime(time));
	glClearColor(backgroundColor.r, backgroundColor.g, backgroundColor.b, renderSettings()->generateAlphaChannel() ? 0 : 1);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Setup lights.
	float ambientLight[4] = {0.2f, 0.2f, 0.2f, 1.0f};
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambientLight);
	float directLightColor[4] = {0.8f, 0.8f, 0.8f, 1.0f};
	glLightfv(GL_LIGHT0, GL_DIFFUSE, directLightColor);
	glLightfv(GL_LIGHT0, GL_SPECULAR, directLightColor);
	Vector3 lightDir = Normalize(Vector3(1.2f, -1.6f, 1.9f));
	float directLightDir1[4] = {lightDir.X, lightDir.Y, lightDir.Z, 0};
	glLightfv(GL_LIGHT0, GL_POSITION, directLightDir1);
	glEnable(GL_LIGHT0);
	float directLightDir2[4] = {-lightDir.X, -lightDir.Y, -lightDir.Z, 0};
	glLightfv(GL_LIGHT1, GL_POSITION, directLightDir2);
	glEnable(GL_LIGHT1);

	// Render scene objects.
	bool ok = renderObjects(time, view, fbWidth, fbHeight, renderingContainer);
	glFlush();

	if(ok) {
		// Read out the OpenGL framebuffer.
		QImage::Format image_format = QImage::Format_ARGB32_Premultiplied;
		QImage img(fbWidth, fbHeight, image_format);
		CHECK_OPENGL(glReadPixels(0, 0, fbWidth, fbHeight, GL_RGBA, GL_UNSIGNED_BYTE, img.bits()));
		// OpenGL gives ABGR (i.e. RGBA backwards); Qt wants ARGB
		img = img.rgbSwapped().mirrored().scaled(frameBuffer->image().width(), frameBuffer->image().height(), Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
		// Copy OpenGL image to the internal frame buffer.
		QPainter painter(&frameBuffer->image());
		painter.drawImage(0, 0, img);
	}

	// Restore old settings.
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
	glPopAttrib();

	// Make the normal window the target again.
	renderingContainer->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	// Clean up
	VerboseLogger() << logdate << "Deleting OpenGL framebuffer." << endl;
	renderingContainer->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
	renderingContainer->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
	renderingContainer->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);

	return ok;
}

/******************************************************************************
* Renders the scene objects.
******************************************************************************/
bool PreviewRenderer::renderObjects(TimeTicks time, const CameraViewDescription& view, int imageWidth, int imageHeight, Window3D* glcontext)
{
	SceneRoot* rootNode = dataset->sceneRoot();
	if(rootNode == NULL) return true;

	for(SceneNodesIterator iter(rootNode); !iter.finished(); iter.next()) {
		SceneNode* node = iter.current();
		if(!node->isObjectNode()) continue;

		ObjectNode* objNode = static_object_cast<ObjectNode>(node);
		// Evaluate geometry pipeline of object node.
		const PipelineFlowState& flowState = objNode->evalPipeline(time);
		if(flowState.result() != NULL) {
			CHECK_OBJECT_POINTER(flowState.result());

			// Setup transformation.
			TimeInterval iv;
			Matrix4 tm = view.viewMatrix * objNode->objectTransform() * objNode->getWorldTransform(time, iv);
			glLoadMatrix(tm.constData());

			// Render object node.
			if(!flowState.result()->renderPreview(time, view, objNode, imageWidth, imageHeight, glcontext))
				return false;
		}
	}
	return true;
}

/******************************************************************************
* Finishes the rendering pass. This is called after all animation frames have been rendered
* or when the rendering operation has been aborted.
******************************************************************************/
void PreviewRenderer::endRender()
{
}

};
