///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file SceneRenderer.h
 * \brief Contains the definition of the Core::SceneRenderer class.
 */

#ifndef __OVITO_SCENE_RENDERER_H
#define __OVITO_SCENE_RENDERER_H

#include <core/Core.h>
#include <core/scene/animation/TimeInterval.h>

namespace Core {

class Viewport;			// defined in Viewport.h
class PickRegion;		// defined in PickRegion.h
class SceneNode;		// defined in SceneNode.h
class ViewportRecord;	// defined in ViewportConfiguration.h

/**
 * \brief This is the base class for a service that renders the scene in
 *       a Viewport.
 *
 * Each Viewport uses an instance of the SceneRenderer class to render
 * the current scene. The SceneRenderer class itself is abstract and defines only an interface.
 * There is one default implementation of that interface: the DefaultSceneRenderer class.
 * By default an instance of this class is used by all viewports to render the scene.
 *
 * A plugin can override this behaviour by implementing a new SceneRenderer derived class
 * and activate it by passing an instance to the SceneRenderer::pushSceneRenderer() method.
 * From then on the custom scene renderer will be used by all viewports to render the scene.
 *
 * \author Alexander Stukowski
 * \sa Viewport
 */
class CORE_DLLEXPORT SceneRenderer : public QObject
{
	Q_OBJECT

public:

	/// Controls the behaviour of the sceneExtents() function.
	enum SceneExtentsMode {
		ALL,				//< The complete scene.
		SELECTED_OBJECTS,	//< Only the selected objects.
		RENDERABLE_OBJECTS,	//< Only objects that a renderable.
	};

public:

	/// \brief Is called by the Viewport class to let this scene renderer do its job.
	/// \param vp The viewport into which the scene will be rendered.
	/// \param time The animation at which the scene should be rendered.
	///
	/// This method calls renderScene() of the sub-class.
	void renderViewport(Viewport* vp, TimeTicks time) {
		viewport = vp;
		currentTime = time;
		renderScene();
	}


	/// \brief Performs hit testing on all visible nodes in the scene and returns a list of nodes that
	///        are within the given pick region.
	///
	/// \param time The animation time to use for picking.
	/// \param vp The viewport to use for hit testing.
	/// \param pickRegion The pick region to use for hit testing.
	/// \return The list of picked nodes.
	///
	QVector<SceneNode*> pickNodes(TimeTicks time, Viewport* vp, const PickRegion& pickRegion) {
		viewport = vp;
		currentTime = time;
		return hitTestSceneNodes(pickRegion);
	}

	/// \brief Computes the bounding box of the scene.
	/// \param settings The settings of the viewport which is used as context for the bounding box computation.
	/// \param time The animation at which to compute the scene bounding box.
	/// \param mode Specifies what should be included in the bounding box.
	/// \return The calculated bounding box.
	///
	/// \sa Viewport::zoomToExtents()
	virtual Box3 sceneExtents(ViewportRecord* settings, TimeTicks time, SceneExtentsMode mode) = 0;

protected:

	/// \brief Renders the scene.
	///
	/// This method must be implemented by sub-classes to do the custom rendering
	/// scheme.
	///
	/// \sa renderViewport()
	virtual void renderScene() = 0;

	/// \brief Performs hit testing on all visible nodes in the scene and returns a list of nodes that
	///        are within the given pick region.
	///
	/// \param pickRegion The pick region to use for hit testing.
	/// \return The list of picked nodes.
	///
	/// This virtual method must be implemented by sub-classes and is called by the pickNodes() method.
	/// \sa pickNodes()
	virtual QVector<SceneNode*> hitTestSceneNodes(const PickRegion& pickRegion) = 0;

protected:

	/// The current viewport into which the scene is rendered.
	Viewport* viewport;

	/// The current time at which the scene should be rendered.
	TimeTicks currentTime;

public:

	//////////////////////////// Static Members ////////////////////////////

	/// \brief Returns the active scene renderer.
	/// \return The renderer that is on top of the scene renderer stack.
	static SceneRenderer* activeRenderer();

	/// \brief Pushes a scene renderer onto the stack.
	/// \param renderer The renderer becomes the new active scene renderer.
	static void pushSceneRenderer(SceneRenderer* renderer);

	/// \brief Removes the given scene renderer from the stack of renderers.
	/// \param renderer The renderer to remove. It will be removed from stack even if it is
	///        not on top of the stack.
	static void popSceneRenderer(SceneRenderer* renderer);

private:

	/// Contains the stack of active scene renderers.
	static QVector<SceneRenderer*> renderers;
};

};

#endif // __OVITO_SCENE_RENDERER_H
