///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file Viewport.h
 * \brief Contains the definition of the Core::Viewport class and some associated classes.
 */

#ifndef __OVITO_VIEWPORT_H
#define __OVITO_VIEWPORT_H

#include <core/Core.h>
#include <mesh/tri/TriMesh.h>
#include <core/reference/RefTarget.h>
#include <core/scene/ObjectNode.h>
#include "Window3D.h"
#include "ViewportGrid.h"
#include "SceneRenderer.h"

namespace Core {

class ObjectNode;				// defined in ObjectNode.h
class ViewportRecord;			// defined in ViewportConfiguration.h

/// The default field of view in world units used for orthogonal view types when the scene is empty.
#define DEFAULT_ORTHOGONAL_FIELD_OF_VIEW		200.0
/// The default field of view in radians used for perspective view types when the scene is empty.
#define DEFAULT_PERSPECTIVE_FIELD_OF_VIEW	(FLOATTYPE_PI/3.0)

/******************************************************************************
* Defines a view projection for rendering.
******************************************************************************/
struct CameraViewDescription
{
	/// The aspect ratio (height/width) of the viewport.
	FloatType aspectRatio;
	/// Indicates whether this is a orthogonal or perspective projection.
	bool isPerspective;
	/// Contains the distance of the front clipping plane in world units.
	FloatType znear;
	/// Contains the distance of the back clipping plane in world units.
	FloatType zfar;
	/// If this is a orthogonal projection then this field contains the visible
	/// field of view in world units.
	/// If this is a perspective projection then this field contains the
	/// field of view angle in radians.
	FloatType fieldOfView;
	/// The world to view space transformation matrix.
	AffineTransformation viewMatrix;
	/// The view space to world space transformation matrix.
	AffineTransformation inverseViewMatrix;
	/// The view space to screen space projection matrix.
	Matrix4 projectionMatrix;
	/// The screen space to view space transformation matrix.
	Matrix4 inverseProjectionMatrix;
	/// The camera view does not change during this time interval.
	TimeInterval validityInterval;
};

/**
 * \brief A viewport window that displays the current scene.
 *
 * \author Alexander Stukowski
 * \sa ViewportPanel, ViewportManager
 */
class CORE_DLLEXPORT Viewport : public Window3D
{
	Q_OBJECT
public:
	/// Standard colors for drawing several things in the viewport.
	/// \sa getVPColor(), setVPColor()
	enum ViewportColor {
		COLOR_VIEWPORT_BKG,				//< The viewport background color
		COLOR_GRID,						//< The color of the minor construction grid lines
		COLOR_GRID_INTENS,				//< The color of the major construction grid lines
		COLOR_GRID_AXIS,				//< The color of the construction grid axis lines
		COLOR_VIEWPORT_CAPTION,			//< The color used to render the viewport caption
		COLOR_SELECTION,				//< The color used for selected objects in wireframe mode
		COLOR_ACTIVE_AXIS,				//< The color used to indicate the active axis in the axis tripod
		COLOR_INACTIVE_AXIS,			//< The color used to indicate the inactive axes in the axis tripod
		COLOR_VIEWPORT_BORDER,			//< The color used to draw the border of inactive viewports
		COLOR_ACTIVE_VIEWPORT_BORDER,	//< The color used to draw the border of the active viewport
		COLOR_SNAPPING_MARKER,			//< The color used to render snapping markers
		COLOR_ANIMATION_MODE,			//< The color used to indicate that the animation mode is active
		COLOR_RENDER_FRAME,				//< The color used to draw the rendering frame around the viewport
		COLOR_CAMERAS,					//< The color used to render camera dummies in the viewports
		COLOR_LIGHTS,					//< The color used to render light dummies in the viewports

		NUMBER_OF_COLORS
	};

	/// View types.
	enum ViewportType {
		VIEW_NONE,
		VIEW_TOP,
		VIEW_BOTTOM,
		VIEW_FRONT,
		VIEW_BACK,
		VIEW_LEFT,
		VIEW_RIGHT,
		VIEW_ORTHO,
		VIEW_PERSPECTIVE,
		VIEW_SCENENODE,
	};

	/// The shading modes for viewports.
	enum ViewportShadingMode {
		SHADING_WIREFRAME,
		SHADING_SHADED,
		SHADING_SHADED_WITH_EDGES,
	};

public:

	/// \brief Constructs a new viewport in the given container widget.
	/// \param container The parent for the new viewportwindow. It will be embedded into the given container.
	Viewport(Window3DContainer* container);

	/// \brief Destructor.
	virtual ~Viewport();

    /// \brief Enqueues the viewport for an update.
    /// \param forceCompleteUpdate If set to \c true the viewport will update several internal caches.
    ///                            This should be done when the viewing parameters have been changed for example.
    ///                            If set to \c false the system will try to redraw only those contents of the viewport
    ///                            have changed.
    ///
	/// Calling this method will redraw the viewport contents unless the viewport is hidden.
	/// This function does not cause an immediate repaint; instead it schedules a
	/// paint event for processing when Qt returns to the main event loop. You can call this method as often
	/// as you want; it will return immediately and will cause only one viewport repaint when Qt returns to the
	/// main event loop.
	///
	/// To update all viewports at once you should use ViewportManager::updateViewports().
	///
    /// \note This method should be used instead of the Window3D::update() method to redraw the viewport contents.
	/// \sa ViewportManager::updateViewports()
	void updateViewport(bool forceCompleteUpdate = false) {
		if(forceCompleteUpdate) _completeUpdate = true;
		update();
	}

	/// \brief Returns the current caption of the viewport.
	/// \return The label of the viewport.
	const QString& caption() const { return _caption; }

	/// \brief Returns the field of view value of the viewport.
	/// \return Horizontal camera angle in radians if the viewport uses a perspective projection or
	///         the field of view in the horizontal direction in world units if the viewport
	///         uses an orthogonal projection.
	/// \sa viewType(), isPerspectiveProjection()
	FloatType fieldOfView() const { return currentView().fieldOfView; }

	/// \brief Sets the zoom of the viewport.
	/// \param fov Horizontal camera angle in radians if the viewport uses a perspective projection or
	///            the field of view in the horizontal direction in world units if the viewport
	///            uses an orthogonal projection.
	void setFieldOfView(FloatType fov);

	/// \brief Changes the viewport's field of view so that the given bounding box is completely in view.
	/// \param box The bounding box in world space that should be made visible.
	/// \note This method does nothing if the viewType() is set to ViewportType::VIEW_SCENENODE.
	/// \sa zoomToExtents()
	void zoomBoundingBox(const Box3& box);

	/// \brief Changes the viewport's field of view so that the whole scene or the selection is completely in view.
	/// \param mode Controls whether the whole scene or only the selected object should be made visible.
	/// \sa zoomBoundingBox()
	void zoomToExtents(SceneRenderer::SceneExtentsMode mode = SceneRenderer::ALL);

	/// Sets the current view matrix. It transforms from world space to view space.
	/// This is overriden from Window3D class.
	void setViewMatrix(const AffineTransformation& tm);

	/// \brief Returns whether this viewport uses a perspective projection.
	/// \return \c true if the current viewType() is a perspective projection; \c false if it is orthogonal.
	bool isPerspectiveProjection() const { return currentView().isPerspective; }

	/// \brief Returns a description the viewport's view at the given animation time.
	/// \param time The animation time for which the view description is requested.
	/// \param aspectRatio Specifies the desired aspect ratio (height/width) of the output image.
	/// \param sceneBoundingBox The bounding box of the scene in world coordinates. This is used to calculate the near and far z-clipping planes.
	/// \return This structure can be used by a PluginRenderer to render the scene as it is currently displayed in the viewport.
	/// \sa currentView()
	CameraViewDescription getViewDescription(TimeTicks time, FloatType aspectRatio, const Box3& sceneBoundingBox = Box3()) const;

	/// \brief Returns a description the viewport's current view.
	/// \return The current viewing parameters used by the viewport.
	const CameraViewDescription& currentView() const { return _cameraView; }

	/// \brief Returns the grid of this viewport.
	/// \return A reference to the internal grid object used by this viewport.
	ViewportGrid& grid() { return _grid; }

	/// \brief Returns the object that stores the settings for this viewport.
	const intrusive_ptr<ViewportRecord>& settings() const { return viewportRecord; }

	/// \brief Displays the context menu for the viewport.
	/// \param pos The position in the window where the context menu should be shown.
	void showViewportMenu(const QPoint& pos = QPoint(0,0));

	/// \brief Computes the size of an object that should appear always in the
	///        same size, independent of the current field of view of the viewport.
	/// \param worldPoint A world point at which the non-scaling object size should be computed.
	/// \return A scaling factor that, if used to render an object at the given world point, will
	///         make the object appear always at the same size.
	///         If the \a worldPoint lies behind the camera then the sclaing factor cannot be computed
	///         and the method returns the constant 1.0.
	///
	/// This method can by used to render markers or gismos in the viewport that should always
	/// keep their default size even if the user zooms into the scene.
	FloatType nonScalingSize(const Point3& worldPoint);

	/// \brief Computes a point in the given coordinate system based on the given screen position and the current snap settings.
	/// \param[in] screenPoint A screen point relative to the upper left corner of the 3d window.
	/// \param[out] snapPoint The resulting snap point in the coordinate system specified by \a snapSystem. If the method returned
	///                       \c false then the value of this output variable is undefined.
	/// \param[in] snapSystem Specifies the coordinate system in which the snapping point should be determined.
	/// \return \c true if a snapping point has been found; \c false if no snapping point was found for the given screen position.
	bool snapPoint(const Point2I& screenPoint, Point3& snapPoint, const AffineTransformation& snapSystem);

	/// \brief Computes a point in the grid coordinate system based on a screen position and the current snap settings.
	/// \param[in] screenPoint A screen point relative to the upper left corner of the 3d window.
	/// \param[out] snapPoint The resulting snap point in the viewport's grid coordinate system. If the method returned
	///                       \c false then the value of this output variable is undefined.
	/// \return \c true if a snapping point has been found; \c false if no snapping point was found for the given screen position.
	bool snapPoint(const Point2I& screenPoint, Point3& snapPoint) {
		return this->snapPoint(screenPoint, snapPoint, grid().gridMatrix());
	}

	/// \brief Computes a ray in world space going through a window pixel.
	/// \param screenPoint A screen point relative to the upper left corner of the 3d window.
	/// \return The ray that goes from the camera point through the specified pixel of the 3d window.
	/// \sa viewportRay()
	Ray3 screenRay(const Point2I& screenPoint);

	/// \brief Computes a ray in world space going through a viewport pixel.
	/// \param viewportPoint A viewport point in the range [-1,+1].
	/// \return The ray that goes from the camera point through the specified pixel of the viewport.
	/// \sa screenRay()
	Ray3 viewportRay(const Point2& viewportPoint);

	/// \brief Computes the projection of a world point.
	/// \param[in] worldPoint The world point to project.
	/// \param[out] screenPoint Contains the projected point after the method returns with \c true.
	///                         The point coordinates are relative to upper left corner of the 3d window.
	/// \return \c true if the world point is visible in the viewport or \c false if the point was clipped away
	///         because it is outside the viewing frustum.
	///
	/// If the point was clipped, (0,0) is returned as screen point.
	bool projectWorldPoint(const Point3& worldPoint, Point2& screenPoint);

	/// \brief Computes the projection of a point given in object space.
	/// \param[in] objPoint The object point to project.
	/// \param[out] screenPoint Contains the projected point after the method returns with \c true.
	///                         The point coordinates are relative to upper left corner of the 3d window.
	/// \return \c true if the object point is visible in the viewport or \c false if the point was clipped away
	///         because it is outside the viewing frustum.
	///
	/// If the point was clipped, (0,0) is returned as screen point.
	bool projectObjectPoint(const Point3& objPoint, Point2& screenPoint) {
		return projectWorldPoint(worldMatrix() * objPoint, screenPoint);
	}

	/// \brief Renders a triangle mesh in the viewport using the current shading context
	///        and the material of the given scene node.
	/// \param mesh The mesh to be rendered.
	/// \param contextNode The scene node which is used as source for the material and shading settings.
	void renderNodeMesh(const TriMesh& mesh, ObjectNode* contextNode);

	/// \brief Renders a bezier shape in the viewport.
	/// \param shape The shape to be rendered.
	/// \param contextNode The scene node which is used as source for the color settings.
	void renderNodeBezierShape(const BezierShape& shape, ObjectNode* contextNode);

	/// \brief Returns a color value for drawing something in the viewport.
	/// \param which The enum constant that specifies what to draw.
	/// \return The color that should be used for the given entity.
	static Color getVPColor(ViewportColor which);

	/// \brief Sets the color for drawing something in the viewport.
	/// \param which The enum constant that specifies what to draw.
	/// \param color The color that should be used for the given entity.
	static void setVPColor(ViewportColor which, const Color& color);

protected:

	/// Render the 3d contents of the window.
	void renderWindow();

	/// Renders the scene as seen through the viewport.
	void renderViewportScene();

	/// Computes the current projection matrix.
	void updateProjectionMatrix();

	/// Updates the title string of the viewport based on the
	/// current view type of the viewport.
	void updateViewportTitle();

	/// Renders the axis tripod symbol in the corner of the viewport that shows
	/// the view orientation.
	void renderViewOrientationIndicator();

	/// Handles mouse down events for the window.
	virtual void mousePressEvent(QMouseEvent* event);
	/// Handles mouse up events for the window.
	virtual void mouseReleaseEvent(QMouseEvent* event);
	/// Handles mouse move events for the window.
	virtual void mouseMoveEvent(QMouseEvent* event);
	/// Handles mouse wheel events for the window.
	virtual void wheelEvent(QWheelEvent* event);
	/// Handles mouse enter events for the window.
	virtual void enterEvent(QEvent* event);
	/// Handles mouse leave events for the window.
	virtual void leaveEvent(QEvent* event);
	/// Handles resize events for the window.
	virtual void resizeEvent(QResizeEvent* event);

private:

	/// View type description.
	QString _caption;

	/// Indicates that the complete scene should be rendered during the next update.
	bool _completeUpdate;

	/// The current view projection.
	CameraViewDescription _cameraView;

	/// The construction grid displayed in the viewport.
	ViewportGrid _grid;

	/// The zone in the upper left corner of the viewport where
	/// the context emnu can be actived by the user.
	QRect contextMenuArea;

	/// This object holds most of the properties and settings of this viewport.
	intrusive_ptr<ViewportRecord> viewportRecord;

	/// Default ambient light used by the viewport.
	Window3DLight ambientLight;

	// First default light used by the viewport.
	Window3DLight defaultLight1;

	// Second default light used by the viewport.
	Window3DLight defaultLight2;

private:

	/// The default colors for viewport drawing.
	static Color viewportColors[NUMBER_OF_COLORS];

	friend class ViewportRecord;
};

};

#endif // __OVITO_VIEWPORT_H
