///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/viewport/snapping/SnappingProvider.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(SnappingProvider, PluginClass)


/******************************************************************************
* Renders the marker of the hit point found.
******************************************************************************/
void SnappingRecord::render(Viewport* vp) const
{
	CHECK_POINTER(vp);
	if(isValid()) {
		// Turn off z-buffer.
		vp->setDepthTest(false);
		vp->setLightingEnabled(false);
		// Render marker.
		marker->render(*vp, worldPoint);
	}
}

/******************************************************************************
* Records a new hit during the snapping process.
******************************************************************************/
void SnappingContext::recordHit(const Point3& worldPoint, const shared_ptr<SnappingMarker>& marker)
{
	CHECK_POINTER(viewport());

	// Project world point to screen.
	Point2 screenPoint;
	if(!viewport()->projectWorldPoint(worldPoint, screenPoint))
		return;

	int sqdist = square((int)screenPoint.X - mousePos().X) + square((int)screenPoint.Y - mousePos().Y);

	// Is in snapping range?
	if(sqdist > _squaredSnappingStrength)
		return;

	// Compare distance.
	if(_bestHit.isValid() && _bestHit.squaredDistance < sqdist) 
		return;

	_bestHit.worldPoint = worldPoint;
	_bestHit.squaredDistance = sqdist;
	_bestHit.marker = marker;
	OVITO_ASSERT(_bestHit.isValid());
}

};
