///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/gui/properties/IntegerControllerUI.h>
#include <core/gui/properties/BooleanControllerUI.h>
#include <core/gui/properties/BooleanGroupBoxControllerUI.h>
#include "POVRayRenderer.h"
#include "POVRayRendererEditor.h"

namespace POVRay {

IMPLEMENT_PLUGIN_CLASS(POVRayRendererEditor, PropertiesEditor)

/******************************************************************************
* Creates the UI controls for the editor.
******************************************************************************/
void POVRayRendererEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("POV-Ray settings"), rolloutParams, "core.rendering.povray_renderer");

	QVBoxLayout* mainLayout = new QVBoxLayout(rollout);
	mainLayout->setContentsMargins(4,4,4,4);

	QGroupBox* generalGroupBox = new QGroupBox(tr("Quality settings"));
	mainLayout->addWidget(generalGroupBox);

	QGridLayout* layout = new QGridLayout(generalGroupBox);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Quality level
	IntegerControllerUI* qualityLevelUI = new IntegerControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _qualityLevel));
	layout->addWidget(qualityLevelUI->label(), 0, 0);
	layout->addWidget(qualityLevelUI->textBox(), 0, 1);
	layout->addWidget(qualityLevelUI->spinner(), 0, 2);
	qualityLevelUI->setMinValue(0);
	qualityLevelUI->setMaxValue(11);

	// Antialiasing
	BooleanGroupBoxControllerUI* enableAntialiasingUI = new BooleanGroupBoxControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _enableAntialiasing));
	QGroupBox* aaGroupBox = enableAntialiasingUI->groupBox();
	mainLayout->addWidget(aaGroupBox);

	layout = new QGridLayout(aaGroupBox);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Sampling Method
	IntegerControllerUI* samplingMethodUI = new IntegerControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _samplingMethod));
	layout->addWidget(samplingMethodUI->label(), 1, 0);
	layout->addWidget(samplingMethodUI->textBox(), 1, 1);
	layout->addWidget(samplingMethodUI->spinner(), 1, 2);
	samplingMethodUI->setMinValue(1);
	samplingMethodUI->setMaxValue(2);
	connect(aaGroupBox, SIGNAL(toggled(bool)), samplingMethodUI, SLOT(setEnabled(bool)));

	// AA Threshold
	FloatControllerUI* AAThresholdUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _AAThreshold));
	layout->addWidget(AAThresholdUI->label(), 2, 0);
	layout->addWidget(AAThresholdUI->textBox(), 2, 1);
	layout->addWidget(AAThresholdUI->spinner(), 2, 2);
	AAThresholdUI->setMinValue(0);
	AAThresholdUI->setMaxValue(1);
	connect(aaGroupBox, SIGNAL(toggled(bool)), AAThresholdUI, SLOT(setEnabled(bool)));

	// AA Depth
	IntegerControllerUI* AADepthUI = new IntegerControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _antialiasDepth));
	layout->addWidget(AADepthUI->label(), 3, 0);
	layout->addWidget(AADepthUI->textBox(), 3, 1);
	layout->addWidget(AADepthUI->spinner(), 3, 2);
	AADepthUI->setMinValue(1);
	AADepthUI->setMaxValue(9);
	connect(aaGroupBox, SIGNAL(toggled(bool)), AADepthUI, SLOT(setEnabled(bool)));

	// Jitter
	BooleanControllerUI* enableJitterUI = new BooleanControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _enableJitter));
	layout->addWidget(enableJitterUI->checkBox(), 4, 0, 1, 3);
	connect(aaGroupBox, SIGNAL(toggled(bool)), enableJitterUI, SLOT(setEnabled(bool)));

	// Radiosity
	BooleanGroupBoxControllerUI* enableRadiosityUI = new BooleanGroupBoxControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _enableRadiosity));
	QGroupBox* radiosityGroupBox = enableRadiosityUI->groupBox();
	mainLayout->addWidget(radiosityGroupBox);

	layout = new QGridLayout(radiosityGroupBox);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// Ray count
	IntegerControllerUI* radiosityRayCountUI = new IntegerControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _radiosityRayCount));
	layout->addWidget(radiosityRayCountUI->label(), 0, 0);
	layout->addWidget(radiosityRayCountUI->textBox(), 0, 1);
	layout->addWidget(radiosityRayCountUI->spinner(), 0, 2);
	radiosityRayCountUI->setMinValue(1);
	radiosityRayCountUI->setMaxValue(1600);
	connect(radiosityGroupBox, SIGNAL(toggled(bool)), radiosityRayCountUI, SLOT(setEnabled(bool)));

	// Recursion limit
	IntegerControllerUI* radiosityRecursionLimitUI = new IntegerControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _radiosityRecursionLimit));
	layout->addWidget(radiosityRecursionLimitUI->label(), 1, 0);
	layout->addWidget(radiosityRecursionLimitUI->textBox(), 1, 1);
	layout->addWidget(radiosityRecursionLimitUI->spinner(), 1, 2);
	radiosityRecursionLimitUI->setMinValue(1);
	radiosityRecursionLimitUI->setMaxValue(20);
	connect(radiosityGroupBox, SIGNAL(toggled(bool)), radiosityRecursionLimitUI, SLOT(setEnabled(bool)));

	// Error bound
	FloatControllerUI* radiosityErrorBoundUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _radiosityErrorBound));
	layout->addWidget(radiosityErrorBoundUI->label(), 2, 0);
	layout->addWidget(radiosityErrorBoundUI->textBox(), 2, 1);
	layout->addWidget(radiosityErrorBoundUI->spinner(), 2, 2);
	radiosityErrorBoundUI->setMinValue(1e-5f);
	radiosityErrorBoundUI->setMaxValue(100.0f);
	connect(radiosityGroupBox, SIGNAL(toggled(bool)), radiosityErrorBoundUI, SLOT(setEnabled(bool)));

	// Preferences group
	QGroupBox* preferencesGroupBox = new QGroupBox(tr("Preferences"));
	mainLayout->addWidget(preferencesGroupBox);
	layout = new QGridLayout(preferencesGroupBox);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(0, 1);

	// POV-Ray executable path
	layout->addWidget(new QLabel(tr("POV-Ray executable:")), 0, 0, 1, 2);
	executablePathBox = new QLineEdit(POVRayRenderer::renderExecutable());
	connect(executablePathBox, SIGNAL(editingFinished()), this, SLOT(onExecutablePathEdited()));
	layout->addWidget(executablePathBox, 1, 0);
	QPushButton* selectExecutablePathButton = new QPushButton("...");
	connect(selectExecutablePathButton, SIGNAL(clicked(bool)), this, SLOT(onExecutablePathChoose()));
	layout->addWidget(selectExecutablePathButton, 1, 1);

	// Hide POV-Ray window
	BooleanControllerUI* hidePOVRayUI = new BooleanControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(POVRayRenderer, _hidePOVRayDisplay));
	layout->addWidget(hidePOVRayUI->checkBox(), 2, 0, 1, 2);
}

/******************************************************************************
* Is called when the user has entered a new POV-Ray executable path.
******************************************************************************/
void POVRayRendererEditor::onExecutablePathEdited()
{
	POVRayRenderer::setRenderExecutable(executablePathBox->text());
	executablePathBox->setText(POVRayRenderer::renderExecutable());
}

/******************************************************************************
* Is called when the user has pressed the button for the POV-Ray executable path.
******************************************************************************/
void POVRayRendererEditor::onExecutablePathChoose()
{
	QString path = QFileDialog::getOpenFileName(container(), tr("Select POV-Ray Executable"),
		 executablePathBox->text());

	if(!path.isEmpty()) {
		POVRayRenderer::setRenderExecutable(path);
		executablePathBox->setText(POVRayRenderer::renderExecutable());
	}
}

};
