///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/scene/ObjectNode.h>
#include <core/gui/properties/FloatControllerUI.h>
#include <core/rendering/RenderSettings.h>
#include "CameraObject.h"
#include "FreeCameraCreationMode.h"
#include "TargetCameraCreationMode.h"

namespace StdObjects {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CameraObject, AbstractCameraObject)
DEFINE_REFERENCE_FIELD(CameraObject, FloatController, "FOV", _fov)
SET_PROPERTY_FIELD_LABEL(CameraObject, _fov, "Field of View")
SET_PROPERTY_FIELD_UNITS(CameraObject, _fov, AngleParameterUnit)

IMPLEMENT_PLUGIN_CLASS(CameraObjectEditor, PropertiesEditor)
IMPLEMENT_PLUGIN_CLASS(FreeCameraCreationMode, SimpleCreationMode)
IMPLEMENT_PLUGIN_CLASS(TargetCameraCreationMode, CreationMode)

/******************************************************************************
* Constructs a camera object.
******************************************************************************/
CameraObject::CameraObject(bool isLoading)
	: AbstractCameraObject(isLoading), meshValidity(TimeNever)
{
	INIT_PROPERTY_FIELD(CameraObject, _fov);
	if(!isLoading) {
		_fov = CONTROLLER_MANAGER.createDefaultController<FloatController>();
		_fov->setValue(0, FLOATTYPE_PI/3.0);
	}
}

/******************************************************************************
* Asks the object for its validity interval at the given time.
******************************************************************************/
TimeInterval CameraObject::objectValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	_fov->validityInterval(time, interval);
	return interval;
}

/******************************************************************************
* Fills in the missing fields of the camera view descriptor structure.
******************************************************************************/
void CameraObject::getCameraDescription(TimeTicks time, CameraViewDescription& descriptor)
{
	descriptor.isPerspective = true;
	// Get the camera angle value.
	_fov->getValue(time, descriptor.fieldOfView, descriptor.validityInterval);
	if(descriptor.fieldOfView < 0.01f) descriptor.fieldOfView = 0.01f;
	if(descriptor.fieldOfView > FLOATTYPE_PI-0.01f) descriptor.fieldOfView = FLOATTYPE_PI-0.01f;

	// Choose valid clipping values.
	descriptor.znear = max(descriptor.znear, (FloatType)1e-6);
	descriptor.zfar = max(descriptor.zfar, descriptor.znear * (FloatType)1.0001);
	descriptor.znear = max(descriptor.znear, descriptor.zfar * (FloatType)1e-8);

	// Setup a perspective projection matrix.
	descriptor.projectionMatrix = Matrix4::perspective(descriptor.fieldOfView, 1.0/descriptor.aspectRatio, descriptor.znear, descriptor.zfar);
	descriptor.inverseProjectionMatrix = descriptor.projectionMatrix.inverse();
}

/******************************************************************************
* Makes the object render itself into the viewport.
* The viewport transformation is already set up, when this method is called by the
* system. The object has to be rendered in the local object coordinate system.
******************************************************************************/
void CameraObject::renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp)
{
	if(!meshValidity.contains(time))
		buildMesh(time);

	// Adjust icon size.
	AffineTransformation worldMat = vp->worldMatrix();
	Vector3 cameraPos = worldMat.getTranslation();
	FloatType scaling = vp->nonScalingSize(ORIGIN + cameraPos) * 2.0f;
	vp->setWorldMatrix(worldMat * AffineTransformation::scaling(scaling));

    vp->setLightingEnabled(false);
    vp->setBackfaceCulling(true);

	// Setup rendering color.
    bool isCameraSelected = contextNode->isSelected();
	vp->setRenderingColor(Viewport::getVPColor(isCameraSelected ? Viewport::COLOR_SELECTION : Viewport::COLOR_CAMERAS));

	// Render icon
	vp->renderMeshWireframe(mesh);

	if(contextNode->targetNode() && !vp->isPicking()) {
		TimeInterval iv = TimeForever;
        const AffineTransformation& targetTM = contextNode->targetNode()->getWorldTransform(time, iv);
		const Vector3& targetPos = targetTM.getTranslation();
		bool isSelected = (isCameraSelected || contextNode->targetNode()->isSelected());

		// Render camera-target link
		{
			// Create vertex buffer
			Point3 verts[2] = { ORIGIN + cameraPos, ORIGIN + targetPos };
			Box3 boundingBox;
			boundingBox += verts[0];
			boundingBox += verts[1];

			// Render link line.
			vp->setWorldMatrix(IDENTITY);
			vp->setRenderingColor(Viewport::getVPColor(isSelected ? Viewport::COLOR_SELECTION : Viewport::COLOR_CAMERAS));
			vp->renderLines(2, boundingBox, verts);
		}

		// Render camera frustum.
		RenderSettings* settings = DATASET_MANAGER.currentSet()->renderSettings();
		if(settings && isSelected) {
			// Create vertex buffer
			FloatType aspectRatio = (FloatType)settings->imageHeight() / (FloatType)settings->imageWidth();
			FloatType fov = fovController()->getValueAtTime(time);
			if(fov < 0.01f) fov = 0.01f;
			if(fov > FLOATTYPE_PI-0.01f) fov = FLOATTYPE_PI-0.01f;
			FloatType targetDist = Length(targetPos - cameraPos);
			FloatType fieldY = targetDist * tan(fov * 0.5);
			FloatType fieldX = fieldY / aspectRatio;
			Point3 verts[8*2];
			verts[0] = verts[2] = verts[4] = verts[6] = ORIGIN;
			verts[1] = verts[8] = verts[15] = Point3(fieldX, fieldY, -targetDist);
			verts[3] = verts[9] = verts[10] = Point3(-fieldX, fieldY, -targetDist);
			verts[5] = verts[11] = verts[12] = Point3(-fieldX, -fieldY, -targetDist);
			verts[7] = verts[13] = verts[14] = Point3(fieldX, -fieldY, -targetDist);
			Box3 boundingBox;
			for(size_t i=0; i<16; i++)
				boundingBox += verts[i];

			// Render link line.
			vp->setWorldMatrix(worldMat);
			vp->setRenderingColor(Viewport::getVPColor(Viewport::COLOR_CAMERAS));
			vp->renderLines(16, boundingBox, verts);
		}
	}

	// Restore old matrix
	vp->setWorldMatrix(worldMat);
}

/******************************************************************************
* Returns the bounding box of the object in local object coordinates.
******************************************************************************/
Box3 CameraObject::boundingBox(TimeTicks time, ObjectNode* contextNode)
{
	if(!meshValidity.contains(time))
		buildMesh(time);

	// Adjust icon size
	FloatType scaling = 1;
    Viewport* vp = VIEWPORT_MANAGER.activeViewport();
	if(vp) {
		TimeInterval iv = TimeForever;
		const AffineTransformation& worldMat = contextNode->getWorldTransform(time, iv);
		Vector3 cameraPos = worldMat.getTranslation();
		scaling = vp->nonScalingSize(ORIGIN + cameraPos) * 2.0f;
	}

    return mesh.boundingBox().centerScale(scaling);
}

/******************************************************************************
* Builds the icon mesh for the representation of this camera object in the viewports.
******************************************************************************/
void CameraObject::buildMesh(TimeTicks time)
{
	// Reset mesh
	meshValidity.setInfinite();

	// Query parameters
	FloatType fov;
	_fov->getValue(time, fov, meshValidity);

	// Build vertices.
	mesh.setVertexCount(16);
	mesh.setVertex(0, Point3(-0.15f, -0.15f, 0.3f));
	mesh.setVertex(1, Point3( 0.15f, -0.15f, 0.3f));
	mesh.setVertex(2, Point3( 0.15f,  0.15f, 0.3f));
	mesh.setVertex(3, Point3(-0.15f,  0.15f, 0.3f));
	mesh.setVertex(4, Point3(-0.15f, -0.15f, -0.2f));
	mesh.setVertex(5, Point3( 0.15f, -0.15f, -0.2f));
	mesh.setVertex(6, Point3( 0.15f,  0.15f, -0.2f));
	mesh.setVertex(7, Point3(-0.15f,  0.15f, -0.2f));
    mesh.setVertex(8, Point3(-0.02f, -0.02f, -0.2f));
    mesh.setVertex(9, Point3( 0.02f, -0.02f, -0.2f));
    mesh.setVertex(10, Point3( 0.02f,  0.02f, -0.2f));
    mesh.setVertex(11, Point3(-0.02f,  0.02f, -0.2f));
    mesh.setVertex(12, Point3(-0.10f, -0.10f, -0.5f));
    mesh.setVertex(13, Point3( 0.10f, -0.10f, -0.5f));
    mesh.setVertex(14, Point3( 0.10f,  0.10f, -0.5f));
    mesh.setVertex(15, Point3(-0.10f,  0.10f, -0.5f));

	// Build faces.
	mesh.setFaceCount(24);
	mesh.face(0).setVertices(0, 1, 5);
	mesh.face(0).setEdgeVisibility(true, true, false);
	mesh.face(1).setVertices(0, 5, 4);
	mesh.face(1).setEdgeVisibility(false, true, true);
	mesh.face(2).setVertices(1, 2, 6);
	mesh.face(2).setEdgeVisibility(true, true, false);
	mesh.face(3).setVertices(1, 6, 5);
	mesh.face(3).setEdgeVisibility(false, true, true);
	mesh.face(4).setVertices(2, 3, 7);
	mesh.face(4).setEdgeVisibility(true, true, false);
	mesh.face(5).setVertices(2, 7, 6);
	mesh.face(5).setEdgeVisibility(false, true, true);
	mesh.face(6).setVertices(3, 0, 4);
	mesh.face(6).setEdgeVisibility(true, true, false);
	mesh.face(7).setVertices(3, 4, 7);
	mesh.face(7).setEdgeVisibility(false, true, true);
	mesh.face(8).setVertices(4, 5, 6);
	mesh.face(8).setEdgeVisibility(true, true, false);
	mesh.face(9).setVertices(4, 6, 7);
	mesh.face(9).setEdgeVisibility(false, true, true);
	mesh.face(10).setVertices(0, 3, 2);
	mesh.face(10).setEdgeVisibility(true, true, false);
	mesh.face(11).setVertices(0, 2, 1);
	mesh.face(11).setEdgeVisibility(false, true, true);

	mesh.face(12).setVertices(8, 9, 13);
	mesh.face(12).setEdgeVisibility(true, true, false);
	mesh.face(13).setVertices(8, 13, 12);
	mesh.face(13).setEdgeVisibility(false, true, true);
	mesh.face(14).setVertices(9, 10, 14);
	mesh.face(14).setEdgeVisibility(true, true, false);
	mesh.face(15).setVertices(9, 14, 13);
	mesh.face(15).setEdgeVisibility(false, true, true);
	mesh.face(16).setVertices(10, 11, 15);
	mesh.face(16).setEdgeVisibility(true, true, false);
	mesh.face(17).setVertices(10, 15, 14);
	mesh.face(17).setEdgeVisibility(false, true, true);
	mesh.face(18).setVertices(11, 8, 12);
	mesh.face(18).setEdgeVisibility(true, true, false);
	mesh.face(19).setVertices(11, 12, 15);
	mesh.face(19).setEdgeVisibility(false, true, true);
	mesh.face(20).setVertices(12, 13, 14);
	mesh.face(20).setEdgeVisibility(true, true, false);
	mesh.face(21).setVertices(12, 14, 15);
	mesh.face(21).setEdgeVisibility(false, true, true);
	mesh.face(22).setVertices(8, 11, 10);
	mesh.face(22).setEdgeVisibility(true, true, false);
	mesh.face(23).setVertices(8, 10, 9);
	mesh.face(23).setEdgeVisibility(false, true, true);

	mesh.invalidateVertices();
	mesh.invalidateFaces();
}

/******************************************************************************
* Constructor that creates the UI controls for the editor.
******************************************************************************/
void CameraObjectEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create the rollout.
	QWidget* rollout = createRollout(tr("Camera"), rolloutParams);

	QGridLayout* layout = new QGridLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
	layout->setSpacing(0);
	layout->setColumnStretch(1, 1);

	// FOV parameter.
	FloatControllerUI* fovPUI = new FloatControllerUI(this, PROPERTY_FIELD_DESCRIPTOR(CameraObject, _fov));
	layout->addWidget(fovPUI->label(), 0, 0);
	layout->addWidget(fovPUI->textBox(), 0, 1);
	layout->addWidget(fovPUI->spinner(), 0, 2);
	fovPUI->setMinValue(fovPUI->parameterUnit()->userToNative(0.01f));
	fovPUI->setMaxValue(fovPUI->parameterUnit()->userToNative(179.99f));
}

};
