/*
 * otr.i
 * This file is part of python-otr
 *
 * Copyright (C) 2008 - Kjell Braden <fnord@pentabarf.de>
 *
 * python-otr is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * python-otr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with python-otr; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA  02110-1301  USA
 */
 
 

%module(docstring="This module contains python bindings for libotr (see http://www.cypherpunks.ca/otr/ for more information).
Some libotr functions have been stripped, as they were not useful for writing a frontend to OTR.
For further details on how to use this module, please refer to the documentation.") otr;

%feature("autodoc", "1");


%init %{
	otrl_init(OTRL_VERSION_MAJOR, OTRL_VERSION_MINOR, OTRL_VERSION_SUB);
%}


%{
#include <libotr/context.h>
#include <libotr/privkey-t.h>
#include <libotr/privkey.h>
#include <gcrypt.h>
#include <gpg-error.h>
#include <libotr/proto.h>
#include <libotr/sm.h>
#include <libotr/tlv.h>
#include <libotr/userstate.h>
#include <libotr/version.h>
#include <libotr/message.h>

%}


/* for otrl_context_find{,_fingerprint} */
/* XXX could use %apply int *OUTPUT { int *addedp }; here, if swig would
 * generate tuples/lists with always the same size (it would currently return
 * 0 instead of [None, 0] ) */
%typemap(in,numinputs=0) int *addedp (int temp){
	$1 = &temp;
}
%typemap(argout) int *addedp {
	if (!$result) 
		$result = PyTuple_Pack(2, Py_None, SWIG_From_int(*$1));
	else {
		PyObject * result_copy = $result;
		$result = PyTuple_Pack(2, result_copy, SWIG_From_int(*$1));
		Py_DECREF(result_copy);
	}
}

/* MessageAppOps typemaps */
%typemap(in, numinputs=1) (const OtrlMessageAppOps *ops, void *opdata) {
	$1 = ($1_ltype) malloc(sizeof($*1_ltype));
	ops_new($1);

	if ( !PySequence_Check($input) || PySequence_Length($input) != 2)
	{
		PyErr_Format(PyExc_ValueError, "Expected a tuple in the form (MessageAppOps class, opdata) as argument %d", $argnum);
		SWIG_fail;
	}

	Py_INCREF($input);
	$2 = $input;
}
%typemap(freearg) (const OtrlMessageAppOps *ops, void *opdata) {
	if ($1 != 0) {
		free($1);
	}
	if ($2 != 0) {
		Py_XDECREF((PyObject *) $2);
	}
}

/* for otrl_message_sending and otrl_message_fragment_and_send*/
%typemap(in, numinputs=0) char **messagep ($*1_ltype temp = NULL), char **returnFragment ($*1_ltype temp = NULL) {
	$1 = &temp;
}
/* the tlvs argument is ignored, it's because swig needs the given 
 * parameters in groups */
%typemap(argout) (const char *message, OtrlTLV *tlvs, char **messagep) {
	if (*$3 == NULL) {
		$result = PyString_FromString((const char *) $1);
	}
	else {
		$result = PyString_FromString((const char *) *$3);
		otrl_message_free(*$3);
	}
}
%typemap(argout) char **returnFragment {
	if (*$1 != NULL) {
		$result = PyString_FromString(*$1);
		otrl_message_free(*$1);
	}
}


/* add_appdata typemaps */
%typemap(default, numinputs=1) (void (*add_appdata)(void *data, ConnContext *context), void *data), (void (*add_app_data)(void *data, ConnContext *context), void *data) {
	$2 = PyTuple_New(2);
}
%typemap(in, numinputs=1) (void (*add_appdata)(void *data, ConnContext *context), void *data), (void (*add_app_data)(void *data, ConnContext *context), void *data) {
	$1 = wrap_add_appdata;
	
	if ( !PySequence_Check($input) || PySequence_Length($input) != 2)
	{
		PyErr_Format(PyExc_ValueError, "Expected a tuple in the form (add_appdata function, opdata) as argument %d", $argnum);
		SWIG_fail;
	}

	Py_INCREF($input);
	$2 = $input;
}
%typemap(freearg) (void (*add_appdata)(void *data, ConnContext *context), void *data), (void (*add_app_data)(void *data, ConnContext *context), void *data) {
	Py_XDECREF((PyObject *) $2);
}

/* error handling typemaps */
%typemap(out) gcry_error_t {
	if ($1 != 0) {
		PyObject * exc = PyErr_NewException("otr.GCryError", NULL, NULL);
		gcry_err_code_t err_code = gcry_err_code($1);
		PyObject_SetAttrString(exc, "errno", PyInt_FromLong(err_code));
		PyObject_SetAttrString(exc, "os_errno", PyLong_FromUnsignedLong(
			 gpg_err_code_to_errno(err_code))); // we use the gpg_err call here
						// because gcry_err_code_to_errno returns wrong numbers
		PyObject_SetAttrString(exc, "strerror", PyString_FromString(gcry_strerror($1)));

		PyErr_Format(exc, "[gcry_errno %u] [os_errno %d] %s", err_code,
			gpg_err_code_to_errno(err_code), gcry_strerror($1));
		Py_DECREF(exc);
		$result = NULL;
		SWIG_fail;
	}
	else
		$result = SWIG_Py_Void();
}


/* for otrl_privkey_fingerprint, otrl_privkey_hash_to_human and otrl_privkey_fingerprint_raw*/
%typemap(in, numinputs=0) char MYOUT[ANY], char NOTNULLTERMINATED[ANY] {
	$1 = ($1_ltype) malloc($1_dim0);
}
%typemap(argout) char NOTNULLTERMINATED[ANY] {
	$result = SWIG_FromCharPtrAndSize($1, $1_dim0);
}
%typemap(argout) char MYOUT[ANY] {
	$result = SWIG_FromCharPtrAndSize($1, $1_dim0 - 1);
};
%typemap(freearg) char MYOUT[ANY], char NOTNULLTERMINATED[ANY] {
	free($1);
}
%typemap(out) void * otrl_privkey_fingerprint, void * otrl_privkey_fingerprint_raw {
	if ($1 == NULL) { 
		SWIG_SetErrorMsg(PyExc_LookupError, "Account/Protocol pair not found");
		SWIG_fail;
	}
}

%typemap(out) char fingerprint[20] {
	if (!$1) $result = SWIG_Py_Void();
	else $result = SWIG_FromCharPtrAndSize(result, $1_dim0);
}


%typemap(out) PyObject *app_data {
	if (!$1) $result = Py_None;
	else $result = $1;
	Py_INCREF($result);
}

%apply (char *STRING, int LENGTH) { (const unsigned char *secret, size_t secretlen) };



%{
#define CHECK_ERROR check_and_abort(name, __FILE__, __LINE__)
void check_and_abort(const char *name, const char * f, int l) {
	if (PyErr_Occurred() != NULL) {
		fprintf(stderr, "\n\n================\n%s:%d FATAL: Exception in callback \"%s\"\n================\n\nTraceback:\n", f, l, name);
		PyErr_Print();
		fprintf(stderr,"\n================\n\n");
		abort();
	}
}
void prepare_python_callback(void *opdata, const char * name, PyObject **cb_func, PyObject **kwargs) {
	Py_ssize_t len = PySequence_Size(opdata);
	CHECK_ERROR;

	PyObject *cb_obj = PySequence_GetItem(opdata, 0);
	CHECK_ERROR;

	*cb_func = PyObject_GetAttrString(cb_obj, name);
	CHECK_ERROR;

	PyObject *args_opdata = NULL;
	if (len == 2)
		args_opdata = PySequence_GetItem(opdata, 1);
	CHECK_ERROR;

	*kwargs = PyDict_New();
	PyDict_SetItemString(*kwargs, "opdata", args_opdata);
	CHECK_ERROR;
}

OtrlPolicy wrap_ops_policy(void *opdata, ConnContext *context) {
	char * name = "policy";

	PyObject *cb, *kwargs;

	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = SWIG_NewPointerObj(SWIG_as_voidptr(context), SWIGTYPE_p_context, 0 |  0 );
	PyDict_SetItemString(kwargs, "context", arg1);
	Py_DECREF(arg1);

	PyObject *args = PyTuple_New(0);

	PyObject *result = PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);

	OtrlPolicy retval = (OtrlPolicy) PyInt_AsUnsignedLongMask(result);
	CHECK_ERROR;

	Py_DECREF(result);

	return retval;
}

void wrap_ops_create_privkey(void *opdata, const char *accountname, const char *protocol) {
	const char * name = "create_privkey";

	PyObject *kwargs, *cb;

	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyString_FromString(accountname);
	PyDict_SetItemString(kwargs, "accountname", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg2);
	Py_DECREF(arg2);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
int wrap_ops_is_logged_in(void *opdata, const char *accountname, const char *protocol, const char *recipient) {
	const char * name = "is_logged_in";

	PyObject *kwargs, *cb;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyString_FromString(accountname);
	PyDict_SetItemString(kwargs, "accountname", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg2);
	Py_DECREF(arg2);

	PyObject *arg3 = PyString_FromString(recipient);
	PyDict_SetItemString(kwargs, "recipient", arg3);
	Py_DECREF(arg3);

	PyObject *args = PyTuple_New(0);

	PyObject *result = PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);

	int retval = (int) PyInt_AsLong(result);
	CHECK_ERROR;

	Py_DECREF(result);

	return retval;
}
void wrap_ops_inject_message(void *opdata, const char *accountname, const char *protocol,
		const char *recipient, const char *message) {
	const char *name ="inject_message";

	PyObject *kwargs, *cb;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyString_FromString(accountname);
	PyDict_SetItemString(kwargs, "accountname", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg2);
	Py_DECREF(arg2);

	PyObject *arg3 = PyString_FromString(recipient);
	PyDict_SetItemString(kwargs, "recipient", arg3);
	Py_DECREF(arg3);

	PyObject *arg4 = PyString_FromString(message);
	PyDict_SetItemString(kwargs, "message", arg4);
	Py_DECREF(arg4);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);

}
void wrap_ops_notify(void *opdata, OtrlNotifyLevel level,
		const char *accountname, const char *protocol,
		const char *username, const char *title,
		const char *primary, const char *secondary) {
	const char *name = "notify";

	PyObject *kwargs, *cb;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyInt_FromLong(level);
	PyDict_SetItemString(kwargs, "level", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyString_FromString(accountname);
	PyDict_SetItemString(kwargs, "accountname", arg2);
	Py_DECREF(arg2);

	PyObject *arg3 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg3);
	Py_DECREF(arg3);

	PyObject *arg4 = PyString_FromString(username);
	PyDict_SetItemString(kwargs, "username", arg4);
	Py_DECREF(arg4);

	PyObject *arg5 = PyString_FromString(title);
	PyDict_SetItemString(kwargs, "title", arg5);
	Py_DECREF(arg5);

	PyObject *arg6 = PyString_FromString(primary);
	PyDict_SetItemString(kwargs, "primary", arg6);
	Py_DECREF(arg6);

	PyObject *arg7 = PyString_FromString(secondary);
	PyDict_SetItemString(kwargs, "secondary", arg7);
	Py_DECREF(arg7);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
int wrap_ops_display_otr_message(void *opdata, const char *accountname,
		const char *protocol, const char *username, const char *msg) {
	const char * name = "display_otr_message";

	PyObject *kwargs, *cb;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyString_FromString(accountname);
	PyDict_SetItemString(kwargs, "accountname", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg2);
	Py_DECREF(arg2);

	PyObject *arg3 = PyString_FromString(username);
	PyDict_SetItemString(kwargs, "username", arg3);
	Py_DECREF(arg3);

	PyObject *arg4 = PyString_FromString(msg);
	PyDict_SetItemString(kwargs, "msg", arg4);
	Py_DECREF(arg4);

	PyObject *args = PyTuple_New(0);

	PyObject *result = PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);

	int retval = (int) PyInt_AsLong(result);
	CHECK_ERROR;

	Py_DECREF(result);

	return retval;
}
void wrap_ops_update_context_list(void *opdata) {
	char * name = "update_context_list";

	PyObject *cb, *kwargs;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
const char * wrap_ops_protocol_name(void *opdata, const char *protocol) {
	const char * name = "protocol_name";

	PyObject *cb, *kwargs;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg1);
	Py_DECREF(arg1);

	PyObject *args = PyTuple_New(0);

	PyObject *result = PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);

	char * ret =  (char*) malloc (PyString_Size(result)+1);
	CHECK_ERROR;
	strcpy(ret, PyString_AsString(result));
	CHECK_ERROR;

	Py_DECREF(result);

	return ret;
}
void wrap_ops_protocol_name_free(void *opdata, const char *protocol_name) {
	free((void *)protocol_name);
}
void wrap_ops_new_fingerprint(void *opdata, OtrlUserState us,
		const char *accountname, const char *protocol,
		const char *username, unsigned char fingerprint[20]) {
	const char * name = "new_fingerprint";

	PyObject *kwargs, *cb;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = SWIG_NewPointerObj(SWIG_as_voidptr(us), SWIGTYPE_p_s_OtrlUserState, 0 |  0 );
	PyDict_SetItemString(kwargs, "userstate", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyString_FromString(accountname);
	PyDict_SetItemString(kwargs, "accountname", arg2);
	Py_DECREF(arg2);

	PyObject *arg3 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg3);
	Py_DECREF(arg3);

	PyObject *arg4 = PyString_FromString(username);
	PyDict_SetItemString(kwargs, "username", arg4);
	Py_DECREF(arg4);

	PyObject *arg5 = PyString_FromStringAndSize((char*) fingerprint, 20); 
	PyDict_SetItemString(kwargs, "fingerprint", arg5);
	Py_DECREF(arg5);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
void wrap_ops_write_fingerprints(void *opdata) {
	const char * name = "write_fingerprints";

	PyObject *cb, *kwargs;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
void wrap_ops_gone_secure(void *opdata, ConnContext *context) {
	const char * name = "gone_secure";

	PyObject *cb, *kwargs;

	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = SWIG_NewPointerObj(SWIG_as_voidptr(context), SWIGTYPE_p_context, 0 |  0 );
	PyDict_SetItemString(kwargs, "context", arg1);
	Py_DECREF(arg1);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
void wrap_ops_gone_insecure(void *opdata, ConnContext *context) {
	const char * name = "gone_insecure";

	PyObject *cb, *kwargs;

	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = SWIG_NewPointerObj(SWIG_as_voidptr(context), SWIGTYPE_p_context, 0 |  0 );
	PyDict_SetItemString(kwargs, "context", arg1);
	Py_DECREF(arg1);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
void wrap_ops_still_secure(void *opdata, ConnContext *context, int is_reply) {
	const char * name = "still_secure";

	PyObject *cb, *kwargs;

	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = SWIG_NewPointerObj(SWIG_as_voidptr(context), SWIGTYPE_p_context, 0 |  0 );
	PyDict_SetItemString(kwargs, "context", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyInt_FromLong(is_reply);
	PyDict_SetItemString(kwargs, "is_reply", arg2);
	Py_DECREF(arg2);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
void wrap_ops_log_message(void *opdata, const char *message) {
	const char * name = "log_message";

	PyObject *cb, *kwargs;

	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyString_FromString(message);
	PyDict_SetItemString(kwargs, "message", arg1);
	Py_DECREF(arg1);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);
}
int wrap_ops_max_message_size(void *opdata, ConnContext *context) {
	const char * name = "max_message_size";

	PyObject *cb, *kwargs;

	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = SWIG_NewPointerObj(SWIG_as_voidptr(context), SWIGTYPE_p_context, 0 |  0 );
	PyDict_SetItemString(kwargs, "context", arg1);
	Py_DECREF(arg1);

	PyObject *args = PyTuple_New(0);

	PyObject *result = PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);

	int retval = (int) PyInt_AsLong(result);
	CHECK_ERROR;

	Py_DECREF(result);

	return retval;
}
const char * wrap_ops_account_name(void *opdata, const char *account,
		const char *protocol) {
	const char * name ="account_name";

	PyObject *cb, *kwargs;
	prepare_python_callback(opdata, name, &cb, &kwargs);

	PyObject *arg1 = PyString_FromString(account);
	PyDict_SetItemString(kwargs, "account", arg1);
	Py_DECREF(arg1);

	PyObject *arg2 = PyString_FromString(protocol);
	PyDict_SetItemString(kwargs, "protocol", arg2);
	Py_DECREF(arg2);

	PyObject *args = PyTuple_New(0);

	PyObject *result = PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(cb);
	Py_DECREF(args);
	Py_DECREF(kwargs);

	char * ret =  (char*) malloc (PyString_Size(result)+1);
	CHECK_ERROR;
	strcpy(ret, PyString_AsString(result));
	CHECK_ERROR;

	Py_DECREF(result);

	return ret;
}
void wrap_ops_account_name_free(void *opdata, const char *account_name) {
	free((void *)account_name);
}
void ops_new(OtrlMessageAppOps *ops) {
	ops->policy = wrap_ops_policy;
	ops->create_privkey = wrap_ops_create_privkey;
	ops->is_logged_in = wrap_ops_is_logged_in;
	ops->inject_message = wrap_ops_inject_message;
	ops->notify = wrap_ops_notify;
	ops->display_otr_message = wrap_ops_display_otr_message;
	ops->update_context_list = wrap_ops_update_context_list;
	ops->protocol_name = wrap_ops_protocol_name;
	ops->protocol_name_free = wrap_ops_protocol_name_free;
	ops->new_fingerprint = wrap_ops_new_fingerprint;
	ops->write_fingerprints = wrap_ops_write_fingerprints;
	ops->gone_secure = wrap_ops_gone_secure;
	ops->gone_insecure = wrap_ops_gone_insecure;
	ops->still_secure = wrap_ops_still_secure;
	ops->log_message = wrap_ops_log_message;
	ops->max_message_size = wrap_ops_max_message_size;
	ops->account_name = wrap_ops_account_name;
	ops->account_name_free = wrap_ops_account_name_free;
}
void wrap_appdata_free(PyObject *data) {
	Py_XDECREF(data);
}
void wrap_add_appdata (void *data, ConnContext *context) {
	const char * name = "add_appdata";

	PyObject *cb, *kwargs;

	Py_ssize_t len = PySequence_Size(data);
	CHECK_ERROR;

	cb = PySequence_GetItem(data, 0);
	CHECK_ERROR;

	if (cb == Py_None) // no callback defined
		return;

	PyObject *args_data = NULL;
	if (len == 2)
		args_data = PySequence_GetItem(data, 1);
	CHECK_ERROR;

	kwargs = PyDict_New();
	PyDict_SetItemString(kwargs, "data", args_data);
	CHECK_ERROR;

	PyObject *arg1 = SWIG_NewPointerObj(SWIG_as_voidptr(context), SWIGTYPE_p_context, 0 |  0 );
	PyDict_SetItemString(kwargs, "context", arg1);
	Py_DECREF(arg1);

	PyObject *args = PyTuple_New(0);

	PyObject_Call(cb, args, kwargs);
	CHECK_ERROR;

	Py_DECREF(args);
	Py_DECREF(kwargs);

	context->app_data_free = (void *) wrap_appdata_free;
}
%}







/* * * * libotr/version.h * * * * * */

%immutable;
char * OTRL_VERSION;
unsigned short OTRL_VERSION_MAJOR;
unsigned short OTRL_VERSION_MINOR;
unsigned short OTRL_VERSION_SUB;
%mutable;

/* * * * libotr/context.h * * * * * */
typedef enum {
	OTRL_MSGSTATE_PLAINTEXT,
	OTRL_MSGSTATE_ENCRYPTED,
	OTRL_MSGSTATE_FINISHED
} OtrlMessageState;

typedef struct s_fingerprint {
	struct s_fingerprint *next;
	char fingerprint[20];
	struct context *context;
	char *trust;
} Fingerprint;

%extend Fingerprint {
	~Fingerprint (){
		if (self) otrl_context_forget_fingerprint(self, 0);
	}
}

typedef struct context {
	struct context * next;

	char * username;
	char * accountname;
	char * protocol;

	char *fragment;
	size_t fragment_len;
	unsigned short fragment_n;
	unsigned short fragment_k;

	OtrlMessageState msgstate;

	Fingerprint fingerprint_root;
	Fingerprint *active_fingerprint;
	unsigned int their_keyid;
	unsigned int our_keyid;

	char sessionid[20];
	OtrlSessionIdHalf sessionid_half;

	unsigned int protocol_version;

	unsigned int generation;

	long lastsent;
	char *lastmessage;
	int may_retransmit;

	enum {
	OFFER_NOT,
	OFFER_SENT,
	OFFER_REJECTED,
	OFFER_ACCEPTED
	} otr_offer;

	PyObject *app_data;

	OtrlSMState *smstate;
} ConnContext;

%extend ConnContext {
	~ConnContext (){
		if (self) otrl_context_forget(self);
	}
}

ConnContext * otrl_context_find(OtrlUserState us, const char *user,
	const char *accountname, const char *protocol, int add_if_missing,
	int *addedp,
	void (*add_app_data)(void *data, ConnContext *context), void *data);


Fingerprint *otrl_context_find_fingerprint(ConnContext *context,
	char fingerprint[20], int add_if_missing, int *addedp);

void otrl_context_set_trust(Fingerprint *fprint, const char *trust);

void otrl_context_force_finished(ConnContext *context);

void otrl_context_force_plaintext(ConnContext *context);

/* * * * libotr/privkey-t.h * * * * * */
typedef struct s_OtrlPrivKey {
	struct s_OtrlPrivKey *next;
	struct s_OtrlPrivKey **tous;

	char *accountname;
	char *protocol;
	unsigned short pubkey_type;
	gcry_sexp_t privkey;
	char *pubkey_data;
	size_t pubkey_datalen;
} OtrlPrivKey;

/* * * * libotr/privkey.h * * * * * */
void otrl_privkey_hash_to_human(char MYOUT[45], const char hash[20]);

void * otrl_privkey_fingerprint(OtrlUserState us, char MYOUT[45],
	const char *accountname, const char *protocol);

void * otrl_privkey_fingerprint_raw(OtrlUserState us, char NOTNULLTERMINATED[20], const char *accountname, const char *protocol);

gcry_error_t otrl_privkey_read(OtrlUserState us, const char *filename);

gcry_error_t otrl_privkey_generate(OtrlUserState us, const char *filename,
	const char *accountname, const char *protocol);

gcry_error_t otrl_privkey_read_fingerprints(OtrlUserState us,
	const char *filename,
	void (*add_app_data)(void *data, ConnContext *context),
	void  *data);

gcry_error_t otrl_privkey_write_fingerprints(OtrlUserState us,
	const char *filename);


/* * * * libotr/userstate.h * * * * * */
typedef struct s_OtrlUserState {
	ConnContext *context_root;
	OtrlPrivKey *privkey_root;
}* OtrlUserState;

%newobject otrl_userstate_create;
%extend s_OtrlUserState {
	~s_OtrlUserState() {
		if (self) otrl_userstate_free(self);
	}
}

OtrlUserState otrl_userstate_create(void);


/* * * * libotr/tlv.h * * * * * */
%immutable;
unsigned short OTRL_TLV_PADDING;
unsigned short OTRL_TLV_DISCONNECTED;
unsigned short OTRL_TLV_SMP1;
unsigned short OTRL_TLV_SMP1Q;
unsigned short OTRL_TLV_SMP2;
unsigned short OTRL_TLV_SMP3;
unsigned short OTRL_TLV_SMP4;
unsigned short OTRL_TLV_SMP_ABORT;
%mutable;

typedef struct s_OtrlTLV {
	unsigned short type;
	unsigned short len;
	char *data;
	struct s_OtrlTLV *next;
} OtrlTLV;

%extend OtrlTLV {
	~OtrlTLV() {
		if (self) otrl_tlv_free(self);
	}
}

OtrlTLV *otrl_tlv_find(OtrlTLV *tlvs, unsigned short type);

/* * * * libotr/dh.h * * * * * */
typedef enum {
	OTRL_SESSIONID_FIRST_HALF_BOLD,
	OTRL_SESSIONID_SECOND_HALF_BOLD
} OtrlSessionIdHalf;

/* * * * libotr/sm.h * * * * * */
typedef enum {
	OTRL_SMP_EXPECT1,
	OTRL_SMP_EXPECT2,
	OTRL_SMP_EXPECT3,
	OTRL_SMP_EXPECT4,
	OTRL_SMP_EXPECT5
} NextExpectedSMP;

typedef enum {
		OTRL_SMP_PROG_OK = 0,
		OTRL_SMP_PROG_CHEATED = -2,
		OTRL_SMP_PROG_FAILED = -1,
		OTRL_SMP_PROG_SUCCEEDED = 1
} OtrlSMProgState;

typedef struct {
	NextExpectedSMP nextExpected;
	int received_question;
	OtrlSMProgState sm_prog_state;
} OtrlSMState;

/* * * * libotr/proto.h * * * * * */
typedef unsigned int OtrlPolicy;

%immutable;
char * OTRL_MESSAGE_TAG_BASE;
char * OTRL_MESSAGE_TAG_V1;
char * OTRL_MESSAGE_TAG_V2;

unsigned int OTRL_MSGFLAGS_IGNORE_UNREADABLE;

OtrlPolicy OTRL_POLICY_ALLOW_V1;
OtrlPolicy OTRL_POLICY_ALLOW_V2;
OtrlPolicy OTRL_POLICY_REQUIRE_ENCRYPTION;
OtrlPolicy OTRL_POLICY_SEND_WHITESPACE_TAG;
OtrlPolicy OTRL_POLICY_WHITESPACE_START_AKE;
OtrlPolicy OTRL_POLICY_ERROR_START_AKE;

OtrlPolicy OTRL_POLICY_VERSION_MASK;

OtrlPolicy OTRL_POLICY_NEVER;
OtrlPolicy OTRL_POLICY_OPPORTUNISTIC;
OtrlPolicy OTRL_POLICY_MANUAL;
OtrlPolicy OTRL_POLICY_ALWAYS;
OtrlPolicy OTRL_POLICY_DEFAULT;
%mutable;

typedef enum {
	OTRL_MSGTYPE_NOTOTR,
	OTRL_MSGTYPE_TAGGEDPLAINTEXT,
	OTRL_MSGTYPE_QUERY,
	OTRL_MSGTYPE_DH_COMMIT,
	OTRL_MSGTYPE_DH_KEY,
	OTRL_MSGTYPE_REVEALSIG,
	OTRL_MSGTYPE_SIGNATURE,
	OTRL_MSGTYPE_V1_KEYEXCH,
	OTRL_MSGTYPE_DATA,
	OTRL_MSGTYPE_ERROR,
	OTRL_MSGTYPE_UNKNOWN
} OtrlMessageType;

typedef enum {
	OTRL_FRAGMENT_SEND_ALL,
	OTRL_FRAGMENT_SEND_ALL_BUT_FIRST,
	OTRL_FRAGMENT_SEND_ALL_BUT_LAST
} OtrlFragmentPolicy;

OtrlMessageType otrl_proto_message_type(const char *message);

/* * * * libotr/message.h * * * * * */
typedef enum {
	OTRL_NOTIFY_ERROR,
	OTRL_NOTIFY_WARNING,
	OTRL_NOTIFY_INFO
} OtrlNotifyLevel;

gcry_error_t otrl_message_sending(OtrlUserState us,
	const OtrlMessageAppOps *ops,
	void *opdata, const char *accountname, const char *protocol,
	const char *recipient, const char *message, OtrlTLV *tlvs,
	char **messagep,
	void (*add_appdata)(void *data, ConnContext *context),
	void *data);

%rename(otrl_message_receiving) wrap_otrl_message_receiving;
PyObject* wrap_otrl_message_receiving(OtrlUserState us,
  const OtrlMessageAppOps *ops, void *opdata, const char *accountname,
  const char *protocol, const char *sender, const char *message,
  void (*add_appdata)(void *data, ConnContext *context), void* data);
%{
PyObject* wrap_otrl_message_receiving(OtrlUserState us,
  const OtrlMessageAppOps *ops, void *opdata, const char *accountname,
  const char *protocol, const char *sender, const char *message,
  void (*add_appdata)(void *data, ConnContext *context), void* data) {

	char * newMessage = NULL;
	OtrlTLV *tlvsp = NULL;

	int is_internal = otrl_message_receiving(us, ops, opdata, accountname,
		protocol, sender, message, &newMessage, &tlvsp, add_appdata, data);

	PyObject * return_tuple = PyTuple_New(3);
	PyTuple_SetItem(return_tuple, 0, PyInt_FromLong(is_internal));
	if (newMessage != NULL) {
		PyTuple_SetItem(return_tuple, 1, PyString_FromString(newMessage));
		otrl_message_free(newMessage);
	}
	else if (is_internal == 0)
		PyTuple_SetItem(return_tuple, 1, PyString_FromString(message));
	else
		PyTuple_SetItem(return_tuple, 1, PyString_FromString(""));
	PyTuple_SetItem(return_tuple, 2, SWIG_NewPointerObj(SWIG_as_voidptr(tlvsp), SWIGTYPE_p_s_OtrlTLV, SWIG_POINTER_OWN));
	return return_tuple;
}
%}

gcry_error_t otrl_message_fragment_and_send(const OtrlMessageAppOps *ops,
	void *opdata, ConnContext *context, const char *message,
	OtrlFragmentPolicy fragPolicy, char **returnFragment);

void otrl_message_disconnect(OtrlUserState us, const OtrlMessageAppOps *ops,
	void *opdata, const char *accountname, const char *protocol,
	const char *username);

void otrl_message_initiate_smp(OtrlUserState us, const OtrlMessageAppOps *ops,
	void *opdata, ConnContext *context, const unsigned char *secret,
	size_t secretlen);

void otrl_message_initiate_smp_q(OtrlUserState us, const OtrlMessageAppOps *ops,
	void *opdata, ConnContext *context, const char * question,
	const unsigned char *secret, size_t secretlen);

void otrl_message_respond_smp(OtrlUserState us, const OtrlMessageAppOps *ops,
	void *opdata, ConnContext *context, const unsigned char *secret,
	size_t secretlen); 

void otrl_message_abort_smp(OtrlUserState us, const OtrlMessageAppOps *ops,
	void *opdata, ConnContext *context);

// vim: ts=4 noexpandtab filetype=c
