// This file is part of libptytty. Do not make local modifications.
// http://software.schmorp.de/pkg/libptytty

/*----------------------------------------------------------------------*
 * File:	logging.C
 *----------------------------------------------------------------------*
 *
 * All portions of code are copyright by their respective author/s.
 * Copyright (c) 1992      John Bovey <jdb@ukc.ac.uk>
 *				- original version
 * Copyright (c) 1993      lipka
 * Copyright (c) 1993      Brian Stempien <stempien@cs.wmich.edu>
 * Copyright (c) 1995      Raul Garcia Garcia <rgg@tid.es>
 * Copyright (c) 1995      Piet W. Plomp <piet@idefix.icce.rug.nl>
 * Copyright (c) 1997      Raul Garcia Garcia <rgg@tid.es>
 * Copyright (c) 1998-2001 Geoff Wing <gcw@pobox.com>
 * 				- extensive modifications
 * Copyright (c) 1999      D J Hawkey Jr <hawkeyd@visi.com>
 *				- lastlog support
 * Copyright (c) 2004-2006 Marc Lehmann <schmorp@schmorp.de>
 * Copyright (c) 2006      Emanuele Giaquinta <e.giaquinta@glauco.it>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *----------------------------------------------------------------------*/

#include "config.h"

#include "ptytty.h"

#if UTMP_SUPPORT

#include <cstdio>
#include <cstring>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>

/*
 * BSD style utmp entry
 *      ut_line, ut_name, ut_host, ut_time
 * SYSV style utmp (and utmpx) entry
 *      ut_user, ut_id, ut_line, ut_pid, ut_type, ut_exit, ut_time
 */

/* ------------------------------------------------------------------------- */
/*
 * Write a BSD style utmp entry
 */
#if defined(HAVE_STRUCT_UTMP) && !defined(HAVE_UTMP_PID)
static int
write_bsd_utmp (int utmp_pos, struct utmp *ut)
{
  int             fd;

  if (utmp_pos <= 0 || (fd = open (UTMP_FILE, O_WRONLY)) == -1)
    return 0;

  if (lseek (fd, (off_t) (utmp_pos * sizeof (struct utmp)), SEEK_SET) != -1)
    write (fd, ut, sizeof (struct utmp));
  close (fd);
  return 1;
}
#endif

/* ------------------------------------------------------------------------- */
/*
 * Update a BSD style wtmp entry
 */
#if defined(WTMP_SUPPORT) && !defined(HAVE_UPDWTMP) && defined(HAVE_STRUCT_UTMP)
static void
update_wtmp (const char *fname, const struct utmp *ut)
{
  int             fd, gotlock, retry;
  struct flock    lck;	/* fcntl locking scheme */
  struct stat     sbuf;

  if ((fd = open (fname, O_WRONLY | O_APPEND, 0)) < 0)
    return;

  lck.l_whence = SEEK_END;	/* start lock at current eof */
  lck.l_len = 0;		/* end at ``largest possible eof'' */
  lck.l_start = 0;
  lck.l_type = F_WRLCK;	/* we want a write lock */

  /* attempt lock with F_SETLK; F_SETLKW would cause a deadlock! */
  for (retry = 10, gotlock = 0; retry--;)
    if (fcntl (fd, F_SETLK, &lck) != -1)
      {
        gotlock = 1;
        break;
      }
    else if (errno != EAGAIN && errno != EACCES)
      break;
  if (!gotlock)
    {
      /* give it up */
      close (fd);
      return;
    }
  if (fstat (fd, &sbuf) == 0)
    if (write (fd, ut, sizeof (struct utmp)) != sizeof (struct utmp))
      ftruncate (fd, sbuf.st_size);	/* remove bad writes */

  lck.l_type = F_UNLCK;	/* unlocking the file */
  fcntl (fd, F_SETLK, &lck);
  close (fd);
}
#endif

/* ------------------------------------------------------------------------- */
#ifdef LASTLOG_SUPPORT
static void
update_lastlog (const char *pty, const char *host)
{
# if defined(HAVE_STRUCT_LASTLOGX) && defined(HAVE_UPDLASTLOGX)
  struct lastlogx llx;
# endif
# ifdef HAVE_STRUCT_LASTLOG
  int             fd;
  struct lastlog  ll;
  char            lastlogfile[256];
  struct passwd  *pwent;
  struct stat     st;
# endif

# if defined(HAVE_STRUCT_LASTLOGX) && defined(HAVE_UPDLASTLOGX)
  memset (&llx, 0, sizeof (llx));
  llx.ll_tv.tv_sec = time (NULL);
  llx.ll_tv.tv_usec = 0;
  strncpy (llx.ll_line, pty, sizeof (llx.ll_line));
  strncpy (llx.ll_host, host, sizeof (llx.ll_host));
  updlastlogx (LASTLOGX_FILE, getuid (), &llx);
# endif

# ifdef HAVE_STRUCT_LASTLOG
  pwent = getpwuid (getuid ());
  if (!pwent)
    {
      PTYTTY_WARN ("no entry in password file, not updating lastlog.\n", 0);
      return;
    }

  memset (&ll, 0, sizeof (ll));
  ll.ll_time = time (NULL);
  strncpy (ll.ll_line, pty, sizeof (ll.ll_line));
  strncpy (ll.ll_host, host, sizeof (ll.ll_host));
  if (stat (LASTLOG_FILE, &st) != 0)
    return;
  if (S_ISDIR (st.st_mode))
    {
      snprintf (lastlogfile, sizeof (lastlogfile), "%s/%s", LASTLOG_FILE,
               (!pwent->pw_name || pwent->pw_name[0] == '\0') ? "unknown"
               : pwent->pw_name);
      if ((fd = open (lastlogfile, O_WRONLY | O_CREAT, 0644)) >= 0)
        {
          write (fd, &ll, sizeof (ll));
          close (fd);
        }
    }
  else if (S_ISREG (st.st_mode))
    if ((fd = open (LASTLOG_FILE, O_RDWR)) != -1)
      {
        if (lseek (fd, (off_t) ((long)pwent->pw_uid * sizeof (ll)),
                   SEEK_SET) != -1)
          write (fd, &ll, sizeof (ll));
        close (fd);
      }
# endif /* HAVE_STRUCT_LASTLOG */
}
#endif /* LASTLOG_SUPPORT */

#ifdef HAVE_STRUCT_UTMP
static void
fill_utmp (struct utmp *ut, bool login, int pid, const char *id, const char *line, const char *user, const char *host)
{
  memset (ut, 0, sizeof (struct utmp));

  strncpy (ut->ut_line, line, sizeof (ut->ut_line));
# ifdef HAVE_UTMP_PID
  strncpy (ut->ut_id, id, sizeof (ut->ut_id));
  ut->ut_pid = pid;
  ut->ut_type = login ? USER_PROCESS : DEAD_PROCESS;
# endif
  ut->ut_time = time (NULL);

  if (login)
    {
# ifdef HAVE_UTMP_PID
      strncpy (ut->ut_user, user, sizeof (ut->ut_user));
# else
      strncpy (ut->ut_name, user, sizeof (ut->ut_name));
# endif
# ifdef HAVE_UTMP_HOST
      strncpy (ut->ut_host, host, sizeof (ut->ut_host));
# endif
    }
}
#endif

#ifdef HAVE_STRUCT_UTMPX
static void
fill_utmpx (struct utmpx *utx, bool login, int pid, const char *id, const char *line, const char *user, const char *host)
{
  memset (utx, 0, sizeof (struct utmpx));

  // posix says that ut_line is not meaningful for DEAD_PROCESS
  // records, but most implementations of last use ut_line to
  // associate records in wtmp file
  strncpy (utx->ut_line, line, sizeof (utx->ut_line));
  strncpy (utx->ut_id, id, sizeof (utx->ut_id));
  utx->ut_pid = pid;
  utx->ut_type = login ? USER_PROCESS : DEAD_PROCESS;
  utx->ut_tv.tv_sec = time (NULL);
  utx->ut_tv.tv_usec = 0;
# if HAVE_UTMPX_SESSION
  utx->ut_session = getsid (0);
# endif

  if (login)
    {
      strncpy (utx->ut_user, user, sizeof (utx->ut_user));
# ifdef HAVE_UTMPX_HOST
      strncpy (utx->ut_host, host, sizeof (utx->ut_host));
# endif
    }
}
#endif

/* ------------------------------------------------------------------------- */

/*
 * make and write utmp and wtmp entries
 */
void
ptytty_unix::login (int cmd_pid, bool login_shell, const char *hostname)
{
  const char *pty = name;

  if (!pty || !*pty)
    return;

  this->cmd_pid     = cmd_pid;
  this->login_shell = login_shell;

#ifdef HAVE_STRUCT_UTMP
  struct utmp *ut = &this->ut;
#endif
#ifdef HAVE_STRUCT_UTMPX
  struct utmpx *utx = &this->utx;
#endif
  int i;
  struct passwd *pwent = getpwuid (getuid ());
  const char *user = (pwent && pwent->pw_name) ? pwent->pw_name : "?";

  if (!strncmp (pty, "/dev/", 5))
    pty += 5;		/* skip /dev/ prefix */

#if defined(HAVE_UTMP_PID) || defined(HAVE_STRUCT_UTMPX)
  if (!strncmp (pty, "pty", 3) || !strncmp (pty, "tty", 3))
    strncpy (ut_id, pty + 3, sizeof (ut_id));
  else if (sscanf (pty, "pts/%d", &i) == 1)
    sprintf (ut_id, "vt%02x", (i & 0xff));	/* sysv naming */
  else
    {
      PTYTTY_WARN ("can't parse tty name \"%s\", not adding utmp entry.\n", pty);
      return;
    }
#endif

#ifdef HAVE_STRUCT_UTMP
  fill_utmp (ut, true, cmd_pid, ut_id, pty, user, hostname);
#endif

#ifdef HAVE_STRUCT_UTMPX
  fill_utmpx (utx, true, cmd_pid, ut_id, pty, user, hostname);
#endif

#ifdef HAVE_STRUCT_UTMP
# ifdef HAVE_UTMP_PID
  setutent ();
  pututline (ut);
  endutent ();
# else
  int fd_stdin = dup (STDIN_FILENO);
  dup2 (tty, STDIN_FILENO);

  utmp_pos = ttyslot ();
  if (!write_bsd_utmp (utmp_pos, ut))
    utmp_pos = 0;

  dup2 (fd_stdin, STDIN_FILENO);
  close (fd_stdin);
# endif
#endif

#ifdef HAVE_STRUCT_UTMPX
  setutxent ();
  pututxline (utx);
  endutxent ();
#endif

#ifdef WTMP_SUPPORT
#ifdef LOG_ONLY_ON_LOGIN
  if (login_shell)
#endif
    {
# ifdef HAVE_STRUCT_UTMP
#  ifdef HAVE_UPDWTMP
      updwtmp (WTMP_FILE, ut);
#  else
      update_wtmp (WTMP_FILE, ut);
#  endif
# endif
# if defined(HAVE_STRUCT_UTMPX) && defined(HAVE_UPDWTMPX)
      updwtmpx (WTMPX_FILE, utx);
# endif
    }
#endif
#ifdef LASTLOG_SUPPORT
#ifdef LOG_ONLY_ON_LOGIN
  if (login_shell)
#endif
    update_lastlog (pty, hostname);
#endif
}

/* ------------------------------------------------------------------------- */
/*
 * remove utmp and wtmp entries
 */
void
ptytty_unix::logout ()
{
  if (!cmd_pid)
    return;

  const char *pty = name;

  if (!strncmp (pty, "/dev/", 5))
    pty += 5;

#ifdef HAVE_STRUCT_UTMP
  struct utmp *tmput, *ut = &this->ut;
#endif
#ifdef HAVE_STRUCT_UTMPX
  struct utmpx *tmputx, *utx = &this->utx;
#endif

#ifdef HAVE_STRUCT_UTMP
  fill_utmp (ut, false, cmd_pid, ut_id, pty, 0, 0);
#endif

#ifdef HAVE_STRUCT_UTMPX
  fill_utmpx (utx, false, cmd_pid, ut_id, pty, 0, 0);
#endif

  /*
   * Write ending wtmp entry
   */
#ifdef WTMP_SUPPORT
#ifdef LOG_ONLY_ON_LOGIN
  if (login_shell)
#endif
    {
# ifdef HAVE_STRUCT_UTMP
#  ifdef HAVE_UPDWTMP
      updwtmp (WTMP_FILE, ut);
#  else
      update_wtmp (WTMP_FILE, ut);
#  endif
# endif
# if defined(HAVE_STRUCT_UTMPX) && defined(HAVE_UPDWTMPX)
      updwtmpx (WTMPX_FILE, utx);
# endif
    }
#endif

  /*
   * Write utmp entry
   */
#ifdef HAVE_STRUCT_UTMP
# ifdef HAVE_UTMP_PID
  setutent ();
  tmput = getutid (ut);
  if (tmput && tmput->ut_pid == cmd_pid)
    pututline (ut);
  endutent ();
# else
  write_bsd_utmp (utmp_pos, ut);
# endif
#endif
#ifdef HAVE_STRUCT_UTMPX
  setutxent ();
  tmputx = getutxid (utx);
  if (tmputx && tmputx->ut_pid == cmd_pid)
    pututxline (utx);
  endutxent ();
#endif

  cmd_pid = 0;
}

#else
void
ptytty_unix::login (int cmd_pid, bool login_shell, const char *hostname)
{
}
#endif

